package csbase.client.applications.serverdiagnostic.monitor;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.text.SimpleDateFormat;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.ToolTipManager;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;

import csbase.client.applications.serverdiagnostic.ServerDiagnostic;
import csbase.logic.diagnosticservice.Status;

/**
 * Painel que exibe as informaes de monitorao dos recursos.
 * 
 * @author Tecgraf PUC-Rio
 */
public class StatusListPanel extends JPanel {
  /**
   * Formato de data.
   */
  private SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy "
    + "HH:mm:ss");
  /**
   * Referncia para a aplicao
   */
  private ServerDiagnostic application;
  /**
   * Rendered padro da lista.
   */
  private SwingRenderer defaultRenderer;
  /**
   * SplitPane para separar o painel com a lista de resursos e o painel de
   * detalhes do estado do recurso.
   */
  private JSplitPane splitPane;
  /**
   * O painel com as informaes bsicas do estado.
   */
  private JPanel headlinePanel;
  /**
   * O painel com os detahes do estado.
   */
  private JPanel detailPanel;

  /**
   * O n raiz da rvore de status
   */
  private StatusGroupTreeNode node;

  /**
   * Contrutor para usar o locale padro.
   * 
   * @param application a aplicao
   * @param node n raiz da rvore que  exibida a esquerda com os estados dos
   *        recursos agrupados.
   */
  public StatusListPanel(ServerDiagnostic application, StatusGroupTreeNode node) {
    this.node = node;
    this.application = application;
    this.defaultRenderer = new DefaultSwingRenderer();
    buildPanel();
  }

  /**
   * Constri o painel.
   */
  private void buildPanel() {
    headlinePanel = new JPanel(new BorderLayout());
    detailPanel = new JPanel(new BorderLayout());

    splitPane =
      new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, headlinePanel, detailPanel);
    splitPane.setBorder(BorderFactory.createEmptyBorder());
    splitPane.setOneTouchExpandable(true);
    splitPane.setDividerLocation(200);

    Dimension minimumSize = new Dimension(200, 50);
    headlinePanel.setMinimumSize(minimumSize);
    detailPanel.setMinimumSize(minimumSize);
    detailPanel.add(Box.createRigidArea(new Dimension(450, 450)));

    this.setLayout(new BorderLayout());
    this.add(splitPane);
  }

  /**
   * Atualiza o painel de detalhes.
   * 
   * @param panel o novo contedo do painel
   */
  private void updateDetailPanel(JPanel panel) {
    detailPanel.removeAll();
    detailPanel.setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.anchor = GridBagConstraints.NORTH;
    constraints.insets = new Insets(0, 5, 0, 0);
    constraints.weightx = 1.0;
    constraints.weighty = 1.0;
    detailPanel.add(panel, constraints);
    this.revalidate();
    this.repaint();
  }

  /**
   * Define o renderer padro de {@link csbase.logic.diagnosticservice.Status}
   * do painel.
   * 
   * @param renderer o renderer padro
   */
  public void setDefaultRenderer(SwingRenderer renderer) {
    this.defaultRenderer = renderer;
  }

  /**
   * Refaz o painel. Deve ser chamado sempre que um novo estado  adicionado.
   */
  public void refreshPanel() {
    headlinePanel.removeAll();
    headlinePanel.setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();

    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.anchor = GridBagConstraints.NORTH;
    constraints.insets = new Insets(0, 0, 0, 5);
    constraints.weightx = 1.0;
    constraints.weighty = 1.0;
    headlinePanel.add(new JScrollPane(createStatusTree()), constraints);

    this.revalidate();
    this.repaint();
  }

  /**
   * Cria a rvore de status que  exibida no painel a esquerda. Quando um n de
   * status  selecionado, o painel a direita exibe as informaes detalhadas do
   * estado.
   * 
   * @return a rvore de status
   */
  private JTree createStatusTree() {
    StatusTree statusTree = new StatusTree(application.getLocale(), node);
    statusTree.addTreeSelectionListener(new TreeSelectionListener() {
      @Override
      public void valueChanged(TreeSelectionEvent treeSelectionEvent) {
        JTree tree = (JTree) treeSelectionEvent.getSource();
        DefaultMutableTreeNode node =
          (DefaultMutableTreeNode) tree.getLastSelectedPathComponent();
        if (node != null) {
          JPanel panel;
          if (node.isLeaf()) {
            StatusNode selectedNodeData = (StatusNode) node.getUserObject();
            if (selectedNodeData.name == null) {
              Status selectStatus = selectedNodeData.status;
              SwingRenderer renderer = selectedNodeData.renderer;
              if (renderer == null) {
                renderer = defaultRenderer;
              }
              panel = (JPanel) renderer.getDetails(selectStatus);
            }
            else {
              panel = new JPanel();
              panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
              Status status = selectedNodeData.status;
              SwingRenderer renderer = selectedNodeData.renderer;
              if (renderer == null) {
                renderer = defaultRenderer;
              }
              panel.add(renderer.getDetails(status));
            }
            updateDetailPanel(panel);
          }
        }
      }
    });

    ToolTipManager.sharedInstance().registerComponent(statusTree);

    return statusTree;
  }
}
