package csbase.client.util.event;

import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

/**
 * @author Tecgraf / PUC-Rio
 * 
 * Classe utilizada para gerenciar o envio de eventos assim como seus
 * interessados.
 */
public class EventManager {

  /**
   * Lista de todos os observadores.<br>
   * Utilizada pelo mtodo {@link EventManager#removeEventListeners(Class)} para
   * descobrir quais so os observadores de um determinado tipo de evento, para
   * que estes possam ser removidos da
   * {@link EventManager#eventNotifier estrutura}.
   */
  private List<EventObserver<?>> observers;
  /**
   * Estrutura que armazena os {@link EventObserver observadores} que encapsulam
   * os {@link EventListener interessados} nos eventos. Esta estrutura tambm 
   * utilizada para disparar os eventos.
   */
  private Observable eventNotifier;

  /**
   * Construtor.
   */
  public EventManager() {
    observers = new ArrayList<EventObserver<?>>();
    eventNotifier = new Observable() {
      /**
       * Chama o mtodo {@link Observable#setChanged()} fazendo com que os
       * eventos sejam disparados automaticamente a cada nova notificao.
       */
      @Override
      public void notifyObservers(Object arg) {
        setChanged();
        super.notifyObservers(arg);
      }
    };
  }

  /**
   * Adiciona um ouvinte interessado em receber eventos de um tipo, disparados
   * atravs desta instncia.
   * 
   * @param <E> tipo do evento ao qual o ouvinte tem interesse.
   * @param <L> tipo do ouvinte.
   * @param listener o ouvinte do evento.
   * @param clazz classe de evento ao qual o ouvinte tem interesse.
   */
  public synchronized <E extends IEvent, L extends EventListener<E>> void addEventListener(
    L listener, Class<E> clazz) {
    /*
     * Encapsula o ouvinte em um observador que ir receber todos os eventos
     * disparados atravs desta classe, filtrar os de interesse do ouvinte e
     * repass-los para ele.
     */
    EventObserver observer = new EventObserver(listener, clazz);
    observers.add(observer);
    eventNotifier.addObserver(observer);
  }

  /**
   * Remove todos os ouvintes interessados em um tipo especfico de evento.<br>
   * Uma vez removido, um ouvinte no ir mais receber eventos disparados
   * atraves desta instncia a menos que seja novamente inserido em sua
   * estrutura atravs do mtodo
   * {@link EventManager#addEventListener(EventListener, Class)}.
   * 
   * @param <E> tipo do evento.
   * @param clazz classe do evento a ter seus ouvintes removidos.
   */
  public synchronized <E extends IEvent> void removeEventListeners(
    Class<E> clazz) {
    for (EventObserver<?> observer : observers) {
      if (clazz.isAssignableFrom(observer.clazz)) {
        observers.remove(observer);
        eventNotifier.deleteObserver(observer);
      }
    }
  }

  /**
   * Remove um ouvinte.
   * 
   * @param <E> tipo do evento a que este ouvinte tem interesse.
   * @param <L> tipo do ouvinte.
   * @param listener ouvinte a ser removido da estrutura. Uma vez removido ele
   *        no ir mais receber eventos disparados atraves desta instncia a
   *        menos que seja novamente inserido em sua estrutura atravs do mtodo
   *        {@link EventManager#addEventListener(EventListener, Class)}.
   */
  public synchronized <E extends IEvent, L extends EventListener<E>> void removeEventListener(
    L listener) {
    for (EventObserver<?> observer : observers) {
      if (observer.listener.equals(listener)) {
        observers.remove(observer);
        eventNotifier.deleteObserver(observer);
        return;
      }
    }
  }

  /**
   * Dispara um evento.<br>
   * O evento disparado ser repassado a todos os ouvintes interessados, na
   * ordem em que foram cadastrados, atravs do mtodo
   * {@link EventListener#eventFired(IEvent)}.<br>
   * <br>
   * Obs.: No ser criada uma {@link Thread} para cada entrega do evento a ser
   * feita. Todas as entregas sero feitas na {@link Thread} em que este mtodo
   * foi chamado, ento tome cuidado para um ouvinte no travar o processo ou
   * atrasar demais a entrega do evento para os outros.
   * 
   * @param event evento a ser disparado.
   */
  public void fireEvent(IEvent event) {
    eventNotifier.notifyObservers(event);
  }

  /**
   * Obtm o nmero de ouvintes de eventos cadastrados neste gerente.
   * 
   * @return o nmero de ouvintes de eventos cadastrados neste gerente.
   */
  public int countListeners() {
    return eventNotifier.countObservers();
  }
}

/**
 * Encapsula um {@link EventListener} funcionando como um filtro de eventos.
 * 
 * @param <T> tipo de evento observado
 */
class EventObserver<T extends IEvent> implements Observer {

  /*
   * Ouvinte de evento.
   */
  public EventListener listener;
  /*
   * Classe do tipo de evento que se tem interesse em tratar.
   */
  public Class<T> clazz;

  /**
   * Construtor.
   * 
   * @param listener ouvinte de evento que ser encapsulado.
   * @param clazz classe do tipo de evento que interessa ao listener.
   */
  protected EventObserver(EventListener listener, Class<T> clazz) {
    super();
    this.listener = listener;
    this.clazz = clazz;
  }

  /**
   * Recebe eventos do {@link EventManager#eventNotifier} e repassa somente os
   * eventos a que o listener estiver interessado.
   */
  public void update(Observable o, Object arg) {
    IEvent event = (IEvent) arg;
    if (this.clazz.isAssignableFrom(event.getClass())) {
      this.listener.eventFired(event);
    }
  }
}
