/*
 * $Id: UserInfo.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Date;
import java.util.Hashtable;

/**
 * A classe <code>UserInfo</code> representa os dados de um usurio.
 * 
 * @author $Author: fpina $
 * @version $Revision: 176168 $
 */
public class UserInfo implements Serializable, Cloneable {
  private Hashtable<String, Object> attributes;

  /**
   * Cria uma UserInfo.
   * 
   * @param login o login do usurio
   * @param name o nome do usurio
   * @param emails os emails do usurio
   * @param roleIds o array de ids de perfis do usurio
   * @param permissionIds o array de ids de permisses do usurio
   */
  public UserInfo(String login, String name, String[] emails, Object[] roleIds,
    Object[] permissionIds) {
    this(login, name, emails, roleIds, permissionIds, 0);
  }
  
  /**
   * Cria uma UserInfo.
   * 
   * @param login o login do usurio
   * @param name o nome do usurio
   * @param emails os emails do usurio
   * @param roleIds o array de ids de perfis do usurio
   * @param permissionIds o array de ids de permisses do usurio
   * @param creationDate a data de criao do usurio
   */
  public UserInfo(String login, String name, String[] emails, Object[] roleIds,
    Object[] permissionIds, Date creationDate) {
    this(login, name, emails, roleIds, permissionIds, 0, creationDate);
  }

  /**
   * Cria uma UserInfo.
   * 
   * @param login o login do usurio
   * @param name o nome do usurio
   * @param emails os emails do usurio
   * @param roleIds o array de ids de perfis do usurio
   * @param permissionIds o array de ids de permisses do usurio
   * @param lastUpdate a data de ltima atualizao do usurio
   */
  public UserInfo(String login, String name, String[] emails, Object[] roleIds,
    Object[] permissionIds, long lastUpdate) {
    this(login, name, emails, roleIds, permissionIds, lastUpdate, null);
  }
  
  /**
   * Cria uma UserInfo.
   * 
   * @param login o login do usurio
   * @param name o nome do usurio
   * @param emails os emails do usurio
   * @param roleIds o array de ids de perfis do usurio
   * @param permissionIds o array de ids de permisses do usurio
   * @param lastUpdate a data de ltima atualizao do usurio
   * @param creationDate a data de criao do usurio
   */
  public UserInfo(String login, String name, String[] emails, Object[] roleIds,
    Object[] permissionIds, long lastUpdate, Date creationDate) {
    this.attributes = new Hashtable<String, Object>();
    attributes.put(User.LOGIN, (login == null) ? "" : login);
    attributes.put(User.NAME, (name == null) ? "" : name);
    attributes.put(User.EMAILS, (emails == null) ? new String[0] : emails);
    attributes
      .put(User.ROLE_IDS, (roleIds == null) ? (new Object[0]) : roleIds);
    attributes.put(User.PERMISSION_IDS,
      (permissionIds == null) ? (new Object[0]) : permissionIds);
    attributes.put(User.LAST_UPDATE, new Long(lastUpdate));
    if (creationDate != null) {
      attributes.put(User.CREATION_DATE, creationDate);
    }
  }

  /**
   * Construtor default.
   */
  public UserInfo() {
    this(null, null, null, null, null, 0);
  }

  /**
   * Cria um UserInfo.
   * 
   * @param attributes Hash com os atributos do usurio.
   */
  public UserInfo(Hashtable<String, Object> attributes) {
    this.attributes = attributes;
  }

  /**
   * Obtm a Hashtable com os atributos do usurio.
   * 
   * @return Os atributos do usurio.
   */
  public Hashtable<String, Object> getAttributes() {
    return attributes;
  }

  /**
   * Obtm um atributo do usurio.
   * 
   * @param key Chave do atributo na Hashtable.
   * 
   * @return Valor do atributo.
   */
  public Object getAttribute(String key) {
    return attributes.get(key);
  }

  /**
   * Coloca um atributo do usurio na Hashtable.
   * 
   * @param key Chave do atributo na Hashtable.
   * @param value Valor do atributo.
   */
  public void setAttribute(String key, Object value) {
    if (key == null || value == null) {
      return;
    }
    attributes.put(key, value);
  }

  /**
   * Remove um atributo do usurio
   * 
   * @param key chave do atributo.
   * 
   * @return O objeto removido.
   */
  public Object removeAttribute(String key) {
    return attributes.remove(key);
  }

  /**
   * Obtm login do usurio.
   * 
   * @return Login do usurio.
   */
  public String getLogin() {
    return (String) attributes.get(User.LOGIN);
  }

  @Override
  public Object clone() {
    try {
      return super.clone();
    }
    catch (CloneNotSupportedException e) {
      return null;
    }
  }
}
