/*
 * $Author: lmoreira $ $Date: 2005/07/14 14:28:19 $ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.validators.IntegerParameterValidator;

/**
 * Parmetro do Tipo Inteiro.
 *
 * @author lmoreira
 */
public final class IntegerParameter extends SimpleParameter<Integer> {

  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE = "INTEGER";

  /**
   * Valor indicativo de erro.
   */
  public static final int ERROR_VALUE = Integer.MIN_VALUE;

  /**
   * Os observadores.
   */
  private transient List<IntegerParameterListener> listeners;

  /**
   * O valor mximo considerado vlido.
   */
  private Integer maximum;
  /**
   * O valor mnimo considerado vlido.
   */
  private Integer minimum;

  /**
   * Cria um parmetro.
   *
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se  o valor  opcional/obrigatrio.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   */
  public IntegerParameter(String name, String label, String description,
    Integer defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maximum, Integer minimum) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    listeners = new LinkedList<IntegerParameterListener>();
    setMaximum(maximum);
    setMinimum(minimum);
  }

  /**
   * Adiciona um observador a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addIntegerParameterListener(IntegerParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "listener"));
    }
    listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    Integer value = getValue();
    if (value == null) {
      return null;
    }
    return new Double(value);
  }

  /**
   * Obtm o valor mximo.
   *
   * @return O valor mximo ou {@code null} se ele no existir.
   */
  public Integer getMaximum() {
    return maximum;
  }

  /**
   * Obtm o valor mnimo.
   *
   * @return O valor mnimo ou {@code null} se ele no existir.
   */
  public Integer getMinimum() {
    return minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue != null) {
      Integer value;
      try {
        int intValue = Integer.parseInt(parameterValue);
        value = new Integer(intValue);
      }
      catch (NumberFormatException e) {
        throw new ParseException(e,LNG.get(
    		"csbase.logic.algorithms.parameters.InvalidToInteger"),
    		parameterValue, this);
      }
      setValue(value);
    }
    else {
      setValue(null);
    }
  }

  /**
   * Atribui o mximo a este parmetro.
   *
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o valor mximo
   *         atual for igual ao valor mximo fornecido.
   */
  public boolean setMaximum(Integer maximum) {
    if (minimum != null) {
      if ((maximum != null) && (maximum.compareTo(minimum) < 0)) {
        throw new IllegalArgumentException(LNG.get(
        		"csbase.logic.algorithms.parameters.MaxLessThanMin"));
      }
    }
    if ((this.maximum == null) && (maximum == null)) {
      return false;
    }
    if ((this.maximum != null) && (this.maximum.equals(maximum))) {
      return false;
    }
    this.maximum = maximum;
    updateValidator();
    for (IntegerParameterListener listener : listeners) {
      listener.maximumWasChanged(this);
    }
    return true;
  }

  /**
   * Atribui o mnimo a este parmetro.
   *
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o valor mnimo
   *         atual for igual ao valor mnimo fornecido.
   */
  public boolean setMinimum(Integer minimum) {
    if (maximum != null) {
      if ((minimum != null) && (minimum.compareTo(maximum) > 0)) {
        throw new IllegalArgumentException(LNG.get(
    		"csbase.logic.algorithms.parameters.MinGreaterThenMax"));
      }
    }
    if ((this.minimum == null) && (minimum == null)) {
      return false;
    }
    if ((this.minimum != null) && (this.minimum.equals(minimum))) {
      return false;
    }
    this.minimum = minimum;
    updateValidator();
    for (IntegerParameterListener listener : listeners) {
      listener.minimumWasChanged(this);
    }
    return true;
  }

  /**
   * Atualiza os valores utilizados para validar o parmetro.
   */
  private void updateValidator() {
    IntegerParameterValidator validator =
      (IntegerParameterValidator) getParameterValidator();
    validator.setMaximum(this.maximum);
    validator.setMinimum(this.minimum);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IntegerParameterValidator createParameterValidator() {
    return new IntegerParameterValidator(isOptional(), maximum, minimum);
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
  ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<IntegerParameterListener>();
  }
}
