package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import csbase.logic.AccessSGAPathPermission;
import csbase.logic.ClientSGAFile;
import csbase.logic.SGASet;
import csbase.logic.User;
import csbase.logic.algorithms.parameters.ClientSGAFileParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationSuccess;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Validador do {@link ClientSGAFileParameter Parmetro do tipo path de um
 * arquivo que est no SGA}.
 * 
 * @author Tecgraf
 */
public class ClientSGAFileValidator extends
  SimpleParameterValidator<ClientSGAFile> {

  /**
   * Construtor padro.
   * 
   * @param isOptional - indica se o valor  opcional/obrigatrio.
   */
  public ClientSGAFileValidator(boolean isOptional) {
    super(isOptional);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validateValue(SimpleParameter<?> parameter,
    ClientSGAFile value, ValidationContext context) throws RemoteException {
    Validation result = super.validateValue(parameter, value, context);
    if (!result.isWellSucceeded()) {
      return result;
    }

    ClientSGAFileParameter param = (ClientSGAFileParameter) parameter;

    if (value != null) {
      result = validateSGA(param, value);
      if (!result.isWellSucceeded()) {
        return result;
      }

      if (!param.isMultipleSelection()) {
        result = validateFileOrDir(param, value);
        if (!result.isWellSucceeded()) {
          return result;
        }
      }

      result = validateMultipleSelection(param, value);
      if (!result.isWellSucceeded()) {
        return result;
      }

      result = validatePathsFormat(param, value);
      if (!result.isWellSucceeded()) {
        return result;
      }

      if (param.isInputMode()) {
        result = validatePermissions(param, value, context);
        if (!result.isWellSucceeded()) {
          return result;
        }
      }
    }

    return new ValidationSuccess();
  }

  /**
   * Verifica a disponibilidade do SGA.
   * 
   * @param parameter - objeto que encapsula o parmetro.
   * @param value - objeto que encapsula os paths de um SGA.
   * @return resultado da validao.
   * @throws RemoteException - em caso de erro na comunicao com servidor.
   */
  private Validation validateSGA(ClientSGAFileParameter parameter,
    ClientSGAFile value) throws RemoteException {
    SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    SGASet sgaSet = sgaService.getSGASet(value.getSGAName());
    if (sgaSet == null) {
      LocalizedMessage message =
        new LocalizedMessage(ClientSGAFileValidator.class, "sga.not.found",
          new Object[] { value.getSGAName() });
      return new ValidationError(message);
    }
    if (!sgaSet.getAlive()) {
      LocalizedMessage message =
        new LocalizedMessage(ClientSGAFileValidator.class, "sga.not.alive",
          new Object[] { value.getSGAName() });
      return new ValidationError(message);
    }
    return new ValidationSuccess();
  }

  /**
   * Verifica se os paths so diretrios ou arquivos de acordo com a
   * configurao do parmetro.
   * 
   * @param parameter - objeto que encapsula o parmetro.
   * @param value - objeto que encapsula os paths de um SGA.
   * @return resultado da validao.
   */
  private Validation validateFileOrDir(ClientSGAFileParameter parameter,
    ClientSGAFile value) {
    boolean onlyFiles = parameter.isShowFiles();

    List<ClientSGAFile> files = new ArrayList<ClientSGAFile>();
    files.add(value);
    files.addAll(value.getBrothers());

    for (ClientSGAFile file : files) {
      if (onlyFiles) {
        if (file.isDir()) {
          LocalizedMessage message =
            new LocalizedMessage(ClientSGAFileValidator.class,
              "should.be.file", new Object[] { file.getSGAName(),
                  file.getStringPath() });
          return new ValidationError(message);
        }
      }
      else {
        if (!file.isDir()) {
          LocalizedMessage message =
            new LocalizedMessage(ClientSGAFileValidator.class,
              "should.be.directory", new Object[] { file.getSGAName(),
                  file.getStringPath() });
          return new ValidationError(message);
        }
      }
    }

    return new ValidationSuccess();
  }

  /**
   * Verifica se a quantidade de paths escolhidos pelo usurio  aceitvel.
   * 
   * @param parameter - objeto que encapsula o parmetro.
   * @param value - objeto que encapsula os paths de um SGA.
   * @return resultado da validao.
   */
  private Validation validateMultipleSelection(
    ClientSGAFileParameter parameter, ClientSGAFile value) {
    if (!parameter.isMultipleSelection() && !value.getBrothers().isEmpty()) {
      LocalizedMessage message =
        new LocalizedMessage(ClientSGAFileValidator.class, "only.one",
          new Object[] { value.getSGAName() });
      return new ValidationError(message);
    }

    return new ValidationSuccess();
  }

  /**
   * Verifica o formato dos paths.
   * 
   * @param parameter - objeto que encapsula o parmetro.
   * @param value - objeto que encapsula os paths de um SGA.
   * @return resultado da validao.
   */
  private Validation validatePathsFormat(ClientSGAFileParameter parameter,
    ClientSGAFile value) {

    List<ClientSGAFile> files = new ArrayList<ClientSGAFile>();
    files.add(value);
    files.addAll(value.getBrothers());

    for (ClientSGAFile file : files) {
      Validation result =
        FileParameterValidator.validateFileName(file.getPath());
      if (!result.isWellSucceeded()) {
        return result;
      }

      if (!file.canRead()) {
        LocalizedMessage message =
          new LocalizedMessage(ClientSGAFileValidator.class, "sga.cannot.read",
            new Object[] { file.getName() });
        return new ValidationError(message);
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Valida se o usurio tem permisso de acesso ao path.
   * 
   * @param parameter - objeto que encapsula o parmetro.
   * @param value - objeto que encapsula os paths de um SGA.
   * @param context contexto da validao.
   * @return resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private Validation validatePermissions(ClientSGAFileParameter parameter,
    ClientSGAFile value, ValidationContext context) throws RemoteException {

    List<ClientSGAFile> files = new ArrayList<ClientSGAFile>();
    files.add(value);
    files.addAll(value.getBrothers());

    User user = User.getUser(context.getUserId());

    for (ClientSGAFile file : files) {
      if (!file.canRead()
        || !AccessSGAPathPermission.canReadPath(user, file.getSGAName(),
          file.getStringPath())) {
        LocalizedMessage message =
          new LocalizedMessage(ClientSGAFileValidator.class, "sga.cannot.read",
            new Object[] { file.getSGAName(), file.getStringPath() });
        return new ValidationError(message);
      }
    }
    return new ValidationSuccess();
  }
}
