package csbase.logic;

import java.util.List;

/**
 * Permisso para indicar se um usurio, em um determinado sistema, pode
 * executar um ou mais algoritmos instalados no sistema atual.
 * 
 * Essa permisso possui dois tipos de atributos:
 * 
 * - sistema=<id_do_sistema> : representa a identificao (string nica) do
 * sistema em que o usurio ter permisso para executa os algoritmos
 * especificados na permisso. Para representar o sistema atual, voc deve criar
 * um atributo "sistema=." Um exemplo de uso desse atributo com o valor
 * diferente de "." seria no caso de uso do OpenDreams para execuo de comandos
 * por fora do sistema atual. Nesse caso, o mesmo valor usado pelo Opendreams
 * para representar o sistema deve ser passado para o administrador do sistema
 * atual, para ser inserido na permisso.
 * 
 * - nome=<nome do algoritmo> : representa o nome do algoritmo que o usuario
 * pode executar ou uma expresso regular. Por exemplo, para permitir executar
 * todos os algoritmos, basta criar um atributo "nome=.*".
 * 
 */
public class CategoryAlgorithmsExecutionPermission extends AttributesPermission {
  /** Atributo que define o nome do algoritmo */
  public static final String CATEGORY_NAME = "categoria=";

  /**
   * Atributo que define o identificador (string) nico do sistema em que essa
   * permisso vai ser vlida.
   */
  public static final String SYSTEM_ID = "sistema=";

  /**
   * Identificador do sistema corrente, caso nenhum sistema tenha sido
   * especificado. Esse sistema indica que a chamada remota foi originada no
   * sistema atual.
   */
  private final static String CURRENT_SYSTEM_ID = ".";

  /**
   * Constri uma permisso para execuo de algoritmos.
   */
  public CategoryAlgorithmsExecutionPermission() {
    super();
  }

  /**
   * Constri uma permisso para execuo de algoritmos.
   * 
   * @param name nome da permisso
   * @param description descrio
   * @param attributes atributos da permisso
   */
  public CategoryAlgorithmsExecutionPermission(String name, String description,
    String[] attributes) {
    super(name, description, attributes);
  }

  /**
   * Verifica se o usurio, no sistema especificado, tem permisso para executar
   * os algoritmos que pertencem a uma das categorias especificadas.
   * 
   * @param user usurio que solicita a execuo de um algoritmo que faz parte
   *        das categorias especificadas
   * @param systemId identificador do sistema, em que foi originada a chamada
   *        remota
   * @param categoriesFullNames nomes completos das categorias que contm o
   *        algoritmo a ser executado
   * 
   * @return retorna true se o usurio, no sistema especificado, tem permisso
   *         para executar o algoritmo que faz parte das categorias
   *         especificadas, caso contrrio, retorna false
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   * 
   */
  public static boolean checkSystemAndCategoriesExecPermission(User user,
    String systemId, List<String> categoriesFullNames) throws Exception {
    for (String categoryFullName : categoriesFullNames) {
      if (checkSystemAndCategoryExecPermission(user, systemId, categoryFullName)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se o usurio, no sistema especificado, tem permisso para executar
   * um algoritmo na categoria especificada.
   * 
   * @param user usurio que solicita a execuo de um algoritmo que faz parte
   *        da categoria especificada
   * @param systemId identificador do sistema, em que foi originada a chamada
   *        remota
   * @param categoryFullName nome completo da categoria que deseja verificar se
   *        o algoritmo tem permisso para ser executado
   * 
   * @return retorna true se o usurio, no sistema especificado, tem permisso
   *         para executar o algoritmo que faz parte das categorias
   *         especificadas, caso contrrio, retorna false
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   * 
   */
  public static boolean checkSystemAndCategoryExecPermission(User user,
    String systemId, String categoryFullName) throws Exception {

    // Cria o atributo que indica o sistema que originou a chamada
    String systemAttribute = CategoryAlgorithmsExecutionPermission.SYSTEM_ID
      + ((systemId == null) ? CURRENT_SYSTEM_ID : systemId);

    // Cria o atributo que indica o nome da categoria do algoritmo a ser
    // executado
    String categoryAttribute = CategoryAlgorithmsExecutionPermission.CATEGORY_NAME
      + categoryFullName;

    String[] attributes = new String[] { systemAttribute, categoryAttribute };
    CategoryAlgorithmsExecutionPermission categoryExecPermission = (CategoryAlgorithmsExecutionPermission) user
      .getMatchAttributesPermission(
        CategoryAlgorithmsExecutionPermission.class, attributes);
    if (categoryExecPermission != null) {
      return true;
    }
    return false;
  }
}