/*
 * $Id: Session.java 157822 2014-10-30 13:25:28Z pietroguedes $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.Remote;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Representa uma sesso de um usurio logado. Esse objeto deve existir durante
 * todo o perodo em que o servidor estiver sendo utilizado.
 */
public final class Session implements Serializable {
  /**
   * O usurio da sesso.
   */
  private final User user;
  /**
   * Chave
   */
  private final Serializable key;
  /**
   * Atributos.
   */
  final private Map<String, Serializable> attributes;

  /**
   * Spy
   */
  final private Remote spy;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object object) {
    if (object == null) {
      return false;
    }
    if (!(object.getClass().equals(Session.class))) {
      return false;
    }
    final Session session = (Session) object;
    return this.key.equals(session.key);
  }

  /**
   * Obtm um atributo da sesso.
   * 
   * @param attributeName O nome do atributo.
   * 
   * @return O atributo da sesso, ou null, caso no exista.
   */
  public Serializable getAttribute(final String attributeName) {
    return this.attributes.get(attributeName);
  }

  /**
   * Retorna a coleo de atributos.
   * 
   * @return atributos
   */
  public Map<String, Serializable> getAttributes() {
    return Collections.unmodifiableMap(this.attributes);
  }

  /**
   * Getter.
   * 
   * @return Chave da sesso corrente.
   */
  public Object getKey() {
    return this.key;
  }

  /**
   * Getter.
   * 
   * @return O usurio da sesso corrente.
   */
  public User getUser() {
    return this.user;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.key.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.key.toString();
  }

  /**
   * Cria uma sesso com a opo de um espio de conexo. Esse espio  um
   * objeto remoto que pode ser criado para monitorar a conexo com o cliente
   * atravs da callback {@link java.rmi.server.Unreferenced#unreferenced()}.
   * 
   * @param user O usurio da sesso.
   * @param key Chave de identificao de uma sesso.
   * @param spy Espio da conexo.
   * @param attributes Os atributos da sesso.
   */
  public Session(final User user, final Serializable key, final Remote spy,
    final Map<String, Serializable> attributes) {
    this.user = user;
    this.key = key;
    this.spy = spy;
    if (attributes == null) {
      this.attributes = new HashMap<String, Serializable>();
    }
    else {
      this.attributes = new HashMap<String, Serializable>(attributes);
    }
  }
}
