/*
 * $Author: lmoreira $ $Date: 2005/07/14 14:28:19 $ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.BooleanParameterValidator;

/**
 * Parmetro do tipo booleano.
 *
 * @author lmoreira
 */
public final class BooleanParameter extends SimpleParameter<Boolean> {
  /**
   * A representao textual do tipo do parmetro.
   */
  public static final String TYPE = "BOOLEAN";

  /**
   * O valor-padro da linha de comando para um parmetro cujo valor 
   * verdadeiro.
   */
  private static final String DEFAULT_TRUE_COMMAND = "1";

  /**
   * O valor-padro da linha de comando para um parmetro cujo valor  falso.
   */
  private static final String DEFAULT_FALSE_COMMAND = "0";

  /**
   * O valor da linha de comando para um parmetro cujo valor  verdadeiro.
   */
  private String trueCommand;

  /**
   * O valor da linha de comando para um parmetro cujo valor  falso.
   */
  private String falseCommand;

  /**
   * Cria um parmetro booleano.
   *
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param falseCommand O valor da linha de comando para valor corrente igual a
   *        {@code false} (Aceita {@code null} - se for nulo o valor ser
   *        {@value #DEFAULT_FALSE_COMMAND}).
   * @param trueCommand O valor da linha de comando para valor corrente igual a
   *        {@code true} (Aceita {@code null} - se for nulo o valor ser
   *        {@value #DEFAULT_TRUE_COMMAND}).
   */
  public BooleanParameter(String name, String label, String description,
    boolean defaultValue, boolean isVisible, String commandLinePattern,
    String falseCommand, String trueCommand) {
    super(name, label, description, defaultValue, false, isVisible,
      commandLinePattern);
    setFalseCommand(falseCommand);
    setTrueCommand(trueCommand);
  }

  /**
   * Alterna os valores deste parmetro, ou seja, se estiver {@code true}, ele
   * se tornar {@code false} e vice-versa.
   */
  public void toggleValue() {
    Boolean value = getValue();
    setValue(new Boolean(!value.booleanValue()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return getValue();
  }

  /**
   * Obtm o valor da linha de comando quando o parmetro tem como valor falso.
   *
   * @return .
   */
  public String getFalseCommand() {
    return this.falseCommand;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Obtm o valor da linha de comando quando o parmetro tem como valor
   * verdadeiro.
   *
   * @return .
   */
  public String getTrueCommand() {
    return this.trueCommand;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue == null) {
      setValue(getDefaultValue());
    }
    else if (parameterValue.equals(Boolean.TRUE.toString())) {
      setValue(new Boolean(true));
    }
    else if (parameterValue.equals(Boolean.FALSE.toString())) {
      setValue(new Boolean(false));
    }
    else {
      throw new ParseException(
        MessageFormat
        .format(
          "O valor {0} atribudo ao parmetro {1} no  um valor booleano vlido.",
          new Object[] { this, parameterValue }));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandValue(CommandLineContext context) {
    Boolean value = getValue();
    if (value.booleanValue()) {
      return getTrueCommand();
    }
    return getFalseCommand();
  }

  /**
   * <p>
   * Atribui o valor da linha de comando quando o valor do parmetro  falso.
   * </p>
   *
   * <p>
   * Se o parmetro {@code falseCommand} for {@code null}, este valor ser igual
   * a {@link #DEFAULT_FALSE_COMMAND} ({@value #DEFAULT_FALSE_COMMAND})
   * </p>
   *
   * @param falseCommand O valor (Aceita {@code null}).
   */
  private void setFalseCommand(String falseCommand) {
    if (falseCommand == null) {
      this.falseCommand = DEFAULT_FALSE_COMMAND;
    }
    else {
      this.falseCommand = falseCommand;
    }
  }

  /**
   * <p>
   * Atribui o valor da linha de comando quando o valor do parmetro 
   * verdadeiro.
   * </p>
   *
   * <p>
   * Se o parmetro {@code trueCommand} for {@code null}, este valor ser igual
   * a {@link #DEFAULT_TRUE_COMMAND} ({@value #DEFAULT_TRUE_COMMAND})
   * </p>
   *
   * @param trueCommand O valor (Aceita {@code null}).
   */
  private void setTrueCommand(String trueCommand) {
    if (trueCommand == null) {
      this.trueCommand = DEFAULT_TRUE_COMMAND;
    }
    else {
      this.trueCommand = trueCommand;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public BooleanParameterValidator createParameterValidator() {
    return new BooleanParameterValidator();
  }
}
