package csbase.logic.algorithms.parameters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.exception.ParseException;

/**
 * Coluna de Enumerao.
 * 
 * @author lmoreira
 */
public final class EnumerationColumn extends Column<EnumerationItem> {

  /**
   * Os tems da enumerao.
   */
  private List<EnumerationItem> items;

  /**
   * Cria a coluna.
   * 
   * @param id O identificador da coluna (No aceita {@code null}).
   * @param label O rtulo da coluna (No aceita {@code null}).
   * @param defaultItem O valor-padro da coluna (No aceita {@code null}).
   * @param items Os tens da enumerao (No aceita {@code null} e no pode
   *        estar vazio).
   */
  public EnumerationColumn(String id, String label,
    EnumerationItem defaultItem, List<EnumerationItem> items) {
    super(id, label, defaultItem, false, true);
    setItems(items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(EnumerationItem itemValue) {
    return itemValue.getValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getItemValueAsText(EnumerationItem itemValue) {
    return itemValue.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getItemValueFromText(String itemValue)
    throws ParseException {
    if (itemValue == null) {
      return null;
    }
    for (EnumerationItem item : items) {
      if (itemValue.equals(item.getId())) {
        return item;
      }
    }
    throw new ParseException(
      "O tem cujo identificador  {0} no foi encontrado.", itemValue);
  }

  /**
   * <p>
   * Obtm os tens desta coluna.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return a lista.
   */
  public List<EnumerationItem> getItems() {
    return Collections.unmodifiableList(items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueToExport(EnumerationItem itemValue) {
    if (itemValue == null) {
      return null;
    }
    return itemValue.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getValueToImport(Object importedValue) {
    String enumerationItemId = (String) importedValue;
    if (enumerationItemId == null) {
      return null;
    }
    for (EnumerationItem item : items) {
      if (enumerationItemId.equals(item.getId())) {
        return item;
      }
    }
    throw new IllegalArgumentException(String.format(
      "O valor %s no  vlido para a coluna de enumerao %s.",
      enumerationItemId, getLabel()));
  }

  /**
   * Atribui os tems desta coluna.
   * 
   * @param items Os tens da enumerao (No aceita {@code null} e no pode
   *        estar vazio).
   */
  private void setItems(List<EnumerationItem> items) {
    if (items == null) {
      throw new IllegalArgumentException("O parmetro items est nulo.");
    }
    if (items.isEmpty()) {
      throw new IllegalArgumentException("O parmetro items est vazio.");
    }
    this.items = new ArrayList<EnumerationItem>(items);
  }
}
