package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.DoubleColumn;
import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros <T> com
 * valores numricos reais como {@link DoubleParameter} e
 * {@link DoubleListParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractDoubleParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #DOUBLE_ELEMENT_INCLUDE_MAXIMUM_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica se o valor mximo deve ser considerado um valor vlido,  opcional,
   * o valor-padro  {@link #DOUBLE_ELEMENT_INCLUDE_MAXIMUM_DEFAULT_VALUE} e 
   * do tipo booleano.
   * </p>
   */
  static final String DOUBLE_ELEMENT_INCLUDE_MAXIMUM_ATTRIBUTE =
    "incluir_maximo";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #DOUBLE_ELEMENT_INCLUDE_MAXIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #DOUBLE_ELEMENT_INCLUDE_MAXIMUM_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean DOUBLE_ELEMENT_INCLUDE_MAXIMUM_DEFAULT_VALUE = true;

  /**
   * <p>
   * O atributo {@value #DOUBLE_ELEMENT_INCLUDE_MINIMUM_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica se o valor mnimo deve ser considerado um valor vlido,  opcional,
   * o valor-padro  {@link #DOUBLE_ELEMENT_INCLUDE_MINIMUM_DEFAULT_VALUE} e 
   * do tipo booleano.
   * </p>
   */
  static final String DOUBLE_ELEMENT_INCLUDE_MINIMUM_ATTRIBUTE =
    "incluir_minimo";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #DOUBLE_ELEMENT_INCLUDE_MINIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #DOUBLE_ELEMENT_INCLUDE_MINIMUM_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean DOUBLE_ELEMENT_INCLUDE_MINIMUM_DEFAULT_VALUE = true;

  /**
   * <p>
   * O atributo {@value #DOUBLE_ELEMENT_MAXIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o valor mximo,  opcional e  do tipo real.
   * </p>
   */
  static final String DOUBLE_ELEMENT_MAXIMUM_ATTRIBUTE = "maximo";

  /**
   * <p>
   * O atributo {@value #DOUBLE_ELEMENT_MINIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link DoubleColumn} de um {@link TableParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o valor mnimo,  opcional e  do tipo real.
   * </p>
   */
  static final String DOUBLE_ELEMENT_MINIMUM_ATTRIBUTE = "minimo";

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group) throws ParseException {
    Double maximum =
      parser.extractAttributeValueAsDouble(DOUBLE_ELEMENT_MAXIMUM_ATTRIBUTE,
        null, null, null);
    boolean isMaximumIncluded =
      parser.extractAttributeValueAsBoolean(
        DOUBLE_ELEMENT_INCLUDE_MAXIMUM_ATTRIBUTE,
        DOUBLE_ELEMENT_INCLUDE_MAXIMUM_DEFAULT_VALUE);
    Double minimum =
      parser.extractAttributeValueAsDouble(DOUBLE_ELEMENT_MINIMUM_ATTRIBUTE,
        null, maximum, null);
    boolean isMinimumIncluded =
      parser.extractAttributeValueAsBoolean(
        DOUBLE_ELEMENT_INCLUDE_MINIMUM_ATTRIBUTE,
        DOUBLE_ELEMENT_INCLUDE_MINIMUM_DEFAULT_VALUE);
    return createDoubleParameter(parser, name, label, description, isOptional,
      isVisible, commandLinePattern, maximum, isMaximumIncluded, minimum,
      isMinimumIncluded);
  }

  /**
   * Cria uma instncia do parmetro de tipo <T>, a partir dos atributos bsicos
   * de um parmetro numrico do tipo real. As subclasses devem implementar esse
   * mtodo, fazendo a extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param maximum Valor mximo aceito pelo parmetro.
   * @param isMaximumIncluded Indica se o valor mximo est entre os valores
   *        vlidos do parmetro.
   * @param minimum Valor mnimo aceito pelo parmetro.
   * @param isMinimumIncluded Indica se o valor mnimo est entre os valores
   *        vlidos do parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createDoubleParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, Double maximum, boolean isMaximumIncluded,
    Double minimum, boolean isMinimumIncluded) throws ParseException;

}
