package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.InputFileListParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;

/**
 * 
 * <p>
 * Fbrica-base para a criao de parmetros do tipo arquivo. Como os parmetros
 * para arquivos simples ou lista de arquivos podem ser criados a partir de uma
 * mesma tag XML, essa fbrica foi criada para direcionar a anlise para o
 * parser apropriado para cada um dos casos.
 * </p>
 */
public abstract class FileParameterFactory implements ParameterFactory {

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE}
   * dos elementos {@link InputFileParameter} e {@link InputFileListParameter}.
   * Indica se o parmetro aceita seleo mltipla. Caso positivo, o parmetro
   * deve ser instanciado como {@link InputFileListParameter}, caso contrrio,
   * ser um {@link InputFileParameter}.
   * </p>
   * 
   * Indica se o arquivo de entrada aceita seleo mltipla,  opcional, o seu
   * valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE} e  do
   * tipo booleano. </p>
   */
  public static final String FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE =
    "selecao_multipla";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE} dos elementos
   * {@link InputFileParameter} e {@link InputFileListParameter}.
   * </p>
   * <p>
   * O seu valor 
   * {@value #FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE}.
   * </p>
   */
  public static final boolean FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public Parameter<?> createParameter(XmlParser parser,
    String commandLinePattern, ParameterGroup group) throws ParseException {
    boolean isMultipleSelection =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE);
    AbstractFileParameterParser<?> fileParser = null;
    if (isMultipleSelection) {
      fileParser = createFileListParser();
    }
    else {
      fileParser = createSimpleFileParser();
    }
    return fileParser.createParameter(parser, commandLinePattern, group);
  }

  /**
   * Cria o parser de parmetro para arquivo simples.
   * 
   * @return o parser.
   */
  protected abstract AbstractFileParameterParser<?> createFileListParser();

  /**
   * Cria o parser de parmetro para lista de arquivos.
   * 
   * @return o parser.
   */
  protected abstract AbstractFileParameterParser<?> createSimpleFileParser();

}
