/*
 * RemoteObservable.java
 */
package csbase.remote;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * A interface <code>RemoteObservable</code> representa um objeto que pode ser
 * observado remotamente. Um objeto observado pode ter um ou mais observadores
 * remotos. Um observador remoto precisa implementar a interface
 * {@link RemoteObserver} .
 * 
 * @author Maria Julia Lima
 * @version $Revision: 116055 $
 */
public interface RemoteObservable extends Remote {
  /**
   * Inclui um novo observador remoto. Um observador remoto  notificado sempre
   * que ocorrer alguma mudana nesse objeto observado. No existe especificao
   * sobre a ordem em que as notificaes so enviadas. O arg  associado ao
   * observador. Este objeto pode fazer filtro da notificao, fazendo com que o
   * mtodo notifyObservers notifique apenas os observadores interessados numa
   * determinada notificao.
   * 
   * @param observer o novo observador remoto
   * @param arg Argumento do observador que indica quando um observador deve
   *        receber uma notificao.
   * 
   * @throws RemoteException
   */
  void addObserver(RemoteObserver observer, Object arg) throws RemoteException;

  /**
   * Remove um observador remoto da lista de observadores registrados.
   * 
   * @param observer o observador remoto a ser removido
   * @param arg o argumento do observador
   * @return <code>true</code> se o observador foi removido
   * 
   * @throws RemoteException
   */
  boolean deleteObserver(RemoteObserver observer, Object arg)
    throws RemoteException;

  /**
   * Envia uma notificao para todos os observadores registrados quando o
   * objeto tiver alguma mudana indicada pelo mtodo <code>hasChanged</code>.
   * 
   * @param event um evento qualquer para a notificao de mudana
   * @throws RemoteException
   */
  public void notifyObservers(RemoteEvent event) throws RemoteException;

  /**
   * Remove todos os observadores registrados nesse objeto.
   * 
   * @throws RemoteException
   */
  public void deleteObservers() throws RemoteException;

  /**
   * Retorna o nmero de observadores registrados nesse objeto.
   * 
   * @return o nmero de observadores
   * @throws RemoteException
   */
  public int countObservers() throws RemoteException;
}
