package csbase.console;

import java.io.File;
import java.io.InputStream;

import csbase.client.ClientLocalFile;
import csbase.exception.PermissionException;
import csbase.logic.ClientFile;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Cliente sem interface grfica, para realizar a importao de uma verso de
 * Algoritmo. Esse cliente deve ser executado diretamente da linha de comando.
 *
 * TODO: Avaliar se vale a pena substituir ClientLocalFile por RandomAccessFile,
 * para remover dependncia de csbase-client.
 *
 * @author Tecgraf
 */
class ImportAlgoVersion extends AbstractConsoleApp {

  /**
   * Construtor do cliente para importar uma verso de algoritmo. Obtm a senha
   * interativamente.
   *
   * @param args parmetros fornecidos na linha de comando
   */
  ImportAlgoVersion(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ImportAlgoVersionParams createParams() {
    return new ImportAlgoVersionParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void postLoginInit() {
  }

  /**
   * Efetua a transferncia do arquivo local atravs do canal remoto.
   *
   * @param file referncia para o arquivo local (aplicao standalone)
   * @param info informaes do canal remoto de transferncia (FTC)
   * @return true se a transferncia ocorreu com sucesso, caso contrrio,
   *         retorna false
   * @throws Exception Em caso de falha na operao.
   */
  private static boolean transfer(ClientFile file, RemoteFileChannelInfo info)
    throws Exception {
    SyncRemoteFileChannel channel = new SyncRemoteFileChannel(info
      .getIdentifier(), info.isWritable(), info.getHost(), info.getPort(), info
        .getKey());
    channel.open(!info.isWritable());
    try (InputStream inputStream = file.getInputStream()) {
      long totalBytes = channel.syncTransferFrom(inputStream, 0, file.size());
      return (totalBytes == file.size());
    }
    finally {
      channel.close();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean preLogout() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    ImportAlgoVersionParams params = (ImportAlgoVersionParams) getParams();
    String login = (String) (params.userLogin == null ? User.getAdminId()
      : params.userLogin);
    return login;
  }

  /**
   * Obtm o arquivo local que ser copiado para o diretrio remoto.
   *
   * @param localSrcPath path para o arquivo local
   * @return o arquivo local, ou <code>null</code> caso este no exista ou seja
   *         um diretrio
   */
  private File getLocalSrcFile(String localSrcPath) {
    File localFile;
    if (localSrcPath.charAt(0) != '/') {
      localFile = new File(getCurrentDir(), localSrcPath);
    }
    else {
      localFile = new File(localSrcPath);
    }
    if (!localFile.exists()) {
      printError(localSrcPath + " no existe");
      return null;
    }
    if (localFile.isDirectory()) {
      printError(localSrcPath + "  um diretrio");
      return null;
    }
    return localFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExitCode execute() throws Exception {
    ImportAlgoVersionParams params = (ImportAlgoVersionParams) getParams();
    return startAlgorithmImport(params.algorithmId, params.versionZipPath);
  }

  /**
   * Realiza a transferncia da verso do algoritmo para o servidor.
   *
   * @param algorithmId Id do algoritmo.
   * @param zipPath caminho do arquivo zipado da verso do algoritmo.
   * @return Cdigo de sada.
   * @throws Exception
   */
  private ExitCode startAlgorithmImport(String algorithmId, String zipPath)
    throws Exception {
    /*
     * TODO: Cast pra Object necessrio pois alm do getInfo(Object) desejado,
     * existe um mtodo getInfo(String) que espera no o id mas o nome do
     * algoritmo. Talvez fosse o caso destes mtodos terem nomes diferentes.
     */
    AlgorithmInfo info = ClientRemoteLocator.algorithmService.getInfo(
      (Object) algorithmId);
    if (info == null) {
      System.err.println("Id de algoritmo invlido.");
      // Exit code de id de algoritmo invlido.
      return ExitCode.FAILURE;
    }

    File localFile = getLocalSrcFile(zipPath);
    if (localFile == null) {
      System.err.println("Path inexistente para verso de algoritmo.");
      // Exit code de path invlido para arquivo a ser importado.
      return ExitCode.FAILURE;
    }

    String filename = FileUtils.getFileName(zipPath);

    try {
      RemoteFileChannelInfo channelInfo = ClientRemoteLocator.algorithmService
        .prepareUploadVersionPack(algorithmId, filename);

      ClientFile clientFile = new ClientLocalFile(localFile);
      if (!transfer(clientFile, channelInfo)) {
        System.out.println("Falha na transferncia da verso de algoritmo.");
        // Exit code de falha na importao do algoritmo.
        return ExitCode.FAILURE;
      }
      System.out.println("Verso de algoritmo importada com sucesso.");
      // Exit code de sucesso.
      return ExitCode.SUCCESS;
    }
    catch (PermissionException e) {
      System.out.println(
        "Usurio no tem permisso para incluir verso de algoritmo "
          + algorithmId);
      // Exit code de falha de permisso na incluso de verso de algoritmo.
      return ExitCode.FAILURE;
    }
  }

}
