package csbase.console;

import java.util.Hashtable;

import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.remote.ClientRemoteLocator;

/**
 * Cliente sem interface grfica, para realizar criao de Algoritmo. Esse
 * cliente deve ser executado diretamente da linha de comando.
 *
 * @author Tecgraf
 */
class CreateAlgorithm extends AbstractConsoleApp {

  /**
   * Construtor do cliente para criar um algoritmo. Obtm a senha
   * interativamente.
   *
   * @param args parmetros fornecidos na linha de comando
   */
  CreateAlgorithm(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected CreateAlgorithmParams createParams() {
    return new CreateAlgorithmParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void postLoginInit() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean preLogout() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    CreateAlgorithmParams params = (CreateAlgorithmParams) getParams();
    String login = (String) (params.userLogin == null ? User.getAdminId()
      : params.userLogin);
    return login;
  }

  /**
   * Interpreta os parmetros e faz as respectivas requisies ao servidor.
   *
   * @throws Exception
   */
  @Override
  public ExitCode execute() throws Exception {
    CreateAlgorithmParams params = (CreateAlgorithmParams) getParams();
    Hashtable<String, String> table = new Hashtable<>();
    String errorMsg = extractAttributeTable(params.attributes, table);
    if (errorMsg != null) {
      System.out
        .println("Formato do parmetro \"-a (atributos)\" no  vlido.");
      System.out.println(errorMsg);
      // Exit code de formato de parmetro invlido.
      return ExitCode.FAILURE;
    }
    return startAlgorithmCreation(params.algorithmId, params.algorithmName,
      table);
  }

  /**
   * Extrai os atributos recebidos como parmetro e os adiciona a tabela de
   * atributos.
   *
   * @param attributes String com os atributos recebidos como parmetro.
   * @param table Tabela de atributos a ser preenchida.
   * @return String correspondente ao erro no formato da string de atributos, ou
   *         null caso a string esteja corretamente formatada.
   */
  private static String extractAttributeTable(String attributes,
    Hashtable<String, String> table) {
    if (attributes == null || attributes.trim().isEmpty()) {
      return null;
    }
    String[] attArray = attributes.split(",");
    for (String attribute : attArray) {
      String[] entry = attribute.split("=");
      // Cada atributo separado por "," deve ter o formato <nome> = <valor>.
      if (entry.length != 2) {
        return "O atributo " + entry + " no tem o formato <nome> = <valor>";
      }
      String key = entry[0].trim();
      String value = entry[1].trim();
      // Nenhum atributo extra pode ter como nome as tags "id" ou "name".
      if (key.equalsIgnoreCase("id") || key.equalsIgnoreCase("nome")) {
        return "Nenhum atributo pode ter como id as tags reservadas \"id\" ou"
          + " \"nome\".";
      }
      table.put(key, value);
    }
    return null;
  }

  /**
   * Realiza a criao de um algoritmo.
   *
   * @param algorithmId Id do algoritmo.
   * @param algorithmName Nome do alghoritmo.
   * @param attributeTable Mapa de atributos do algoritmo.
   * @return Cdigo de trmino da execuo do comando.
   * @throws Exception
   */
  private static ExitCode startAlgorithmCreation(String algorithmId,
    String algorithmName, Hashtable<String, String> attributeTable)
    throws Exception {
    /*
     * TODO: Cast pra Object necessrio pois alm do getInfo(Object) desejado,
     * existe um mtodo getInfo(String) que espera no o id mas o nome do
     * algoritmo. Talvez fosse o caso destes mtodos terem nomes diferentes.
     *
     * TODO 2: Rever o que fazer no caso de criao de algoritmo j existente.
     * Ver como o gerenciador de algoritmos trata isso.
     */
    AlgorithmInfo info =
      ClientRemoteLocator.algorithmService.getInfo((Object) algorithmId);
    if (info != null) {
      System.err.println("Id de algoritmo j existente.");
      // Exit code de algoritmo j existente (operao invlida).
      return ExitCode.FAILURE;
    }

    AlgorithmInfo algorithm = ClientRemoteLocator.algorithmService
      .createAlgorithm(algorithmName, algorithmId, attributeTable);
    if (algorithm == null) {
      System.out.println("Falha na criao do algoritmo: " + algorithmId);
      // Exit code de falha na criao de algoritmo.
      return ExitCode.FAILURE;
    }
    System.out.println("Algoritmo " + algorithmId + " criado com sucesso.");
    // Exit code de operao realizada com sucesso.
    return ExitCode.SUCCESS;
  }

}
