package csbase.rest.adapter.job.v1.DAO;

import csbase.logic.CommandInfo;
import csbase.rest.adapter.job.v1.JobUtils;
import csbase.server.Server;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import ibase.common.ServiceUtil;
import ibase.rest.api.job.v1.adapter.JobDAO;
import ibase.rest.model.job.v1.Job;
import ibase.rest.model.job.v1.JobSession;

import java.text.MessageFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

/**
 * DAO que utiliza o Servio de Persistncia de comandos
 * para obter a lista de jobs.
 */
public class CommandPersistJobDAO implements JobDAO {

  /**Mapa dos jobs por jobId. */
  private Map<String, Job> jobs = new HashMap<>();

  private Set<String> projectIds = new HashSet<>();

  @Override
  public boolean containsJob(String jobId) {
    return jobs.containsKey(jobId);
  }

  @Override
  public List<Job> findAllJobs() {
    return jobs.values().stream().collect(Collectors.toList());
  }

  @Override
  public Job findJobById(String jobId) {
    return jobs.get(jobId);
  }

  @Override
  public List<Job> findJobs(String projectId, Long modifiedAfter) {
    if (jobs.isEmpty() || !projectIds.contains(projectId)) {
      loadCommandInfos(projectId);
    }
    List<Job> filteredJobs = jobs.values().stream().filter(j -> {
      if (projectId != null && !j.getProjectId().equals(projectId)) {
        return false;
      }
      if (modifiedAfter != null && LocalDateTime.parse(j.getLastModifiedTime())
                                                .isBefore(LocalDateTime.ofInstant(Instant.ofEpochMilli(modifiedAfter),
                                                  ZoneId.systemDefault()))) {
        return false;
      }
      return true;
    }).collect(Collectors.toList());
    Server.logInfoMessage(MessageFormat.format("Filtered {0} jobs for project {1}", filteredJobs.size(), projectId));
    return filteredJobs;
  }

  @Override
  public boolean insertJob(Job job) {
    jobs.put(job.getJobId(), job);
    return true;
  }

  @Override
  public boolean updateJob(Job job) {
    Job oldJob = jobs.get(job.getJobId());
    jobs.put(job.getJobId(), job);
    return true;
  }

  /***
   * Carrega os commando com base no identificador do projeto.
   *
   * @param encodedProjectId Identificador do projeto codificado em base64.
   */
  private void loadCommandInfos(String encodedProjectId) {
    String projectId = ServiceUtil.decodeFromBase64(encodedProjectId);
    Set<CommandInfo> commandInfos = CommandPersistenceService.getInstance().getCommandInfos(projectId);
    Server.logInfoMessage(MessageFormat.format("Retrieved {0} jobs for project {1}", commandInfos.size(), projectId));
    commandInfos
      .stream()
      .map(commandInfo -> JobUtils.createJob(commandInfo, commandInfo.getStatus()))
      .filter(Objects::nonNull)
      .forEach(
        job -> {
          insertJob(job);
          projectIds.add(job.getProjectId());
        });
    Server.logInfoMessage(MessageFormat.format("Currently there are {0} jobs loaded", jobs.size()));
  }


  @Override
  public List<Job> findJobsBySession(String s) {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public boolean containsJobSession(String sessionId) {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public List<JobSession> findAllJobSessions() {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public JobSession findJobSessionById(String sessionId) {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public void insertJobSession(JobSession jobSession) {
    throw new RuntimeException("Not implemented");
  }
}
