package csbase.rest.adapter.job.v1;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.Executors;

import csbase.logic.CommandInfo;
import csbase.logic.User;
import csbase.server.Service;
import csbase.server.services.sgaservice.SGAService;
import ibase.rest.api.job.v1.adapter.JobDAO;
import ibase.rest.api.job.v1.adapter.JobMonitorListener;
import ibase.rest.model.job.v1.Job;

/**
 * Monitora alteraes de informaes de comando (CommandInfo).
 */
public class JobInfoMonitor {
  private JobDAO jobDAO = null;
  private Collection<JobMonitorListener> jobsListener =
    Collections.synchronizedList(new ArrayList<JobMonitorListener>());

  /**
   * Construtor
   *
   * @param jobDAO DAO de job
   */
  public JobInfoMonitor(JobDAO jobDAO) {
    this.jobDAO = jobDAO;
    Executors.newSingleThreadExecutor().execute(UpdateJobInfoThread);
  }

  /**
   * Adiciona um listener
   *
   * @param listener o listener
   */
  synchronized public void addJobMonitorListener(JobMonitorListener listener) {
    jobsListener.add(listener);
    notify();
  }

  /**
   * Remove um listener
   *
   * @param listener o listener
   */
  synchronized public void removeJobMonitorListener(
    JobMonitorListener listener) {
    jobsListener.remove(listener);
  }

  /**
   * Thread de aquisio das informaes de todos os comandos em execuo.
   * Somentes as novas informaes -- comparadas com as informaes no banco --
   * so notificadass aos listeners.
   */
  private Runnable UpdateJobInfoThread = new Runnable() {
    @Override
    public void run() {
      while (true) {
        synchronized (JobInfoMonitor.this) {
          while (jobsListener.isEmpty()) {
            try {
              JobInfoMonitor.this.wait();
            }
            catch (InterruptedException e) {
              e.printStackTrace();
            }
          }
        }

        Service.setUserId(User.getAdminId());
        Vector<CommandInfo> infos =
          SGAService.getInstance().getAllSGACommands();
        List<Job> jobs = new LinkedList<>();
        for (CommandInfo info : infos) {
          if (info.getProgressData() != null) {
            Job job = jobDAO.findJobById(info.getId());
            if (job != null && !info.getProgressData().getDescription()
              .equals(job.getProgressInfo())) {
              job.setProgressInfo(info.getProgressData().getDescription());
              job.setSpecificData(info.getSpecificData());
              job.setLastModifiedTime(LocalDateTime.now().toString());
              jobDAO.updateJob(job);
              jobs.add(job);
            }
          }
        }

        synchronized (JobInfoMonitor.this) {
          if (!jobs.isEmpty()) {
            // Faz uma cpia da lista de listeners, pois os prprios listeners
            // se retiram da lista, causando a exceo
            // ConcurrentModificationException
            for (JobMonitorListener l : jobsListener
              .toArray(new JobMonitorListener[0])) {
              l.infoChanged(jobs);
            }
          }
        }

        try {
          Thread.sleep(1000);
        }
        catch (Exception e) {
          //faz nada, pois a thread nao pode parar
        }
      }
    }
  };
}
