package csbase.rest.adapter.job.v1;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.server.services.commandpersistenceservice.CommandStatusListener;
import ibase.rest.api.job.v1.adapter.JobDAO;
import ibase.rest.api.job.v1.adapter.JobMonitorListener;
import ibase.rest.model.job.v1.Job;

/**
 * Created by mjulia on 24/10/16.
 */
public class JobStateListener implements CommandStatusListener {
  private ExecutorService executor;
  private JobDAO jobDAO;
  private Collection<JobMonitorListener> jobsListener = Collections.synchronizedList(
    new ArrayList<JobMonitorListener>());

  public JobStateListener(JobDAO jobDAO) {
    this.executor = Executors.newSingleThreadExecutor();
    this.jobDAO = jobDAO;
  }

  /**
   * Implementa uma ao para tratar a mudana de estado de um comando.`
   *
   * @param commandInfo informaes sobre o comando que mudou de estado
   */
  @Override
  public void statusChanged(final CommandInfo commandInfo, CommandStatus newStatus) {
    executor.submit(() -> {
      Job job = null;
      String commandId = commandInfo.getId();
      if (jobDAO.containsJob(commandId)) {
        job = jobDAO.findJobById(commandId);
        jobDAO.updateJob(JobUtils.updateJob(job, commandInfo, newStatus));
      }
      else {
        job = JobUtils.createJob(commandInfo, newStatus);
        jobDAO.insertJob(job);
      }
      //notifica os listeners internos
      for (JobMonitorListener listener : jobsListener.toArray(new JobMonitorListener[0])) {
        listener.statusChanged(job);
      }
    });
  }

  /**
   * Remove um listener da coleo de listeners de monitoramento de Jobs.
   * @param listener o listener que ser removido.
   */
  public void removeJobMonitorLister(JobMonitorListener listener) {
    this.jobsListener.remove(listener);
  }

  /**
   * Adiciona um listener na coleo de listeners de monitoramento de Jobs.
   * @param listener o listener que ser adicionado.
   */
  public void addJobMonitorLister(JobMonitorListener listener) {
    this.jobsListener.add(listener);

  }

}
