/*
 * $Id: ServerUtilities.java 146607 2013-11-07 12:47:01Z mjulia $
 */
package csbase.server;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * A classe <code>ServerUtilities</code> prov uma coleo de mtodos teis
 * utilizados pelas diversas classes (servios ou outros componentes) utilizadas
 * na implementao do Servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ServerUtilities {

  /**
   * Remove um arquivo ou diretrio. No caso de diretrio, remove recursivamente
   * o seu contedo.
   * 
   * @param file arquivo/diretrio a remover
   * @param removeRoot flag que indica se um diretrio deve ser removido (
   *        <code>true</code> ou apenas o seu contedo.
   * 
   * @throws Exception </code> caso no tenha sido possvel remover algum
   *         arquivo/diretrio. A mensagem da exceo indica qual o elemento no
   *         removido. Se este elemento est contido num diretrio a ser
   *         removido, a operao  abortada
   */
  public static void removeFile(File file, boolean removeRoot) throws Exception {
    if (!file.exists()) {
      return;
    }
    if (file.isDirectory()) {
      File[] children = file.listFiles();
      if (!(children == null) && (children.length > 0)) {
        for (int i = 0; i < children.length; i++) {
          removeFile(children[i], true);
        }
      }
    }
    if (removeRoot) {
      if (!file.delete()) {
        throw new Exception("Falha na remoo de " + file.getPath());
      }
    }
  }

  /**
   * Copia o contedo de uma stream de entrada para uma de sada.
   * 
   * @see FileSystem#copyFile(File, File)
   * @param in O stream de entrada.
   * @param out O stream de sada.
   * @param closeInput flag de fechamento de input.
   * @param closeOutput flag de fechamento de output.
   * 
   * @return Verdadeiro se a cpia foi feita com sucesso ou falso, caso
   *         contrrio.
   */
  public static boolean copyStream(InputStream in, OutputStream out,
    boolean closeInput, boolean closeOutput) {
    //abre uma BufferedInputStream a partir de uma InputStream
    BufferedInputStream bin = new BufferedInputStream(in);

    //abre uma BufferedOutputStream a partir de uma OutputStream
    BufferedOutputStream bout = new BufferedOutputStream(out);

    //escreve
    byte[] data = new byte[1024];
    int num;
    try {
      while ((num = bin.read(data, 0, 1024)) != -1) {
        bout.write(data, 0, num);
      }
      return true;
    }
    catch (IOException e) {
      e.printStackTrace();
      return false;
    }
    finally {
      if (closeInput) {
        try {
          //fecha entrada
          bin.close();
        }
        catch (IOException e1) {
          e1.printStackTrace();
        }
      }
      if (closeOutput) {
        try {
          //fecha sada
          bout.close();
        }
        catch (IOException e2) {
          e2.printStackTrace();
        }
      }
    }
  }

  /**
   * Substitui propriedade "nula" por string vazia.
   * 
   * @param minfo tabela de propriedades
   */
  public static void removeNull(String[][] minfo) {
    for (int i = 0; i < minfo.length; i++) {
      if ((minfo[i][1] == null) || minfo[i][1].equals("")) {
        minfo[i][1] = "-";
      }
    }
  }

  /**
   * Formata um array de itens em uma string separada por vrgulas e "e" no
   * final.
   * 
   * @param items array de itens.
   * 
   * @return String com os itens separados por vrgulas e "e" no final.
   */
  public static String format(String[] items) {
    StringBuffer result = new StringBuffer();
    result.append(items[0]);
    int execCount = items.length;
    for (int i = 1; i < execCount - 1; i++) {
      result.append(", ");
      result.append(items[i]);
    }
    if (execCount > 1) {
      result.append(" " + Server.getInstance().getString("server.util.and")
        + " ");
      result.append(items[execCount - 1]);
    }
    return result.toString();
  }
}
