/**
 * $Id: SGAService.java 167814 2015-08-28 15:50:27Z analodi $
 */

package csbase.server.services.sgaservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.Vector;
import java.util.concurrent.atomic.AtomicLong;

import org.omg.CORBA.IntHolder;

import sgaidl.CompletedCommandInfo;
import sgaidl.RetrievedInfo;
import sgaidl.SGACommand;
import sgaidl.SGADynamicInfo;
import sgaidl.SGAServer;
import sgaidl.StaticNodeInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.FormatUtils;
import csbase.exception.OperationFailureException;
import csbase.exception.PermissionException;
import csbase.logic.AccessSGAPathPermission;
import csbase.logic.AdminPermission;
import csbase.logic.ClientSGAFile;
import csbase.logic.CommandEndNotification;
import csbase.logic.CommandErrorNotification;
import csbase.logic.CommandEvent;
import csbase.logic.CommandEvent.EventType;
import csbase.logic.CommandFailedNotification;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandKilledNotification;
import csbase.logic.CommandLostNotification;
import csbase.logic.CommandNotification;
import csbase.logic.CommandStatus;
import csbase.logic.CommandSuccessNotification;
import csbase.logic.CommandWithNoExitCodeNotification;
import csbase.logic.FailureFinalizationType;
import csbase.logic.Permission;
import csbase.logic.SGAAdminPermission;
import csbase.logic.SGAInfo;
import csbase.logic.SGANotification;
import csbase.logic.SGANotificationPermission;
import csbase.logic.SGASet;
import csbase.logic.ServerExecutionPermission;
import csbase.logic.ServerGroupInfo;
import csbase.logic.SimpleCommandFinalizationInfo;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.Column;
import csbase.logic.algorithms.parameters.FileParameter;
import csbase.logic.algorithms.parameters.InputFileListParameter;
import csbase.logic.algorithms.parameters.ListParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.csfsservice.CSFSService;
import csbase.server.services.mailservice.MailService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.ProjectService;
import csbase.util.messages.Message;

/**
 * A classe <code>SGAService</code> implementa o servio de gerncia de SGAs.
 *
 * @see SGA
 * @see SGAServiceInterface
 * @author Tecgraf/PUC-Rio
 */
public class SGAService extends Service implements SGAServiceInterface {

  /**
   * Nome do diretrio de algoritmos utilizado pelo csfs.
   */
  private static final String[] CSFS_ALGORITHMS_ROOT_DIR = { "algorithms" };

  /**
   * Nome da sandbox utilizada pelo csfs.
   */
  private static final String[] CSFS_SANDBOX_ROOT_DIR = { "sandbox" };

  /**
   * Nome do subdiretrio (dentro da persistncia) que vai armazenar os grupos
   * dos usurios.
   */
  public static final String SGA_GROUPS_SUBDIR_NAME = "sga-groups";

  /** Constante para converso para milisegundos */
  private static final int MILLIS = 1000;

  /** Diretrio de persistncia para comandos. */
  private final static String COMMANDS_DIR = "commands";

  /** Intervalo efetivo, em segundos, para verificao de SGAs inativos. */
  private static int sgaWatchdogInterval;

  /** Intervalo efetivo, em segundos, para que o SGA informe que est vivo. */
  private static int sgaUpdateInterval;

  /** Intervalo efetivo (s) para aquisio de dados dos Comandos de SGAs. */
  private static int sgaCommandUpdateInterval;

  /**
   * Intervalo efetivo, em segundos, para que mais uma informao a respeito da
   * execuo dos algoritmos seja registrada no log
   */
  private static int sgaHistoricUpdateIntervalSeconds;

  /**
   * Tempo limite para a coleta da taxa de transferncia de dados pela rede
   * atravs do iperf (segundos)
   */
  private static int iperfTimeout;

  /** Indica se o benchmark de rede dos SGAs podem ser executados */
  private static boolean executeNetBenchmark;

  /** Indica o tempo de durao do benchmark de rede em segundos */
  private static int netBenchDuration;

  /** Indica o perodo de coleta das taxas de transferncia de dados em rede */
  private static int updateDataTransferRate;

  /** Flag indicativo de sada da thread de gravao dos dados dos SGAs. */
  private boolean exitAuditThread = false;

  /** Thread que faz a gravao dos dados dos SGAs */
  private Thread auditThread = null;

  /** Flag indicativo de sada da thread de aquisio de dados dos SGAs. */
  private boolean exitAcquisitionDataThread = false;

  /** Thread que faz a atualizao dos dados relativos aos SGAs */
  private Thread acquisitionDataThread = null;

  /** Thread que faz a atualizao das taxas de transferncia em rede dos SGAs */
  private TransferRateAcquisitionThread acquisitionNetDataThread = null;

  /**
   * Flag indicativo de sada da thread de gravao de histrico de execuo dos
   * SGAs
   */
  private boolean exitHistoricThread = false;

  /** Thread que faz gravao do histrico de execuo dos algoritmos */
  private Thread historicThread = null;

  /** Hashtable que contm os SGA's cadastrados no SSI (servio) */
  private Hashtable<String, SGA> registeredSGAs = null;

  /** HashSet que indica se o SGA est em fase de registro. */
  private Set<String> lockSGAs = null;

  /** Servant para a interface idl SGAManager */
  private SGAHandler sgaHandler = null;

  /** Servidor CORBA responsvel pela interao com os SGAs */
  private ORBHandler orbHandler = null;

  /** Referncia para o servio de mensagens */
  private MessageService messageService;

  //  /** Hashtable para observadores de comandos */
  //  @Deprecated
  //  private Hashtable<Object, Vector<RemoteObserver>> commandObservers;

  /** Flag indicativo que controla o log das informaes dinmicas dos SGAs. */
  private boolean doSGALog;

  /** Timeout para as chamadas aos mtodos do SGA */
  private long timeout;

  /**
   * Nmero de mquinas do cluster, alm do principal, que devem ter as
   * informaes guardadas no log. Se o valor or menor que zero, todos elementos
   * sero usados no log.
   */
  private int numberHostsLog;

  /** Prazo de validade das informaes do comando */
  private long cmdExpirationInfoDelay;

  /** Indica se os comandos devem ser executado como scripts. */
  private boolean executeCommandAsScript;

  /**
   * Indica se informaes da linha de comando devem ser logadas.
   */
  private boolean logCommandLineInfo;

  /**
   * Perodo que o CSFS deve aguardar antes de iniciar a atualizao da rea de
   * projetos, isto , a cpia dos resultados da execuo de um algoritmo na
   * sandbox para a rea de projetos. Essa espera  um <i>workaround</i> para
   * evitar um problema encontrado no EDISE, por conta de um atraso do
   * <i>Filer</i> em disponibilizar os arquivos gerados por ns do
   * <i>cluster</i>.
   */
  private long csfsUpdateProjectAreaDelay;

  /**
   * Mapa para contabilizar os tipos de trminos de execues simples. Usado na
   * coleta de estatsticas.
   *
   * @see #flowResultCounter
   */
  private final Map<CommandFinalizationType, Integer> exeResultCounter =
    new TreeMap<CommandFinalizationType, Integer>();

  /**
   * Mapa para contabilizar os tipos de trminos de execues de fluxos. Usado
   * na coleta de estatsticas.
   *
   * @see #exeResultCounter
   */
  private final Map<CommandFinalizationType, Integer> flowResultCounter =
    new TreeMap<CommandFinalizationType, Integer>();

  /**
   * Mapa para contabilizar as execues por mquina (n SGA). Usado na coleta
   * de estatsticas.
   */
  private final Map<String, Integer> sgasCounter =
    new TreeMap<String, Integer>();

  /**
   * Mapa contendo o total de comandos terminados de acordo tendo como chave o
   * tipo de finalizao do comando.
   */
  private Map<CommandEvent.EventType, AtomicLong> finishedCmdsCount =
    new HashMap<CommandEvent.EventType, AtomicLong>();

  /**
   * Retorno da instncia do servio.
   *
   * @return o servio.
   */
  final public static SGAService getInstance() {
    return (SGAService) getInstance(SERVICE_NAME);
  }

  /**
   * Retorno de todos os sgas cadastrados no SSI (SGA-Service)
   *
   * @return um vetor ordenado alfabeticamente com os nomes dos servidores
   */
  @Override
  final public Vector<String> getAllSGANames() {
    final Vector<String> sgas = new Vector<String>();
    final Enumeration<String> names = registeredSGAs.keys();
    while (names.hasMoreElements()) {
      sgas.add(names.nextElement());
    }
    Collections.sort(sgas);
    return sgas;
  }

  /**
   * Oteno das informaes de um SGA.
   *
   * @param sgaName o nome do SGA.
   * @return um vetor com as informaes.
   */
  @Override
  final public SGAInfo[] getAllInfo(final String sgaName) {
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      return null;
    }
    final SGAInfo[] info = sga.getAllInfo();
    logSGAInfo(info, "(Enviado para o cliente)");
    return info;
  }

  /**
   * Retorna as informaes de um comando em execuo.
   *
   * @param sgaName Nome do SGA
   * @param cmdId Identificador do comando.
   *
   * @return As informaes do comando.
   */
  @Override
  final public CommandInfo getSGACommand(final String sgaName,
    final String cmdId) {
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      return null;
    }
    final Command cmd = sga.getCommand(cmdId);
    if (cmd == null) {
      return null;
    }
    return cmd.createCommandInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<CommandInfo> getSGACommands(String sgaName) {
    SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      return null;
    }
    Object userId = Service.getUser().getId();
    boolean isAdmin = Service.getUser().isAdmin();
    Set<CommandInfo> commands = new HashSet<CommandInfo>();
    final Command[] sgaCmds = sga.getAllCommands();
    ProjectService projectService = ProjectService.getInstance();
    // Percorrendo os comandos do SGA
    for (Command cmd : sgaCmds) {
      // Se o usurio for administrador, o mtodo retorna todos os
      // comandos;
      // caso contrrio, pega somente os comandos do usurio.
      Object projectId = cmd.getProjectId();
      if (isAdmin || projectService.userHasAccess(projectId, userId)) {
        CommandInfo cmdInfo = cmd.createCommandInfo();
        commands.add(cmdInfo);
      }
    }
    return Collections.unmodifiableSet(commands);
  }

  /**
   * Retorna uma Vector (de CommandInfo) com informaes de todos os comandos do
   * em execuo. Se o usurio for um administrador, o mtodo retornar os
   * comandos de todos os usurios sendo executados nos diveros SGAs. Caso
   * contrrio, somente os comandos do usurio logado sero retornados.
   *
   * @return Vector de comandos sendo executados.
   */
  @Override
  final public Vector<CommandInfo> getAllSGACommands() {
    final Vector<CommandInfo> cmds = new Vector<CommandInfo>();
    final Enumeration<SGA> sgas = registeredSGAs.elements();
    final Object userId = Service.getUser().getId();
    final boolean isAdmin = Service.getUser().isAdmin();
    ProjectService projectService = ProjectService.getInstance();
    while (sgas.hasMoreElements()) {
      // Percorrendo SGAs
      final SGA sga = sgas.nextElement();
      final Command[] sgaCmds = sga.getAllCommands();
      // Percorrendo os comandos do SGA
      for (Command cmd : sgaCmds) {
        // Se o usurio for administrador, o mtodo retorna todos os
        // comandos;
        // caso contrrio, pega somente os comandos do usurio.
        Object projectId = cmd.getProjectId();
        if (isAdmin || projectService.userHasAccess(projectId, userId)) {
          CommandInfo cmdInfo = cmd.createCommandInfo();
          cmds.add(cmdInfo);
        }
      }
    }
    return cmds;
  }

  /**
   * Procura em todos os SGAs um comando com o identificador fornecido.
   *
   * @param cmdId identificador do comando
   *
   * @return o comando desejado, ou <code>null</code> caso o comando no tenha
   *         sido encontrado
   */
  private Command getCommand(Object cmdId) {
    final Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      // Percorrendo SGAs
      final SGA sga = sgas.nextElement();
      final Command[] sgaCmds = sga.getAllCommands();
      for (Command cmd : sgaCmds) {
        if (cmdId.equals(cmd.getId())) {
          return cmd;
        }
      }
    }
    return null;
  }

  /**
   * Consulta das informaes de um n do SGA.
   *
   * @param sgaName nome do servidor SGA.
   * @param index ndice relativo dentro do servidor SGA.
   *
   * @return um vetor com as informaes.
   */
  @Override
  final public SGAInfo getInfo(final String sgaName, final int index) {
    if (sgaName == null) {
      return null;
    }
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      return null;
    }
    final SGAInfo info = sga.getInfo(index);
    logSGAInfo(info, "(Enviado para o cliente)");
    return info;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ClientSGAFile> getChildren(String sgaName, String path) {
    SGA sga = registeredSGAs.get(sgaName);
    if (sga == null || !sga.isAlive()) {
      return null;
    }

    List<ClientSGAFile> allChildren = sga.getChildren(path);

    User user = Service.getUser();
    if (user == null) {
      return new ArrayList<ClientSGAFile>();
    }

    for (ClientSGAFile child : allChildren) {
      if (!AccessSGAPathPermission.canReadPath(user, sgaName, child
        .getStringPath())) {
        child.setCanRead(false);
        child.setCanExecute(false);
        child.setCanWrite(false);
      }
    }
    return allChildren;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientSGAFile getFile(String sgaName, String path) {
    SGA sga = registeredSGAs.get(sgaName);
    if (sga == null || !sga.isAlive()) {
      return null;
    }

    ClientSGAFile file = sga.getFile(path);

    User user = Service.getUser();
    if (user == null || file == null) {
      return null;
    }

    if (!AccessSGAPathPermission.canReadPath(user, sgaName, path)) {
      file.setCanRead(false);
      file.setCanExecute(false);
      file.setCanWrite(false);
    }
    return file;
  }

  /**
   * SGA atualiza seus dados dinmicos.
   *
   * @param sgaReference referncia para o SGA
   * @param sgaName nome da mquina hospedeira do SGA
   * @param dynamicInfo informaes dinmicas do SGA
   *
   * @return true se conseguiu atualizar seus dados ou false caso contrrio.
   */
  final public boolean updateSGAInfo(final SGAServer sgaReference,
    final String sgaName, SGADynamicInfo dynamicInfo) {
    final SGA sga = registeredSGAs.get(sgaName);
    if ((sga == null) || (sga.getRemoteReference() == null)
      || !sgaReference.equals(sga.getRemoteReference())) {
      return false;
    }
    sga.updateSGAInfo(dynamicInfo);
    return true;
  }

  /**
   * SGA verifica se o SSI est acessvel e se seu registro est vlido.
   *
   * @param sgaReference referncia para o SGA
   * @param sgaName nome da mquina hospedeira do SGA
   *
   * @return true se o o registro do SGA estiver vlido, false caso contrrio.
   *         Um retorno "false"  uma indicao de que o SSI esteve fora do ar,
   *         o que exige um novo registro do SGA junto ao mesmo.
   */
  final public boolean isRegistered(final SGAServer sgaReference,
    final String sgaName) {
    final SGA sga = registeredSGAs.get(sgaName);
    if ((sga == null) || (sga.getRemoteReference() == null)
      || !sgaReference.equals(sga.getRemoteReference())) {
      return false;
    }
    sga.patWatchDog();
    return true;
  }

  /**
   * SGA solicita a interrupo temporria de envio de comandos.
   *
   * @param sgaName nome do SGA
   */
  final public void setSGADisabled(final String sgaName) {
    Server.logInfoMessage("Notificao de desabilitao de SGA: " + sgaName);
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga != null) {
      sga.setEnabled(false);
    }
    else {
      Server.logWarningMessage("Desabilitao de SGA no registrado: "
        + sgaName);
    }
  }

  /**
   * SGA comunica que est apto a receber novos comandos para execuo.
   *
   * @param sgaName nome do SGA
   */
  final public void setSGAEnabled(final String sgaName) {
    final SGA sga = registeredSGAs.get(sgaName);
    Server.logInfoMessage("Notificao de habilitao de SGA: " + sgaName);
    if (sga != null) {
      sga.setEnabled(true);
    }
    else {
      Server.logWarningMessage("Habilitao de SGA no registrado: " + sgaName);
    }
  }

  /**
   * Obteno de um SGA.
   *
   * @param sgaName o nome do SGA.
   *
   * @return o conjunto representativo do SGA.
   */
  @Override
  final public SGASet getSGASet(final String sgaName) {
    if (sgaName == null) {
      return null;
    }

    Server.logFineMessage("SGA[" + sgaName + "] Obtendo SGA.");
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      Server.logFineMessage("SGA[" + sgaName + "] SGA == null");
      return null;
    }
    Server.logFineMessage("SGA[" + sgaName + "] SGA obtido.");

    boolean denySelection = false;
    try {
      denySelection = sga.meetsRequirement("deny_node_selection");
      Server.logFineMessage("SGA[" + sgaName
        + "] sga.meetsRequirement(\"deny_node_selection\") = " + denySelection);
    }
    catch (Exception e) {
      denySelection = true;
      Server
      .logFineMessage("SGA["
        + sgaName
        + "] sga.meetsRequirement(\"deny_node_selection\") = true devido a exceo. Exceo: "
        + e.getMessage());
    }

    boolean hasPermission = hasExecutionPermission(sgaName);
    Server.logFineMessage("SGA[" + sgaName + "] hasExecutionPermission("
      + sgaName + ") = " + hasPermission);

    final SGAInfo[] info = sga.getAllInfo();
    Server.logFineMessage("SGA[" + sgaName + "] Informaes do SGA obtidas "
      + Arrays.toString(info) + ".");

    logSGAInfo(info, "(Enviado para o cliente)");
    Server.logFineMessage("SGA[" + sgaName + "] Informaes logadas");

    boolean enabled = sga.getEnabled();
    Server.logFineMessage("SGA[" + sgaName + "] sga.getEnabled() = " + enabled);

    boolean alive = sga.isAlive();
    Server.logFineMessage("SGA[" + sgaName + "] sga.isAlive() = " + alive);

    boolean diskaccess = sga.hasDiskAccess();
    Server.logFineMessage("SGA[" + sgaName + "] sga.hasDiskAccess() = "
      + diskaccess);

    String jobsInfo = sga.getJobsInfo();
    Server.logFineMessage("SGA[" + sgaName + "] sga.getJobsInfo() = "
      + jobsInfo);

    boolean CSFSEnabled =
      (sga.getCSFSHost() != null) && (sga.getCSFSRootDir() != null);

    SGASet sgaSet =
      new SGASet(info, sgaName, hasPermission && enabled, alive, diskaccess,
        jobsInfo, CSFSEnabled, !denySelection);
    Server.logFineMessage("SGA[" + sgaName + "] SGASet criado.");

    return sgaSet;
  }

  /**
   * Atualiza os grupos de servidores definidos por um usurio.
   *
   * @param groups array com as informaes dos grupos
   *
   * @return true se a operao foi bem sucedida, false em caso de erro
   */
  @Override
  final public boolean setServerGroups(final ServerGroupInfo[] groups) {
    try {
      final String groupsFilePath = getUserSGAGroupFilePath();
      writeGroupsInfo(groupsFilePath, groups);
      return true;
    }
    catch (ServerException se) {
      Server.logSevereMessage("Falha atualizando grupos de servidores para "
        + Service.getUser().getId() + ": " + se.getMessage());
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean updateCommandDescription(String commandId, String description) {
    Command sgaCommand = getCommand(commandId);
    if (sgaCommand == null) {
      Server.logSevereMessage("Comando no encontrado: " + commandId);
      return false;
    }
    sgaCommand.getCommandInfo().setDescription(description);
    return true;
  }

  /**
   * Obtm os grupos de servidores definidos por um usurio.
   *
   * @return um array com as informaes dos grupos ou null em caso de erro
   */
  @Override
  final public ServerGroupInfo[] getServerGroups() {
    try {
      final String groupsFilePath = getUserSGAGroupFilePath();
      final ServerGroupInfo[] info = readGroupsInfo(groupsFilePath);
      return info;
    }
    catch (ServerException se) {
      Server.logSevereMessage("Falha obtendo grupos de servidores para "
        + Service.getUser().getId() + ": " + se.getMessage());
      return null;
    }
  }

  /**
   * Retorna o intervalo de atualizao das informaes dos SGAs
   *
   * @return intervalo de atualizao em segundos
   */
  @Override
  final public int getUpdateInterval() {
    return sgaUpdateInterval;
  }

  /**
   * Retorna o intervalo de atualizao das informaes dos comandos.
   *
   * @return intervalo de atualizao em segundos
   */
  @Override
  final public int getCommandsUpdateInterval() {
    return sgaCommandUpdateInterval;
  }

  /**
   * Atualizao de dados dos SGAs e sua gravao.
   */
  private void logAllAuditEvents() {
    final Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      final SGA sga = sgas.nextElement();
      if (sga.isAlive() && sga.getEnabled()) {
        sga.logAuditEvents();
      }
    }
  }

  /**
   * Atualizao de dados de execuo dos algoritmos.
   */
  private void logAllAlgorithmHistoricEvents() {
    final Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      final SGA sga = sgas.nextElement();
      boolean enableHistoric;
      try {
        enableHistoric = sga.meetsRequirement("enable_historic");
      }
      catch (ServerException e) {
        enableHistoric = false;
      }
      if (enableHistoric && sga.isAlive() && sga.getEnabled()) {
        sga.logAlgorithmHistoryEvents();
      }
    }
  }

  /* ------------------------------------------------------------------------ */

  /**
   * Notificao de trmino de um comando.
   *
   * @param sgaName nome do SGA.
   * @param cmdReference referncia para o comando concludo.
   * @param cmdId identificador do comando no SGA.
   * @param cmdInfo informaes sobre o comando.
   * @return true se notificao aceita, false se o SSI desconhece o SGA
   */
  final public boolean commandCompleted(final String sgaName,
    final SGACommand cmdReference, final String cmdId,
    final CompletedCommandInfo cmdInfo) {
    try {
      final SGA sga = registeredSGAs.get(sgaName);
      if (sga == null) {
        Server
        .logSevereMessage("Informao de comando terminado em um SGA no "
          + "cadastrado! [" + sgaName + "]");
        return false;
      }
      if (cmdReference == null) {
        Server.logSevereMessage("Comando nulo vindo como terminado no "
          + "servidor [" + sgaName + "]!");
        return true;
      }
      if (cmdInfo == null) {
        Server.logSevereMessage("Informao nula de fim de comando vinda "
          + "do servidor [" + sgaName + "]!");
        return true;
      }

      Command cmd = sga.getCommand(cmdId);
      CommandEvent event = null;
      if (cmd == null) {
        String errorMessage =
          "No foi encontrado no servidor um comando identificado como finalizado pelo SGA.\n";
        Server.logSevereMessage(errorMessage + "\tCausa: " + errorMessage
          + "\n\tId: " + cmdId + "\n\n");
        event =
          new CommandEvent(cmdId, EventType.INIT_FAILURE,
            new SimpleCommandFinalizationInfo(CommandFinalizationType.FAILED,
              FailureFinalizationType.COMMAND_IDENTIFIER_NOT_FOUND));

        notifyCommandEnd(null, event);
        return true;
      }

      sga.finishCommand(cmdId);

      /*
       * Por uma limitao de corba, os atributos de cmdInfo vem como -1 para
       * indicar um valor inexistente.
       */
      Integer elapsedTimeSec =
        cmdInfo.elapsedTimeSec < 0 ? null : cmdInfo.elapsedTimeSec;
      Integer userTimeSec =
        cmdInfo.userTimeSec < 0 ? null : cmdInfo.userTimeSec;
      Integer cpuTimeSec = cmdInfo.cpuTimeSec < 0 ? null : cmdInfo.cpuTimeSec;

      CommandInfo commandInfo = cmd.getCommandInfo();
      commandInfo.setWallTimeSec(elapsedTimeSec);

      if (cmd.finish(cmdInfo)) {
        EventType type;
        switch (commandInfo.getFinalizationType()) {
          case SUCCESS:
            type = EventType.SUCCESS;
            break;

          case EXECUTION_ERROR:
            type = EventType.ERROR;
            break;

          case NO_EXIT_CODE:
            type = EventType.NO_CODE;
            break;

          default:
            type = EventType.COMPLETED;
        }

        event =
          new CommandEvent(cmdId, type, commandInfo.getFinalizationInfo(),
            elapsedTimeSec, userTimeSec, cpuTimeSec);
      }
      else {
        event =
          new CommandEvent(cmdId, EventType.INIT_FAILURE, commandInfo
            .getFinalizationInfo());
      }

      notifyCommandEnd(cmd, event);
      return true;
    }
    catch (RuntimeException e) {
      String errorMessage =
        "Erro na chamada do SGA ao completar o comando " + cmdId;
      Server.logSevereMessage(errorMessage, e);
      return false;
    }
  }

  /**
   * Notificao de comando no recuperado.
   *
   * @param sgaName nome do SGA.
   * @param cmdId identificador do comando no SGA.
   * @return true se notificao aceita, false se o SSI desconhece o SGA
   */
  final public boolean commandLost(final String sgaName, final String cmdId) {
    final SGA sga = registeredSGAs.get(sgaName);
    if (sga == null) {
      Server.logSevereMessage("Informao de comando no recuperado em um "
        + "SGA no cadastrado! [" + sgaName + "]");
      return false;
    }
    Server.logWarningMessage("Informao de comando no recuperado "
      + "vinda do servidor [" + sgaName + "]!");

    // Os comandos no recuperados so dados como terminados, mas no
    // temos os tempos finais de execuo.
    Command cmd = sga.getCommand(cmdId);
    sga.lostCommand(cmdId);
    if (cmd == null) {
      Server.logSevereMessage("Informao de comando no existente "
        + "vinda do servidor [" + sgaName + "]!");
      return true;
    }
    CommandEvent event =
      new CommandEvent(cmdId, EventType.LOST, cmd.getFinalizationInfo());
    notifyCommandEnd(cmd, event);
    return true;
  }

  /**
   * Execuo remota de um comando para um SGA.
   *
   * @param command informaes sobre o comando a ser executado.
   *
   * @throws PermissionException Se o usurio no tiver permisso para executar
   *         o comando.
   * @throws SGANotAvailableException Em caso de falha na submisso do comando
   *         devido ao SGA selecionado no estar disponvel.
   * @throws ProjectNotFoundException Em caso de falha na submisso do comando
   *         devido ao projeto no estar disponvel.
   */
  final public void executeCommand(final CommandInfo command)
    throws PermissionException, SGANotAvailableException,
    ProjectNotFoundException {

    String sgaName = command.getSGAName();
    checkExecutionPermission(sgaName);
    String tip = command.getTip();

    final SGA sga = this.registeredSGAs.get(sgaName);
    if (sga == null) {
      final String err =
        "O SGA selecionado para execuo no est registrado: " + sgaName;
      final String algoStr = "Algoritmo: " + tip;
      throw new SGANotAvailableException(err + "\n\t" + algoStr + "\n");
    }
    if (!sga.mayExecuteCommand()) {
      final String err =
        "O SGA selecionado para execuo no est ativo/habilitado: " + sgaName;
      final String algoStr = "Algoritmo: " + tip;
      throw new SGANotAvailableException(err + "\n\t" + algoStr + "\n");
    }

    try {
      sga.executeCommand(null, command, cmdExpirationInfoDelay,
        executeCommandAsScript, logCommandLineInfo);
    }
    catch (ProjectNotFoundException e) {
      throw e;
    }
    finally {
      collectSGAStats(command);
    }
  }

  /**
   * Inicializao do ORB do servio de gerncia de SGAs. Cria o servidor CORBA
   * responsvel pela interao com os SGAs e ativa o objeto responsvel pela
   * implementao da interface SGAManager.
   *
   * @throws ServerException se ocorrer um erro na criao do servidor CORBA ou
   *         na ativao do servant.
   */
  final private void initORB() throws ServerException {
    try {
      Server.logInfoMessage("Acessando ambiente ORB...");
      final Properties props = getExternalPropertyFile("ORB.file");
      /*
       * As propriedades que definem o ip e porta so definidas nas propriedades
       * do servio.
       */
      if (!isPropertyNull("ORB.port")) {
        int port = getIntProperty("ORB.port");
        String portStr = Integer.toString(port);
        props.setProperty("OAPort", portStr);
      }
      if (!isPropertyNull("ORB.hostAddr")) {
        String addr = getStringProperty("ORB.hostAddr");
        props.setProperty("OAIAddr", addr);
      }
      orbHandler = new ORBHandler(props);
      Server.logInfoMessage("Acesso ao ambiente ORB inicializado");
    }
    catch (ORBException e) {
      throw new ServerException("Impossvel criar servidor CORBA: "
        + e.getMessage(), e);
    }
    try {
      sgaHandler = new SGAHandler(this);
      orbHandler.activateObject("SGAManager", sgaHandler);
    }
    catch (Exception e) {
      throw new ServerException("Impossvel criar handler de SGA's", e);
    }
    Server.logInfoMessage("Acesso ao ambiente ORB inicializado");
  }

  /**
   * Cria a thread para a coleta das taxas de transferncia de dados na rede.
   * Essa thread executa de forma independente ao processamento da fila de
   * comandos em espera para execuo nos SGAs.
   */
  void createTransferRateAcquisitionThread() {
    Server
    .logFineMessage("Preparando a criao de thread para a coleta das taxas de "
      + "transferncia em rede...");
    acquisitionNetDataThread = new TransferRateAcquisitionThread();
    acquisitionNetDataThread.setName(this.getClass().getSimpleName() + "::"
      + "TransferRateAcquisitionThread");
    acquisitionNetDataThread.start();
  }

  /**
   * Inicializao do servio de gerncia de SGAs.
   *
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  final public void initService() throws ServerException {
    initORB();
    messageService = MessageService.getInstance();

    /* Criao das threads de execuo do servio */
    createDataAcquisitionThread();
    createAuditThread();

    /* Flag de log das informaes do SGA */
    doSGALog = getBooleanProperty("sgaLogDynamicInfo");

    /* numberHostsLog < 0: todas mquinas sero usadas no log */
    numberHostsLog = getIntProperty("sgaLogNumberHostsCluster");

    /*
     * Busca da propriedade que determina o intervalo de atualizao dos dados
     * dos Comandos dos SGAs
     */
    synchronized (SGAService.class) {
      sgaCommandUpdateInterval =
        getIntProperty("sgaCommandUpdateIntervalSeconds");
    }
    cmdExpirationInfoDelay = getIntProperty("cmdExpirationInfoDelay");

    executeCommandAsScript = getBooleanProperty("executeCommandAsScript");

    logCommandLineInfo = getBooleanProperty("logCommandLineInfo");

    /* Criao da thread que alimenta log de execuo dos algoritmos */
    createExecutionHistoricThread();

    /*
     * Timeout para as chamadas de mtodo do SGA. Caso o valor seja negativo, o
     * controle de timeout  desativado.
     */
    timeout = getLongProperty("Timeout");
    final String msg =
      (timeout < 0) ? "Timeout na conexo com o SGA desativado"
        : "Timeout para as chamadas ao SGA (seg): " + timeout;
    Server.logInfoMessage(msg);
    // Tempo que o CSFS deve aguardar antes de atualizar a rea de projetos
    csfsUpdateProjectAreaDelay = getIntProperty("csfsUpdateProjectAreaDelay");

    synchronized (SGAService.class) {
      netBenchDuration = getIntProperty("netBenchDuration");
      iperfTimeout = getIntProperty("iperfTimeout") * MILLIS;
      updateDataTransferRate =
        getIntProperty("updateDataTransferRate") * MILLIS;
      executeNetBenchmark = getBooleanProperty("executeNetBenchmark");
    }

    /*
     * Criao da thread para coleta das taxas de transferncia de dados entre o
     * servidor e os SGAs, caso o benchmark de rede seja habilitado no servidor.
     */
    if (executeNetBenchmark == true) {
      Server
      .logInfoMessage("Coleta das taxas de transferncia em rede habilitada.");
      createTransferRateAcquisitionThread();
    }
    else {
      Server
      .logInfoMessage("Coleta das taxas de transferncia em rede desabilitada.");
    }
  }

  /**
   * Mtodo de notificao de todos os usurios sobre um SGA.
   *
   * @param sgaName Nome do SGA.
   * @param sgaFlag Flag de notificao - {@link SGANotification} -.
   */
  private void notifyAllUsersSGA(String sgaName, int sgaFlag) {
    List<User> users;
    try {
      users = User.getAllUsers();
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao obter usurios.", e);
      return;
    }
    for (User user : users) {
      Permission sgaAdminPermission = null;
      Permission sgaNotificationPermission = null;

      try {
        sgaAdminPermission =
          user.getMatchAttributesPermission(SGAAdminPermission.class, sgaName);
        sgaNotificationPermission =
          user.getMatchAttributesPermission(SGANotificationPermission.class,
            sgaName);
      }
      catch (Exception e) {
        String errorMsg =
          String
          .format(
            "Falha ao buscar as permisses para os usurios de gerncia e notificao sobre servidor de execuo de algoritmos %s.",
            sgaName);
        Server.logSevereMessage(errorMsg, e);
      }

      if (user.isAdmin() || sgaAdminPermission != null
        || sgaNotificationPermission != null) {

        setUserId(user.getId());
        try {
          Server.logFineMessage("Obtendo SGASet de " + sgaName);
          SGASet set = getSGASet(sgaName);
          Server.logFineMessage("SGASet obtido de " + sgaName);

          SGANotification data =
            new SGANotification(getSenderName(), sgaFlag, set);

          messageService.send(new Message(data), 0, user.getLogin());
        }
        catch (Exception e) {
          Server.logSevereMessage("Falha de notificao some-users: "
            + e.getMessage(), e);
        }
        finally {
          setUserId(null);
        }
      }
    }
  }

  /**
   * Notificao de fim de comando para um usurio (obrigatria).
   *
   * @param command comando.
   * @param event evento com as informaes do comando terminado.
   */
  void notifyCommandEndToProject(final Command command, final CommandEvent event) {
    if (command == null) {
      Server.logSevereMessage("O comando " + event.getCommandId()
        + " no foi encontrado.\n");
      return;
    }

    finishedCmdsCount.get(event.getType()).incrementAndGet();
    StringBuilder infoMsg =
      new StringBuilder("[FIM DE COMANDO] CmdId[").append(event.getCommandId())
      .append("] ").append(event.getType().name()).append(". SGAService: ");
    for (CommandEvent.EventType type : CommandEvent.EventType.values()) {
      infoMsg.append(" ").append(type.name()).append(": ").append(
        finishedCmdsCount.get(type).get());
    }
    Server.logInfoMessage(infoMsg.toString());

    CommandNotification notData;
    switch (event.getType()) {
      case COMPLETED:
        notData =
        createCommandEndNotificationData(command, event.getElapsedTime(),
          event.getUserTime(), event.getCpuTime());
        break;
      case SUCCESS:
        notData =
        createCommandSuccessNotificationData(command, event.getElapsedTime(),
          event.getUserTime(), event.getCpuTime());
        break;
      case NO_CODE:
        notData = createCommandWithNoExitCodeNotificationData(command);
        break;
      case ERROR:
        notData = createCommandErrorNotificationData(command);
        break;
      case LOST:
        notData = createCommandLostNotificationData(command);
        break;
      case KILLED:
        notData = createCommandKilledNotificationData(command);
        break;
      case INIT_FAILURE:
        notData = createCommandFailedNotificationData(command);
        break;
      default:
        Server.logSevereMessage("Tipo de evento desconhecido: "
          + event.getType() + ".\n");
        return;
    }

    messageService.sendToServer(new Message(notData));

    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(command.getProjectId());
      messageService.send(new Message(notData), users);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + command.getProjectName() + " o fim da execuo do comando ("
        + command.getId() + ")", e);
    }
  }

  /**
   * Cria um objeto com os dados de notificao de comando terminado (sem
   * informaes sobre como terminou).
   *
   * @param command O comando
   * @param elapsedTimeSec Tempo de parede.
   * @param userTimeSec Tempo em modo usurio.
   * @param cpuTimeSec Tempo em modo sistema.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandEndNotification createCommandEndNotificationData(
    final Command command, final Integer elapsedTimeSec, Integer userTimeSec,
    Integer cpuTimeSec) {
    long now = Calendar.getInstance().getTimeInMillis();
    final CommandEndNotification notData =
      new CommandEndNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), elapsedTimeSec, userTimeSec,
        cpuTimeSec, command.getStartTime(), now, command.getSGA().getName(),
        command.getProjectId(), command.getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando terminado com
   * sucesso.
   *
   * @param command O comando
   * @param elapsedTimeSec Tempo de parede.
   * @param userTimeSec Tempo em modo usurio.
   * @param cpuTimeSec Tempo em modo sistema.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandSuccessNotification createCommandSuccessNotificationData(
    final Command command, final Integer elapsedTimeSec, Integer userTimeSec,
    Integer cpuTimeSec) {
    long now = Calendar.getInstance().getTimeInMillis();
    final CommandSuccessNotification notData =
      new CommandSuccessNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), elapsedTimeSec, userTimeSec,
        cpuTimeSec, command.getStartTime(), now, command.getSGA().getName(),
        command.getProjectId(), command.getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando perdido.
   *
   * @param command O comando.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandLostNotification createCommandLostNotificationData(
    final Command command) {
    final CommandLostNotification notData =
      new CommandLostNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), command.getStartTime(), command
        .getSGA().getName(), command.getProjectId(), command
        .getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando com falha.
   *
   * @param command O comando.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandFailedNotification createCommandFailedNotificationData(
    final Command command) {
    final CommandFailedNotification notData =
      new CommandFailedNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), command.getStartTime(), command
        .getSGA().getName(), command.getProjectId(), command
        .getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando com erro de execuo.
   *
   * @param command O comando.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandErrorNotification createCommandErrorNotificationData(
    final Command command) {
    final CommandErrorNotification notData =
      new CommandErrorNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), command.getStartTime(), command
        .getSGA().getName(), command.getProjectId(), command
        .getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando cujo cdigo de
   * retorno no foi possvel encontrar.
   *
   * @param command O comando.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandWithNoExitCodeNotification createCommandWithNoExitCodeNotificationData(
    final Command command) {
    final CommandWithNoExitCodeNotification notData =
      new CommandWithNoExitCodeNotification(getSenderName(), command.getId(),
        command.getDescription(), command.getTip(), command.getStartTime(),
        command.getSGA().getName(), command.getProjectId(), command
          .getFinalizationInfo());
    return notData;
  }

  /**
   * Cria um objeto com os dados de notificao de comando interrompido.
   *
   * @param command O comando.
   *
   * @return O objeto de dados de notificao.
   */
  private CommandKilledNotification createCommandKilledNotificationData(
    final Command command) {
    final CommandKilledNotification notData =
      new CommandKilledNotification(getSenderName(), command.getId(), command
        .getDescription(), command.getTip(), command.getStartTime(), command
        .getSGA().getName(), command.getProjectId(), command
        .getFinalizationInfo());
    return notData;
  }

  /**
   * Aviso de fim de comando por mail do usurio.
   *
   * @param command dados do comando (pr-execuo).
   * @param event dados do comando (ps-execuo).
   */
  private void mailCommandEndToProject(Command command, CommandEvent event) {
    if (command == null) {
      Server.logSevereMessage("O comando " + event.getCommandId()
        + " no foi encontrado.\n");
      return;
    }
    if (event.getType() == EventType.KILLED || !command.requireMail()) {
      return;
    }
    SGA sga = command.getSGA();
    String eventTypeString =
      getString("SGAService.mail.type.command." + event.getType());

    String content =
      String.format(getString("SGAService.mail.content"), eventTypeString, sga
        .getName(), sga.getPlatformId(), command.getProjectName(), command
        .getId(), command.getDescription());

    if (event.getType() == EventType.COMPLETED
      || event.getType() == EventType.ERROR
      || event.getType() == EventType.SUCCESS
      || event.getType() == EventType.NO_CODE) {
      String unknown = getString("SGAService.value.unknown");

      Integer cpuTimeSec = event.getCpuTime();
      String cpuTimeString =
        (cpuTimeSec != null) ? FormatUtils.formatInterval(cpuTimeSec) : unknown;
        Integer userTimeSec = event.getUserTime();
        String userTimeString =
          (userTimeSec != null) ? FormatUtils.formatInterval(userTimeSec)
            : unknown;
          Integer elapsedTimeSec = event.getElapsedTime();
          String elapsedTimeString =
            (elapsedTimeSec != null) ? FormatUtils.formatInterval(elapsedTimeSec)
              : unknown;

            String executionTimeContent =
              String.format(getString("SGAService.mail.content.execution.time"),
                cpuTimeString, userTimeString, elapsedTimeString);
            // Anexa ao contedo do email os tempos finais de execuo.
            content += executionTimeContent;

            try {
              /* Obtm o configurador do algoritmo */
              AlgorithmConfigurator configurator =
                command.getCommandInfo().getConfigurator();
              if (configurator.getConfiguratorType() == ConfiguratorType.SIMPLE) {

                SimpleAlgorithmConfigurator simpleConfigurator =
                  (SimpleAlgorithmConfigurator) configurator;
                List<ParameterGroup> paramGroups = simpleConfigurator.getGroups();

                ParametersReportMaker visitor = new ParametersReportMaker();

                content +=
                  String.format(getString("SGAService.mail.content.params.title"),
                    visitor.makeReport(paramGroups));
              }
            }
            catch (RemoteException e) {
              content += getString("SGAService.mail.content.params.error");
              Server.logSevereMessage(
                "Ocorreu um erro na tentativa de obter o configurador do comando "
                  + command.getId() + ".\n", e);
            }
    }

    Object userId = command.getUserId();
    try {
      mailUser(userId, content);
    }
    catch (Exception e) {
      Server.logSevereMessage("Impossvel enviar mail para o usurio ("
        + userId + ") " + "por fim de comando (" + command.getId() + ")!\n", e);
    }
    String[] aditionalEmails = command.getEmailList();
    if (aditionalEmails != null) {
      for (String email : aditionalEmails) {
        try {
          mailUser(email, content);
        }
        catch (Exception e) {
          Server.logSevereMessage("Impossvel enviar mail para o endereo ("
            + email + ") " + "por fim de comando (" + command.getId() + ")!\n",
            e);
        }
      }
    }
  }

  /**
   * Mtodo de mail para um usurio
   *
   * @param userId um identificador
   * @param content a mensagem.
   */
  private void mailUser(final Object userId, final String content) {
    final MailService mailService = MailService.getInstance();
    if (!mailService.mailUserFromService(this, userId, content)) {
      Server.logSevereMessage("Falha de mail ao usurio " + "[" + userId
        + "]: " + content);
    }
  }

  /**
   * Mtodo de mail para um endereo de email fornecido
   *
   * @param email endereo de email
   * @param content a mensagem.
   */
  private void mailUser(String email, final String content) {
    final MailService mailService = MailService.getInstance();
    if (!mailService.mailFromService(this, new String[] { email }, content)) {
      Server.logSevereMessage("Falha de mail para o endereo " + "[" + email
        + "]: " + content);
    }
  }

  /**
   * Envio de notificao para cancelar a execuo de um comando especfico
   *
   * @param cmdId identificador do comando.
   *
   * @return true se conseguiu cancelar o comando ou false se houver falha na
   *         comunicao.
   */
  @Override
  public synchronized boolean killCommand(final String cmdId) {
    Command cmd = getCommand(cmdId);
    return killCommand(cmd);
  }

  /**
   * Envio de notificao para cancelar a execuo de um comando especfico
   *
   * @param sgaName nome do SGA.
   * @param cmdId identificador do comando.
   *
   * @return true se conseguiu cancelar o comando ou false se houver falha na
   *         comunicao.
   */
  @Override
  public synchronized boolean killCommand(final String sgaName,
    final String cmdId) {
    final SGA sga = registeredSGAs.get(sgaName);
    Command cmd = sga.getCommand(cmdId);
    return killCommand(cmd);
  }

  /**
   * Envio de notificao para cancelar a execuo de um comando especfico
   *
   * @param cmd o comando
   *
   * @return true se conseguiu cancelar o comando ou false se houver falha na
   *         comunicao.
   */
  private boolean killCommand(final Command cmd) {
    if (cmd == null) {
      return false;
    }
    if (!cmd.getStatus().equals(CommandStatus.EXECUTING)) {
      return false;
    }
    final SGA sga = cmd.getSGA();
    try {
      return doKill(cmd, sga);
    }
    catch (PermissionException pe) {
      throw pe;
    }
    catch (Exception e) {
      Server
      .logSevereMessage("Falha ao cancelar comando " + cmd.getId()
        + " do usurio " + Service.getUser().getId() + ": " + e.getMessage(),
        e);
      return false;
    }
  }

  /**
   * Pede para o SGA cancelar a execuco de um comando especfico. Retorna
   * verdadeiro se o cancelamento foi feito com sucesso ou falso, caso
   * contrrio.
   *
   * @param cmd o comando a ser cancelado.
   * @param sga o SGA que est executando o comando
   * @return verdadeiro se o cancelamento foi feito com sucesso ou falso, caso
   *         contrrio.
   * @throws ServerException caso haja falha de comunicao com o servidor
   * @throws PermissionException caso o usurio no tenha permisso para
   *         cancelar o comando.
   */
  private boolean doKill(final Command cmd, final SGA sga)
    throws ServerException, PermissionException {
    final String cmdId = cmd.getId();
    Object commandOwnerId = cmd.getUserId();
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = cmd.getProjectId();
    Object projectOwnerId = projectService.getOwnerId(projectId);
    User loggedUser = Service.getUser();
    Object loggedUserId = loggedUser.getId();
    if (!loggedUser.isAdmin() && !loggedUserId.equals(commandOwnerId)
      && !loggedUserId.equals(projectOwnerId)) {
      throw new PermissionException();
    }
    if (!sga.killCommand(cmdId)) {
      return false;
    }
    CommandEvent event =
      new CommandEvent(cmdId, EventType.KILLED, cmd.getFinalizationInfo());
    notifyCommandEnd(cmd, event);
    return true;
  }

  /**
   * Finalizao de comando pelo administrador (kill). Remove o objeto do
   * comando mesmo que tenha ocorrido erro. Deve ser usado no caso em que no h
   * mais possibilidade do comando estar executando. Por exemplo, ocorreu
   * problema de hardware na mquina.
   *
   * @param sgaName nome do SGA.
   * @param cmdId identificador do comando.
   *
   * @throws PermissionException se o usurio no for administrador.
   */
  @Override
  public synchronized void killCommandAnyway(final String sgaName,
    final String cmdId) throws PermissionException {
    checkSGAAdministrationPermission();
    final SGA sga = registeredSGAs.get(sgaName);
    Command cmd = sga.getCommand(cmdId);
    if (cmd == null) {
      return;
    }
    sga.killCommandAnyway(cmdId);
    CommandEvent event =
      new CommandEvent(cmdId, EventType.KILLED, cmd.getFinalizationInfo());
    notifyCommandEnd(cmd, event);
  }

  /**
   * lock no bloqueante para indicar se existe algum sga em fase de registro.
   * Usado para no deixar que 2 sgas com o mesmo nome tentem se registrar
   * simultaneamente (o que j seria uma situao de erro!).
   *
   * @param sgaName nome do SGA.
   *
   * @return true se conseguiu fazer o lock.
   */
  private boolean lockSGA(String sgaName) {
    return lockSGAs.add(sgaName);
  }

  /**
   * unlock para indicar que um sga saiu da fase de registro.
   *
   * @param sgaName nome do SGA.
   */
  private void unlockSGA(String sgaName) {
    lockSGAs.remove(sgaName);
  }

  /**
   * Registro de um SGA.
   *
   * @param sgaReference referncia para o sga
   * @param sgaName nome do SGA
   * @param staticNodesInfo informaes estticas dos ns
   * @param updateInterval informa de quanto em quanto tempo o SGA deve dizer
   *        que est vivo.
   *
   * @return true se o o registro do SGA for aceito, false caso o registro no
   *         possa ser aceito
   */
  public boolean registerSGA(SGAServer sgaReference, String sgaName,
    StaticNodeInfo[] staticNodesInfo, IntHolder updateInterval) {
    Server.logInfoMessage("Detectado pedido de registro de SGA: " + sgaName);
    updateInterval.value = sgaUpdateInterval;
    try {
      if (!lockSGA(sgaName)) {
        // J existe um SGA com este nome fazendo registro!!
        Server.logSevereMessage("Falha de registro de SGA: " + sgaName
          + ". SGA j est em fase de registro.");
        return false;
      }
      SGA sga = registeredSGAs.get(sgaName);
      if (sga != null && sga.getRemoteReference() != null
        && !sgaReference.equals(sga.getRemoteReference())) {
        // J existe um SGA com este nome cadastrado e ativo. No aceita
        // cadastro do novo SGA!!
        Server.logWarningMessage("Detectado SGA " + sgaName
          + " como j registrado!");
        return false;
      }
      // Configura o timeout para a conexo.
      if (timeout >= 0) {
        try {
          orbHandler.setTimeOut(sgaReference, timeout);
        }
        catch (Exception e) {
          Server.logSevereMessage("Erro configurando o timeout do SGA: "
            + sgaName, e);
        }
      }
      // Faz o registro
      int flag;
      try {
        if (sga == null) {
          Server.logInfoMessage("Detectado SGA " + sgaName + " como novo!");
          sga = new SGA(this, sgaName, sgaReference, staticNodesInfo);
          registeredSGAs.put(sgaName, sga);
          Server.logInfoMessage(sgaName + " com plataforma "
            + sga.getPlatformId() + " includo no gerente de SGA's.");
          flag = SGANotification.SGA_START_UP;

          if (executeNetBenchmark == true) {
            try {
              if (sga.execNetBench() == SGAInfo.NO_CAPACITY) {
                sga.setTransferRate(SGAInfo.NO_CAPACITY);
              }
              // execNetBench == ALL_CAPACITY ou execNetBench == CALC_CAPACITY
              // Se execNetBench == CALC_CAPACITY, o valor da taxa de
              // transferncia
              // desse sga ser sobreescrita quando a thread de execuo do
              // benchmark de rede for ativada.
              else {
                sga.setTransferRate(SGAInfo.ALL_CAPACITY);
              }
            }
            catch (ServerException e) {
              Server.logSevereMessage(
                "Falha na determinao se o benchmark de rede deve "
                  + "ser executado para o SGA: " + sga.getName() + ".", e);
              sga.setTransferRate(SGAInfo.NO_CAPACITY);
            }
          }
          else {
            sga.setTransferRate(SGAInfo.NO_CAPACITY);
          }
        }
        else {
          // SGA est retornando e pode ter sido modificado...
          Server.logInfoMessage("Detectado SGA " + sgaName
            + " como pr-existente!");
          sga.updateSGA(sgaReference, staticNodesInfo);
          Server.logInfoMessage(sgaName
            + " refez registro no gerente de SGA's.");
          flag = SGANotification.SGA_REINIT;
        }
      }
      catch (Exception e) {
        Server.logSevereMessage("Falha de registro de SGA: " + sgaName + ". "
          + e.getClass() + ": " + e.getMessage(), e);
        return false;
      }
      // Notifica os usurios que o sga se registrou.
      try {
        notifyAllUsersSGA(sgaName, flag);
      }
      catch (Exception e) {
        Server.logSevereMessage("Falha de notificao aos usurios sobre "
          + "registro de SGA's (" + sgaName + ").", e);
      }
    }
    finally {
      unlockSGA(sgaName);
    }

    return true;
  }

  /**
   * Envio de notificao de shutdown para todos os SGAs.
   *
   * @throws PermissionException se o usurio no tiver permisso para desativar
   *         os SGAs.
   *
   * @return coleo de SGAs que no puderam ser desativados. Se tudo correr
   *         bem, a lista estar vazia.
   */
  @Override
  public Collection<String> shutdownAllSGAs() throws PermissionException {
    checkSGAAdministrationPermission();
    Collection<String> sgas_failed = new HashSet<String>();
    Server.logInfoMessage("Notificando SGAs para shutdown...");
    Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      SGA sga = sgas.nextElement();
      if (!_shutdownSGA(sga)) {
        sgas_failed.add(sga.getName());
      }
    }
    return sgas_failed;
  }

  /**
   * Pedido de reincio para todos os SGAs.
   *
   * @throws PermissionException se o usurio no tiver permisso para reiniciar
   *         os SGAs.
   *
   * @return coleo de SGAs que no puderam ser reiniciados. Se tudo correr
   *         bem, a lista estar vazia.
   */
  @Override
  public Collection<String> restartAllSGAs() throws PermissionException {
    checkSGAAdministrationPermission();
    Collection<String> sgas_failed = new HashSet<String>();
    Server.logInfoMessage("Notificando SGAs para restart...");
    Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      SGA sga = sgas.nextElement();
      if (!_restartSGA(sga)) {
        sgas_failed.add(sga.getName());
      }
    }
    return sgas_failed;
  }

  /**
   * Envio de notificao de shutdown para um SGA especfico.
   *
   * @param sgaName nome do SGA.
   *
   * @throws PermissionException se o usurio no tiver permisso para desativar
   *         o SGA.
   *
   * @return true se o SGA tiver sido desativado com sucesso.
   */
  @Override
  public boolean shutdownSGA(String sgaName) throws PermissionException {
    checkSGAAdministrationPermission();
    Server.logInfoMessage("Notificando SGA " + sgaName + " para shutdown...");
    SGA sga = registeredSGAs.get(sgaName);
    return _shutdownSGA(sga);
  }

  /**
   * Tenta desativar o SGA especificado.
   *
   * @param sga SGA a ser desativado.
   *
   * @return true se conseguiu desativar o SGA.
   */
  private boolean _shutdownSGA(SGA sga) {
    try {
      sga.shutdownSGA();
      notifyAllUsersSGA(sga.getName(), SGANotification.SGA_SHUTDOWN);
      return true;
    }
    catch (ServerException e) {
      Server.logSevereMessage(e.getMessage(), e);
      return false;
    }
  }

  /**
   * Pedido para reiniciar o SGA especificado.
   *
   * @param sgaName nome do SGA a ser reiniciado.
   *
   * @throws PermissionException se o usurio no tiver permisso para reiniciar
   *         este SGA.
   *
   * @return true se o SGA for reiniciado com sucesso.
   */
  @Override
  public boolean restartSGA(String sgaName) throws PermissionException {
    checkSGAAdministrationPermission();
    Server.logInfoMessage("Notificando SGA " + sgaName + " para restart...");
    SGA sga = registeredSGAs.get(sgaName);
    return _restartSGA(sga);
  }

  /**
   * Tenta reiniciar o SGA especificado.
   *
   * @param sga SGA a ser reiniciado.
   *
   * @return true se conseguiu reiniciar o SGA.
   */
  private boolean _restartSGA(SGA sga) {
    try {
      sga.restartSGA();
      return true;
    }
    catch (ServerException e) {
      Server.logSevereMessage(e.getMessage(), e);
      return false;
    }
  }

  /**
   * Registra comandos recuperados em um SGA. Os demais comandos em execuo no
   * SGA (no-recuperados) so marcados como {@link EventType#LOST perdidos}.
   *
   * @param sgaName nome que identifica o sga.
   * @param cmds informaes sobre os comandos recuperados
   *
   * @return <code>true</code> se existe um SGA com o nome especificado,
   *         <code>false</code> caso contrrio
   */
  public boolean commandRetrieved(String sgaName, RetrievedInfo[] cmds) {
    SGA sga = registeredSGAs.get(sgaName);
    if (sga != null) {
      try {
        for (RetrievedInfo info : cmds) {
          Command cmd = sga.getCommand(info.cmdId);
          if (cmd != null) {
            Service.setUserId(cmd.getUserId());
          }

          Server.logInfoMessage("Informao de comando recuperado (" + sgaName
            + "): " + info.cmdId);
          sga.commandRetrieved(info.cmdId, info.cmdRef);
          Service.setUserId(null);
        }

        // Os comandos no recuperados so dados como terminados.
        Command[] sgaCmds = sga.getAllCommands();
        for (Command sgaCmd : sgaCmds) {
          Command cmd = sgaCmd;

          Service.setUserId(cmd.getUserId());

          String cmdId = cmd.getId();
          if (!cmd.isAlive()) {
            Server.logSevereMessage("Comando no encontrado [" + sgaName
              + "]: " + cmdId + "!");
            sga.lostCommand(cmdId);
            CommandEvent event =
              new CommandEvent(cmdId, EventType.LOST, cmd.getFinalizationInfo());
            notifyCommandEnd(cmd, event);
          }
        }
        return true;
      }
      finally {
        Service.setUserId(null);
      }
    }

    Server.logWarningMessage("SGA no registrado: " + sgaName);
    return false;
  }

  /**
   * Trmino do servio de gerncia de SGAs. Destri o servant que implementa a
   * interface de comunicao com os SGAs, e desativa o servidor CORBA
   * correspondente.
   *
   * @throws ServerException caso no consiga desativar o servant.
   */
  @Override
  public void shutdownService() throws ServerException {
    exitAcquisitionDataThread = true;
    exitAuditThread = true;
    if (acquisitionDataThread != null) {
      acquisitionDataThread.interrupt();
    }
    if (auditThread != null) {
      auditThread.interrupt();
    }
    if (acquisitionNetDataThread != null) {
      acquisitionNetDataThread.interrupt();
    }

    final Enumeration<SGA> sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      SGA sga = sgas.nextElement();
      sga.stop();
    }
    try {
      if (orbHandler != null) {
        orbHandler.shutdown();
      }
      sgaHandler = null;
      orbHandler = null;
    }
    catch (ORBException e) {
      throw new ServerException("Impossvel desativar servidor CORBA", e);
    }
  }

  /**
   * Solicitao de de-registro de um SGA.
   *
   * @param sgaReference referncia para o sga
   * @param sgaName nome do SGA
   */
  public void unregisterSGA(SGAServer sgaReference, String sgaName) {
    Server.logInfoMessage("Pedido de de-registro de SGA: " + sgaName);
    SGA sga = registeredSGAs.get(sgaName);
    if ((sga == null) || (sga.getRemoteReference() == null)
      || !sgaReference.equals(sga.getRemoteReference())) {
      Server.logWarningMessage("De-registro de SGA no registrado: " + sgaName);
      return;
    }
    registeredSGAs.remove(sgaName);
  }

  /**
   * Obtm o path para o arquivo de persistncia de grupos de um usurio.
   *
   * @return o path
   *
   * @throws ServerException caso no seja possvel obter as informaes do
   *         usurio.
   */
  private String getUserSGAGroupFilePath() throws ServerException {
    final User user = Service.getUser();
    try {
      if (user == null) {
        throw new ServerException(
          "getUserSGAGroupFilePath: Usurio no cadastrado");
      }
      final String fileName = "groups.dat";
      final String sep = File.separator;
      final Server server = Server.getInstance();
      final String persistDirPath = server.getPersistencyRootDirectoryName();
      final String dirPath =
        persistDirPath + sep + SGAService.SGA_GROUPS_SUBDIR_NAME + sep
        + user.getLogin();
      Server.checkDirectory(dirPath);
      final String filePath = dirPath + sep + fileName;
      return filePath;
    }
    catch (Exception ue) {
      throw new ServerException("getUserDirPath: "
        + "Falha obtendo informaes do usurio (" + ue.getMessage() + ")");
    }
  }

  /**
   * Verifica se o usurio pode executar comandos neste SGA.
   *
   * @param sgaName nome do SGA.
   *
   * @throws PermissionException se houver falha de permisso para acesso.
   */
  private void checkExecutionPermission(final String sgaName)
    throws PermissionException {
    if (!hasExecutionPermission(sgaName)) {
      throw new PermissionException();
    }
  }

  /**
   * Indica se o usurio possui permisso de executar comandos no servidor dado.
   *
   * @param sgaName Nome do servidor
   *
   * @return true se o usurio tiver permisso para executar comandos no
   *         servidor dado
   */
  private boolean hasExecutionPermission(String sgaName) {
    final User user = Service.getUser();
    /*
     * Se este mtodo no for chamado pelo cliente, no h um usurio associado
     * a chamada. Por isso, considera que o SGA no tem permisso para executar
     * comandos.
     */
    if (user == null) {
      return false;
    }
    if (user.isAdmin()) {
      return true;
    }
    Permission p = null;
    try {
      p =
        user.getMatchAttributesPermission(ServerExecutionPermission.class,
          sgaName);
    }
    catch (Exception e) {
      String errorMsg =
        String
        .format(
          "Ocorreu um erro ao buscar a permisso para executar comandos no servidor %s para o usurio %s.",
          sgaName, user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    if (p == null) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso de administrao sobre o servio de SGA do servidor corrente.
   *
   * @throws PermissionException se houver falha de permisso.
   */
  private void checkSGAAdministrationPermission() throws PermissionException {
    final User user = Service.getUser();
    if (user.isAdmin()) {
      return;
    }
    final Server server = Server.getInstance();
    final String serverName = AdminPermission.LOCAL + server.getSystemName();
    Permission p = null;
    try {
      p =
        user.getMatchAttributesPermission(SGAAdminPermission.class, serverName);
    }
    catch (Exception e) {
      String errorMsg =
        String
        .format(
          "Ocorreu um erro ao buscar a permisso para gerenciar o servidor de execuo %s para o usurio %s.",
          serverName, user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    if (p == null) {
      throw new PermissionException();
    }
  }

  /**
   * Criao da thread que faz a gravao dos dados dos SGAs para auditagem.
   *
   * @see #initService()
   */
  private void createAuditThread() {
    final boolean activeDump = getBooleanProperty("activeDump");
    if (!activeDump) {
      Server.logInfoMessage("AUDITAGEM DOS SGAs DESATIVADA! (activeDump)");
      auditThread = null;
      return;
    }

    Server.logInfoMessage("Gravao de dados dos SGAs ativa.");
    Server.logFineMessage("Preparando a criao de thread de auditagem...");

    final int ONE_SECOND = 1;
    final int ONE_MINUTE = ONE_SECOND * 60;
    final int ONE_HOUR = ONE_MINUTE * 60;

    /*
     * Busca das propriedades que identificam os tempos para gravao
     * (amostragem) dos dados dos SGAs em disco
     */
    int dump = getIntProperty("dumpIntervalSeconds");

    /* Checagem de valores razoveis */
    if (dump < ONE_SECOND) {
      dump = ONE_SECOND;
    }
    if (dump > ONE_HOUR) {
      dump = ONE_HOUR;
    }
    final long sleepTimeMs = dump * MILLIS;
    Server.logInfoMessage("Intervalo de gravao dos SGAs: " + dump
      + " segundos.");
    auditThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitAuditThread) {
            try {
              // Server.logInfoMessage("Thread iniciada: createAuditThread");
              logAllAuditEvents();
              // Server.logInfoMessage("Thread terminada: createAuditThread");
              Thread.sleep(sleepTimeMs);
            }
            catch (InterruptedException ie) {
              Server
              .logWarningMessage("Gravao de dados do SGA interrompido!");
            }
          }
          Server
          .logWarningMessage("Finalizando thread de gravao de dados...");
        }
        catch (Throwable t) {
          final String msg = (new Date()) + " - Erro na auditagem de SGAs:";
          System.err.println(msg);
          t.printStackTrace(System.err);
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitAuditThread = false;
    auditThread.setName(this.getClass().getSimpleName() + "::" + "AuditThread");
    auditThread.start();
  }

  /**
   * Criao da thread que verifica se os SGAs esto ativos.
   *
   * @see #initService()
   */
  private void createDataAcquisitionThread() {
    /*
     * Busca da propriedade que determina o intervalo de varredura do watchdog
     * dos SGAs
     */
    synchronized (SGAService.class) {
      sgaWatchdogInterval = getIntProperty("sgaWatchdogIntervalSeconds");
      sgaUpdateInterval = getIntProperty("sgaUpdateIntervalSeconds");
    }
    final long updateSleepTime = (long) sgaWatchdogInterval * MILLIS;
    acquisitionDataThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitAcquisitionDataThread) {
            updateAllSGAs();
            try {
              Thread.sleep(updateSleepTime);
            }
            catch (InterruptedException ie) {
              Server
              .logWarningMessage("Aquisio de dados do SGA interrompido!");
            }
          }
          Server
          .logWarningMessage("Finalizando thread de aquisio de dados...");
        }
        catch (Throwable t) {
          String msg = (new Date()) + " - Erro na aquisio de dados de SGAs:";
          System.err.println(msg);
          t.printStackTrace(System.err);
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitAcquisitionDataThread = false;
    acquisitionDataThread.setName(this.getClass().getSimpleName() + "::"
      + "AcquisitionDataThread");
    acquisitionDataThread.start();
  }

  /**
   * Criao da thread que faz a gravao dos dados de execuo dos algoritmos
   * para auditagem.
   *
   * @see #initService()
   */
  private void createExecutionHistoricThread() {
    final boolean createHistoric =
      getBooleanProperty("sgaLogAlgorithmExecutionHistoric");
    if (!createHistoric) {
      Server.logInfoMessage("AUDITAGEM DE EXECUO DOS ALGORITMOS DESATIVADA! "
        + "(sgaLogAlgorithmExecutionHistory)");
      historicThread = null;
      return;
    }

    Server.logInfoMessage("Gravao de histrico dos algoritmos ativa.");
    Server
    .logFineMessage("Preparando a criao de thread de auditagem de execuo "
      + "dos algoritmos...");

    /*
     * Busca da propriedade que determina o intervalo para escrita de novos
     * dados de execuo dos algoritmos no log.
     */
    synchronized (SGAService.class) {
      sgaHistoricUpdateIntervalSeconds =
        getIntProperty("sgaHistoricUpdateIntervalSeconds");
    }

    final long sleepTimeMs = sgaHistoricUpdateIntervalSeconds * MILLIS;
    Server.logInfoMessage("Intervalo de gravao do histrico dos algoritmos: "
      + sgaHistoricUpdateIntervalSeconds + " segundos.");
    historicThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitHistoricThread) {
            try {
              logAllAlgorithmHistoricEvents();
              Thread.sleep(sleepTimeMs);
            }
            catch (InterruptedException ie) {
              Server
              .logWarningMessage("Gravao de dados de execuo dos algoritmos "
                + "interrompida!");
            }
          }
          Server
          .logWarningMessage("Finalizando thread de gravao de dados de "
            + "histrico de execuo...");
        }
        catch (Throwable t) {
          final String msg =
            (new Date()) + " - Erro na auditagem de execuo de algoritmos:";
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitHistoricThread = false;
    historicThread.setName(this.getClass().getSimpleName() + "::"
      + "HistoricThread");
    historicThread.start();
  }

  /**
   * Retorna sempre verdadeiro.
   *
   * @see Service#has2Update(Object,Object)
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * Gera notificaes externas de que um comando terminou (observadores, mail e
   * notificao).
   *
   * @param command o comando terminado
   * @param event evento associado ao trmindo da execuo de um comando
   */
  private void notifyCommandEnd(final Command command, CommandEvent event) {
    /*
     * incrementamos as estatsticas
     */
    collectExeResultsStats(command, event);
    /*
     * enviamos as notificaes
     */
    notifyCommandEndToProject(command, event);
    mailCommandEndToProject(command, event);
  }

  /**
   * Coleta as estatsticas de trmino das execues.
   *
   * @param command o comando terminado
   * @param event evento associado ao trmindo da execuo de um comando
   */
  private void collectExeResultsStats(Command command, CommandEvent event) {
    if (command == null) {
      return;
    }
    CommandFinalizationInfo finalizationInfo = event.getFinalizationInfo();
    CommandFinalizationType finalizationType =
      finalizationInfo.getFinalizationType();
    try {
      if (command.getCommandInfo().getConfigurator().getConfiguratorType() == ConfiguratorType.FLOW) {
        /*
         * estamos coletando informaes de um fluxo, precisamos obter de cada
         * um dos comandos que foram executados
         */
        incrCounter(flowResultCounter, finalizationType);
        // TODO coletar estatsticas por ns
      }
      else {
        incrCounter(exeResultCounter, finalizationType);
      }
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao obter o configurador do comando "
        + command.getCommandInfo().getId());
    }
  }

  /**
   * Coleta as estatsticas de execuo por ns.
   *
   * @param cmd comando
   */
  private void collectSGAStats(CommandInfo cmd) {
    incrCounter(sgasCounter, cmd.getSGAName());
  }

  /**
   * Obtm as informaes dos grupos definidos por um usurio persistidas em
   * arquivo correspondente.
   *
   * @param groupsFilePath path para o arquivo
   *
   * @return um array com as informaes dos grupos
   *
   * @throws ServerException se houver falha no servidor.
   */
  private ServerGroupInfo[] readGroupsInfo(String groupsFilePath)
    throws ServerException {
    File groupsFile = new File(groupsFilePath);
    if (!groupsFile.exists() || !groupsFile.isFile()) {
      return new ServerGroupInfo[0];
    }
    Properties gprops = new Properties();
    BufferedInputStream inStream = null;
    try {
      inStream = new BufferedInputStream(new FileInputStream(groupsFile));
      gprops.load(inStream);
    }
    catch (Exception e) {
      throw new ServerException("readGroupsInfo: Falha na leitura de "
        + groupsFilePath + "( " + e.getMessage() + ")");
    }
    finally {
      try {
        if (inStream != null) {
          inStream.close();
        }
      }
      catch (IOException e) {
        throw new ServerException("Erro fechando stream de leitura", e);
      }
    }
    int numProps = gprops.size();
    ServerGroupInfo[] info = new ServerGroupInfo[numProps];
    Enumeration<Object> groups = gprops.keys();
    for (int i = 0; i < numProps; i++) {
      String name = (String) groups.nextElement();
      StringTokenizer sList =
        new StringTokenizer((String) gprops.get(name), ",");
      int size = sList.countTokens();
      String[] servers = new String[size];
      for (int j = 0; j < size; j++) {
        servers[j] = sList.nextToken();
      }
      info[i] = new ServerGroupInfo(name, servers);
    }
    return info;
  }

  /**
   * Aquisio dos dados dos SGAs
   */
  private void updateAllSGAs() {
    Enumeration<SGA> sgas;
    sgas = registeredSGAs.elements();
    while (sgas.hasMoreElements()) {
      SGA sga = sgas.nextElement();
      if (sga == null || !sga.isAlive()) {
        continue;
      }

      if (sga.getWatchDog()) {
        /*
         * O SGA est vivo e atualizando seus dados. Reseta o watchdog para
         * saber se o SGA atualizar seus dados novamente.
         */
        sga.resetWatchDog();
        notifyAllUsersSGA(sga.getName(), SGANotification.SGA_UPDATE);
      }
      else {
        sga.setAlive(false);
        Server.logWarningMessage("SGA " + sga.getName() + " inacessvel! - ");

        notifyAllUsersSGA(sga.getName(), SGANotification.SGA_SHUTDOWN);
      }
    }
  }

  /**
   * Efetua o log das informaes dinmicas recebidas do SGA.
   *
   * @param info Informaes recebidas do SGA
   * @param prefix String que ser coloca antes das informaes
   */
  void logSGAInfo(SGAInfo[] info, String prefix) {
    if (doSGALog) {
      /*
       * Restringe no mximo de ns para o log, alm do principal. Valor < 0
       * para log de todas as mquinas.
       */
      int max;
      if (numberHostsLog < 0) {
        max = info.length;
      }
      else {
        max =
          (info.length > (numberHostsLog + 1)) ? (numberHostsLog + 1)
            : info.length;
      }
      for (int i = 0; i < max; i++) {
        logSGAInfo(info[i], prefix);
      }
    }
  }

  /**
   * Efetua o log das informaes dinmicas recebidas do SGA.
   *
   * @param msg Informaes recebidas do SGA
   */
  void logSGAMessage(String msg) {
    if (doSGALog) {
      Server.logInfoMessage(msg);
    }
  }

  /**
   * Efetua o log das informaes dinmicas recebidas do SGA.
   *
   * @param info Informaes recebidas do SGA
   * @param prefix String que ser coloca antes das informaes
   */
  void logSGAInfo(SGAInfo info, String prefix) {
    if (doSGALog) {
      String format =
        "{0} {1}: RAM {2,number,#.##}, " + "Swap {3,number,#.##}, "
          + "Load {4,number,#.##}/{5,number,#.##}/{6,number,#.##}";
      MessageFormat formatter = new MessageFormat(format);
      Object[] args = new Object[7];
      args[0] = (prefix != null) ? prefix : "";
      StringBuffer result = new StringBuffer();
      args[1] = info.getHostName();
      args[2] = new Double(info.getRAMFreeMemory());
      args[3] = new Double(info.getSwapFreeMemory());
      args[4] = new Double(info.getCPULoad1());
      args[5] = new Double(info.getCPULoad5());
      args[6] = new Double(info.getCPULoad15());
      formatter.format(args, result, null);
      Server.logInfoMessage(result.toString());
    }
  }

  /**
   * Salva as informaes dos grupos definidos por um usurio no arquivo
   * correspondente.
   *
   * @param groupsFilePath path para o arquivo
   * @param info array com as informaes dos grupos
   *
   * @throws ServerException caso ocorra falha na gravao das informaes.
   */
  private void writeGroupsInfo(String groupsFilePath, ServerGroupInfo[] info)
    throws ServerException {
    Properties gprops = new Properties();
    for (ServerGroupInfo group : info) {
      String[] servers = group.getServers();
      if (servers.length < 1) {
        break;
      }
      StringBuilder sList = new StringBuilder(servers[0]);
      for (int j = 1; j < servers.length; j++) {
        sList.append(',');
        sList.append(servers[j]);
      }
      gprops.put(group.getName(), sList.toString());
    }
    BufferedOutputStream grpFile = null;
    try {
      grpFile = new BufferedOutputStream(new FileOutputStream(groupsFilePath));
      gprops.store(grpFile, null);
    }
    catch (Exception e) {
      throw new ServerException("writeGroupsInfo: Falha na gravao de "
        + groupsFilePath + "( " + e.getMessage() + ")");
    }
    finally {
      try {
        if (grpFile != null) {
          grpFile.close();
        }
      }
      catch (IOException e) {
        Server.logSevereMessage("Erro fechando stream", e);
      }
    }
  }

  // Metodos relacionados ao CSFSService (foram deixados "friendly" de proposito
  // (em vez de public/private) para facilitar o uso na classe Command.

  /**
   * <p>
   * Quando um comando deve ser executado, ele solicita ao SGAService que
   * prepare um ambiente de execuo no SGA. Essa preparao consiste em
   * garantir que todos os binrios, arquivos de entrada e diretrios dos quais
   * o comando dependa estejam disponveis. Isso inclui os diretrios-pai dos
   * arquivos de sada, evitando que um comando falhe por no conseguir acessar
   * um diretrio onde ele pretende gerar uma sada.
   * </p>
   * <p>
   * Nessa operao, o SGAService interage com o prprio SGA e o CSFSService,
   * visando verificar o estado do sistema de arquivos do SGA. Se a rea de
   * projetos e os diretrios de binrios j estiverem acessveis pelo SGA
   * (provavelmente por conta da existncia do NFS), a execuo do comando 
   * liberada sem a necessidade de recorrer ao CSFSService. Se a rea de
   * projetos no estiver acessvel, todos os arquivos sao copiados pro sistemas
   * de arquivos de SGA utilizando o CSFSService, que utiliza a API do CSFS. Os
   * binrios so copiados para o diretrio raiz do CSFS e l permanecem mesmo
   * aps terminada a execuo, para possibilitar o reuso. J os arquivos de
   * entrada sao copiados para um "sandbox" temporrio, isto , um diretrio que
   * ser removido ao trmino da execuo.
   * </p>
   * <p>
   * Caso seja necessrio utilizar o CSFSService, a linha do comando a ser
   * executado  modificada para refletir as posies relativas dos arquivos e
   * binrios dentro da raiz do CSFS.
   * </p>
   * <p>
   * Um <i>snapshot</i> do sandbox  obtido imediatamente antes da execuo. Ao
   * trmino, esse <i>snapshot</i>  comparado com o novo estado. Os arquivos
   * que tenham sido criados/alterados so ento copiados de para a rea de
   * projetos (ver mtodo {@link #clearExecutionEnvironment(Command)}).
   * </p>
   *
   * @param command dados do comando.
   *
   * @return <code>false</code> se a execuo do comando tiver de ser
   *         interrompida, <code>true</code> se tudo correr normalmente.
   */
  boolean setupExecutionEnvironment(Command command) {
    if (isCSFSExecution(command)) {
      return setupCSFSExecutionEnvironment(command);
    }
    else {
      return checkExecutionEnvironment(command);
    }
  }

  /**
   * Verifica se o ambiente de execuo do SGA est configurado corretamente,
   * testando se o SGA tem acesso aos binrios do algoritmo a ser executado e ao
   * projeto selecionada, por exemplo.
   *
   * @param command o comando a ser executado.
   * @return verdadeiro se o ambiente est configurador corretamente ou falso,
   *         caso contrrio.
   */
  private boolean checkExecutionEnvironment(Command command) {
    SGAInfo host = getExecutionHost(command);

    AlgorithmConfigurator algorithmConfigurator =
      getAlgorithmConfigurator(command);
    if (algorithmConfigurator == null) {
      return false;
    }

    Set<String[]> binaryDirs =
      algorithmConfigurator.getBinaryDirectoriesAsArray(host.getPlatformId());

    SGA sga = command.getSGA();
    String[] projectPath =
      combinePaths(command.getNode().getProjectRootDirectory(), command
        .getProjectPath());
    char fileSeparator = host.getFileSeparator();

    // Projeto visto pelo SGA
    String sgaProjectFullPath =
      FileUtils.joinPath(true, fileSeparator, projectPath);

    try {
      /* Verifica se o SGA tem acesso aos arquivos em sua rea */
      if (sga.isDir(sgaProjectFullPath)) {
        // S  necessrio testar o acesso a um dos diretrios de binrios,
        // pois esto todos na mesma rvore.
        String[] binaryPath =
          combinePaths(command.getNode().getAlgorithmRootDirectory(),
            binaryDirs.iterator().next());
        // Diretrio dos binrios visto pelo SGA
        String sgaBinaryPath =
          FileUtils.joinPath(true, fileSeparator, binaryPath);
        if (sga.isDir(sgaBinaryPath)) {
          return true;
        }
        else {
          // XXX - No quer mesmo notificar os observadores e enviar
          // email? Talvez esteja redundante!
          StringBuilder msg =
            new StringBuilder("O SGA no tem acesso ao diretrio de binrios '");
          msg.append(sgaBinaryPath);
          msg.append("', \n  embora tenha encontrado o diretrio de projeto '");
          msg.append(sgaProjectFullPath);
          msg.append("'.\n  Por favor, verifique a configurao do ambiente.");
          Server.logSevereMessage(msg.toString());
          return false;
        }
      }
      else {
        StringBuilder msg =
          new StringBuilder("O SGA no tem acesso ao diretrio do projeto '");
        msg.append(sgaProjectFullPath);
        msg.append("'.\n  Por favor, verifique a configurao do ambiente.");
        Server.logSevereMessage(msg.toString());
        return false;
      }
    }
    catch (ServerException exc) {
      Server.logSevereMessage("Ocorreu um erro na tentativa de verificar o "
        + "ambiente de execuo no SGA [" + sga.getName() + "] - id: "
        + command.getId() + ".\n", exc);
      return false;
    }
  }

  /**
   * Indica se o comando deve ser executado utilizando o CSFS.
   *
   * @param command O comando.
   * @return verdadeiro se o comando deve ser executado via CSFS ou falso, caso
   *         contrrio.
   */
  private boolean isCSFSExecution(Command command) {
    SGA sga = command.getSGA();
    String csfsHost = sga.getCSFSHost();
    String[] csfsRootDir = sga.getCSFSRootDir();
    if ((!command.existsProject()) || (csfsHost == null)
      || (csfsRootDir == null)) {
      return false;
    }
    return true;
  }

  /**
   * Prepara o ambiente de execuo para SGAs no acessam a rea de projetos e
   * algoritmos. Para isso,  necessrio utilizar o servio CSFS para criar um
   * sandbox contendo os arquivos de projeto necessrios e tambm copiar os
   * binrios para que fiquem devem disponveis dentro da raiz do CSFS. O
   * servio do CSFS precisa estar disponvel e habilitado para essa opo
   * funcionar.
   *
   * @param command dados do comando.
   *
   * @return <code>false</code> se a execuo do comando tiver de ser
   *         interrompida, <code>true</code> se tudo correr normalmente.
   */
  private boolean setupCSFSExecutionEnvironment(Command command) {
    SGA sga = command.getSGA();

    CSFSService csfsService = CSFSService.getInstance();
    if ((csfsService == null) || !csfsService.isActive()) {
      Server.logSevereMessage("CSFS inacessvel ou inibido. SGA "
        + sga.getName() + " precisa deste servio para execuo.");
      return false;
    }

    AlgorithmConfigurator algorithmConfigurator =
      getAlgorithmConfigurator(command);
    if (algorithmConfigurator == null) {
      return false;
    }
    SGAInfo host = getExecutionHost(command);

    Set<String[]> binaryDirs =
      algorithmConfigurator.getBinaryDirectoriesAsArray(host.getPlatformId());

    try {

      setCSFSExecutionPaths(command, sga.getCSFSRootDir());
      setupSandbox(command);
      saveSandboxSnapshot(command);
      setupBinaryDir(binaryDirs, sga.getCSFSHost(), sga.getCSFSPort());
      return true;
    }
    catch (ServerException exc) {
      // TODO: No teria que limpar qualquer coisa que tenha sido copiada?
      Server.logSevereMessage(
        "Ocorreu um erro na tentativa de verificar/criar um "
          + "ambiente de execuo CSFS no SGA [" + sga.getName() + "] - id: "
          + command.getId() + ".\n", exc);
      return false;
    }
  }

  /**
   * Monta o caminho para a sandbox CSFS do comando.
   *
   * @param command O comando.
   * @return O caminho para a sandbox do comando.
   */
  private String[] getCSFSSandboxPath(Command command) {
    String[] sandbox =
    { command.getUserId().toString(), command.getProjectName(),
      FileUtils.fixDirectoryName(command.getId()) };
    return combinePaths(CSFS_SANDBOX_ROOT_DIR, sandbox);
  }

  /**
   * Copia os scripts do comando para o ambiente de execuo.
   *
   * @param command O comando
   * @throws OperationFailureException Se no for possvel obter os scripts do
   *         comando.
   * @throws ServerException Se houver algum problema no upload dos arquivos via
   *         CSFS.
   */
  public void uploadScripts(Command command) throws OperationFailureException,
  ServerException {
    if (isCSFSExecution(command)) {
      CSFSService csfsService = CSFSService.getInstance();
      SGA sga = command.getSGA();
      String[] csfsProjectPath =
        combinePaths(csfsService.getCSFSProjectsRootPath(), command
          .getProjectPath());
      String[] sandboxPath = getCSFSSandboxPath(command);
      CommandPersistenceService persistence =
        CommandPersistenceService.getInstance();
      String[][] commandScriptPaths =
        persistence.getCommandScripts(command.getId(), command.getProjectId());
      /* Copia os scripts do comando para o sandbox */
      for (String[] scriptPath : commandScriptPaths) {
        String[] localScriptFile = combinePaths(csfsProjectPath, scriptPath);
        String[] remoteScriptFile = combinePaths(sandboxPath, scriptPath);
        csfsService.copyTo(sga.getCSFSHost(), sga.getCSFSPort(),
          localScriptFile, remoteScriptFile);
      }
    }
  }

  /**
   * Utilitrio para combinar mltiplos paths em formato de array em um nico
   * array.
   *
   * @param arrays Lista de arrays a serem combinados.
   * @return um nico array combinado.
   */
  static String[] combinePaths(String[]... arrays) {
    // Determina o tamanho do novo array
    int count = 0;
    for (String[] array : arrays) {
      count += array.length;
    }
    String[] mergedArray = new String[count];

    int start = 0;
    for (String[] array : arrays) {
      System.arraycopy(array, 0, mergedArray, start, array.length);
      start += array.length;
    }
    return mergedArray;
  }

  /**
   * Salva o estado da sandbox do CSFS.
   *
   * @param command O comando.
   * @throws ServerException Se houver algum problema na comunicao com o
   *         servio de CSFS.
   */
  private void saveSandboxSnapshot(Command command) throws ServerException {
    SGA sga = command.getSGA();
    CSFSService csfsService = CSFSService.getInstance();
    String sandboxPath[] = getCSFSSandboxPath(command);
    /*
     * Recupera um snapshot do sandbox no SGA e armazena como estado inicial no
     * comando, antes da execuo do algoritmo.
     */
    Map<String[], Long> sandboxInitState =
      csfsService.getTimestamps(sga.getCSFSHost(), sga.getCSFSPort(),
        sandboxPath);
    command.setSandboxSnapshot(sandboxInitState);
  }

  /**
   * Prepara os diretrios de binrios dos algoritmos do comando no ambiente de
   * execuo do CSFS.
   *
   * @param binaryDirs Os diretrio de binrios dos algoritmos.
   * @param csfsHost Host do CSFS.
   * @param csfsPort Porta do CSFS.
   * @throws ServerException Se houver algum problema na comunicao com o
   *         servio de CSFS.
   */
  private void setupBinaryDir(Set<String[]> binaryDirs, String csfsHost,
    int csfsPort) throws ServerException {

    CSFSService csfsService = CSFSService.getInstance();
    // Cria o diretrio de algoritmos.
    csfsService.createDirectory(csfsHost, csfsPort, CSFS_ALGORITHMS_ROOT_DIR);

    /*
     * Verifica se todos os diretrios de binrios necessrios esto presentes
     * ou sofreram alguma modificao desde a ltima cpia para a sandbox (caso
     * contrrio, copia).
     */
    String[] csfsAlgorithmPath = csfsService.getCSFSAlgorithmsRootPath();

    for (String[] binaryDir : binaryDirs) {
      String[] localBinDir = combinePaths(csfsAlgorithmPath, binaryDir);
      String[] remoteBinDir = combinePaths(CSFS_ALGORITHMS_ROOT_DIR, binaryDir);
      if (!csfsService.checkExistence(csfsHost, csfsPort, binaryDir)
        || binaryDirUpdated(localBinDir, binaryDir, csfsHost, csfsPort)) {
        csfsService.copyTo(csfsHost, csfsPort, localBinDir, remoteBinDir);
      }
    }
  }

  /**
   * Prepara a sandbox do comando no ambiente de execuo do CSFS.
   *
   * @param command O comando.
   * @throws ServerException Se houver algum problema na comunicao com o
   *         servio de CSFS.
   */
  private void setupSandbox(Command command) throws ServerException {

    SGA sga = command.getSGA();
    String csfsHost = sga.getCSFSHost();
    int csfsPort = sga.getCSFSPort();
    String[] sandboxPath = getCSFSSandboxPath(command);

    CSFSService csfsService = CSFSService.getInstance();
    csfsService.createDirectory(csfsHost, csfsPort, sandboxPath);

    String[] csfsFullProjectPath =
      combinePaths(csfsService.getCSFSProjectsRootPath(), command
        .getProjectPath());

    AlgorithmConfigurator algorithmConfigurator =
      getAlgorithmConfigurator(command);

    /* Copia os arquivos de entrada para o sandbox */
    Collection<FileURLValue> inputFiles = algorithmConfigurator.getInputFiles();
    for (FileURLValue file : inputFiles) {
      String[] inputFilePath = file.getPathAsArray();
      String[] localInputFile =
        combinePaths(csfsFullProjectPath, inputFilePath);
      String[] remoteInputFile = combinePaths(sandboxPath, inputFilePath);
      csfsService.copyTo(csfsHost, csfsPort, localInputFile, remoteInputFile);
    }

    /*
     * Copia os diretrios necessrios (se for o caso) para o sandbox. Uma lista
     * de diretorios que podem ser usados pelo algoritmo (temporariamente podem
     * diretorios de entrada e saida)
     */
    Collection<FileURLValue> inputDirs =
      algorithmConfigurator.getInputDirectories();
    for (FileURLValue dir : inputDirs) {
      String[] dirPath = dir.getPathAsArray();
      String[] localDirectory = combinePaths(csfsFullProjectPath, dirPath);
      String[] remoteDirectory = combinePaths(sandboxPath, dirPath);
      csfsService.copyTo(csfsHost, csfsPort, localDirectory, remoteDirectory);
    }

    /*
     * Cria, se necessrio, uma estrutura de diretrios de sada dentro da
     * sandbox.
     */
    Collection<FileURLValue> outputDirs =
      algorithmConfigurator.getOutputDirectories();
    for (FileURLValue file : outputDirs) {
      String[] outputDirPath = file.getPathAsArray();
      String[] remoteOutputPath = combinePaths(sandboxPath, outputDirPath);
      csfsService.createDirectory(csfsHost, csfsPort, remoteOutputPath);
    }

    /*
     * Cria, se necessrio, uma estrutura de diretrios dentro da sandbox para
     * os arquivos de sada.
     */
    Collection<FileURLValue> outputFiles = algorithmConfigurator.getOutputFiles();
    for (FileURLValue file : outputFiles) {
      String[] outputFilePath = file.getPathAsArray();
      if (outputFilePath.length > 1) {
        /*
         * Faz um array s com o caminho at o diretrio pai do arquivo
         * (removendo o nome do arquivo).
         */
        String[] outputDirPath = new String[outputFilePath.length - 1];
        System.arraycopy(outputFilePath, 0, outputDirPath, 0,
          outputFilePath.length - 1);
        String[] remoteOutputDirPath = combinePaths(sandboxPath, outputDirPath);
        csfsService.createDirectory(csfsHost, csfsPort, remoteOutputDirPath);
      }
    }

    /* Copia os arquivos de sada padro a sandbox */
    Set<FileURLValue> stdouts = algorithmConfigurator.getStandardOutputFiles();
    for (FileURLValue file : stdouts) {
      String[] stdoutPath = file.getPathAsArray();
      String[] localStdoutFile = combinePaths(csfsFullProjectPath, stdoutPath);
      String[] remoteStdoutFile = combinePaths(sandboxPath, stdoutPath);
      csfsService.copyTo(csfsHost, csfsPort, localStdoutFile, remoteStdoutFile);
    }
  }

  /**
   * Configura os caminhos para execuo no ambiente CSFS.
   *
   * @param command O comando.
   * @param remoteRootPath Caminho para a raiz do servidor remoto.
   */
  private void setCSFSExecutionPaths(Command command, String[] remoteRootPath) {

    String[] remoteAlgorithmRootPath =
      combinePaths(remoteRootPath, CSFS_ALGORITHMS_ROOT_DIR);
    String[] sandboxPath = getCSFSSandboxPath(command);

    command.setProjectRootPath(remoteRootPath);
    command.setAlgorithmRootPath(remoteAlgorithmRootPath);
    command.setSandbox(sandboxPath);
    command.setProjectPath(sandboxPath);
  }

  /**
   * Obtm informaes sobre o sga que executar o comando.
   *
   * @param command O comando.
   * @return Informaes sobre o sga.
   */
  private SGAInfo getExecutionHost(Command command) {
    String hostName = command.getNode().getHostName();
    SGA sga = command.getSGA();
    if (hostName == null) {
      return sga.getInfo(0);
    }
    else {
      return sga.getInfo(hostName);
    }
  }

  /**
   * Obtm o configurador a partir do comando. Retorna nulo se no for possvel
   * obter o configurador.
   *
   * @param command O comando.
   * @return O configurador.
   */
  private AlgorithmConfigurator getAlgorithmConfigurator(Command command) {
    AlgorithmConfigurator algorithmConfigurator = null;
    try {
      CommandInfo commandInfo = command.getCommandInfo();
      if (commandInfo != null) {
        algorithmConfigurator = commandInfo.getConfigurator();
      }
    }
    catch (RemoteException e) {
      Server.logSevereMessage(
        "Ocorreu um erro na tentativa de obter o configurador do comando "
          + command.getId() + ".\n", e);
      return null;
    }
    return algorithmConfigurator;
  }

  /**
   * Notifica que o comando no foi iniciado.
   *
   * @param command o comando que no foi iniciado.
   */
  void handleCommandInitFailure(Command command) {
    String cmdId = command.getId();
    CommandEvent event =
      new CommandEvent(cmdId, EventType.INIT_FAILURE, command
        .getFinalizationInfo());
    notifyCommandEnd(command, event);
  }

  /**
   * <p>
   * Indica se o diretrio de binrio indicado foi modificado desde a ltima
   * cpia para a sandbox. Se for o caso, remove o diretrio desatualizado da
   * sandbox.
   * </p>
   * <p>
   * Este mtodo reflete uma premissa atual no sistema, segundo a qual os
   * usurios podem modificar os binrios de um algoritmo sem criar uma nova
   * verso. Se esta premissa for modificada, isto , se no for mais permitido
   * a edio de verses mas to somente a incluso de novas verses
   * (analogamente ao que  feito nos controladores de verso tradicionais),
   * este mtodo deixa de fazer sentido.
   * </p>
   *
   * @param localBinaryDir caminho para um diretrio local contendo binrios.
   * @param remoteBinaryDir caminho para um diretrio remoto contendo binrios.
   * @param csfsHost servidor onde est a sandbox.
   * @param csfsPort porta para acessar o CSFS.
   *
   * @return true se o diretrio de binrio especificado tiver sido modificado
   *         desde a ltima cpia para a sandbox.
   *
   * @throws ServerException Se houver algum problema na comunicao com o
   *         servio de CSFS.
   */
  private boolean binaryDirUpdated(String[] localBinaryDir,
    String[] remoteBinaryDir, String csfsHost, int csfsPort)
      throws ServerException {
    CSFSService csfsService = CSFSService.getInstance();
    if (csfsService == null) {
      throw new ServerException("CSFSService no encontrado");
    }
    Map<String[], Long> localBinaryDirs =
      csfsService.getTimestamps(localBinaryDir);
    Map<String[], Long> remoteBinaryDirs =
      csfsService.getTimestamps(csfsHost, csfsPort, remoteBinaryDir);
    if (localBinaryDirs.size() != remoteBinaryDirs.size()) {
      csfsService.remove(csfsHost, csfsPort, remoteBinaryDir);
      return true;
    }
    Set<Entry<String[], Long>> remoteTimestamps = remoteBinaryDirs.entrySet();
    for (Entry<String[], Long> timestamps : remoteTimestamps) {
      String[] file = timestamps.getKey();
      Long remoteTime = timestamps.getValue();
      Long localTime = localBinaryDirs.get(file);
      if (localTime == null || remoteTime < localTime.longValue()) {
        csfsService.remove(csfsHost, csfsPort, remoteBinaryDir);
        return true;
      }
    }
    return false;
  }

  /**
   * <p>
   * Ao trmino de um comando, pode ser necessria a sincronizao do sandbox
   * com a rea de projetos: se o comando tiver sido executado sem o uso do NFS,
   * os arquivos de saida foram gerados em uma rea local no SGA (sandbox) e
   * precisam ser transferidos para a rea de projetos.
   * </p>
   * Uma falha nessa operao caracteriza uma falha do comando, pois o resultado
   * no pode ser armazenado na rea de projetos.
   *
   * @param command dados do comando.
   *
   * @return <code>false</code> se tiver ocorrido algum problema durante a
   *         transferncia para a rea de projetos, situao em que a
   *         finalizao do comando; ou <code>true</code> se a finalizao puder
   *         continuar normalmente (mesmo se a remoo do sandbox der erro, no
   *         prejudica o resultado para o usurio).
   */
  boolean clearExecutionEnvironment(Command command) {
    SGA sga = command.getSGA();
    String csfsHost = sga.getCSFSHost();
    int csfsPort = sga.getCSFSPort();
    String[] sandbox = command.getSandbox();
    CSFSService csfsService = CSFSService.getInstance();
    if ((!command.existsProject()) || csfsService == null || csfsHost == null
      || sandbox == null) {
      return true;
    }
    try {
      Map<String[], Long> sandboxInitState = command.getSandboxSnapshot();
      Map<String[], Long> sandboxFinalState =
        csfsService.getTimestamps(csfsHost, csfsPort, sandbox);
      updateProjectArea(command, sandboxInitState, sandboxFinalState);
    }
    catch (ServerException exc) {
      Server.logSevereMessage("Falha na recuperao de dados do sandbox", exc);
      return false;
    }
    try {
      /*
       * BUG WARNING: Eh necessario copiar ao menos o arquivo de log do comando,
       * para permitir que seja possivel avaliar o funcionamento do comando.
       */
      // remove o sandbox
      csfsService.remove(csfsHost, csfsPort, sandbox);
    }
    catch (ServerException exc) {
      Server.logSevereMessage("Falha na remocao de dados do sandbox", exc);
    }
    return true;
  }

  /**
   * Responsvel por comparar o estado do sandbox antes e depois da execucao do
   * comando (copiando os arquivos necessarios para a rea de projetos).
   *
   * @param command O comando.
   * @param sandboxInit Estado inicial da sandbox.
   * @param sandboxFinal Estado final da sandbox.
   * @throws ServerException caso haja algum problema na cpia dos arquivos para
   *         a rea de projetos via CSFS.
   */
  private void updateProjectArea(Command command,
    Map<String[], Long> sandboxInit, Map<String[], Long> sandboxFinal)
      throws ServerException {
    SGA sga = command.getSGA();
    CSFSService csfsService = CSFSService.getInstance();
    if (csfsService == null) {
      String cmdId = command.getId();
      String errorMsg =
        "O CSFSService nao foi localizado"
          + "durante a sincronizacao do sandbox.\n";
      Server
      .logSevereMessage(errorMsg
        + "\tCausa: Falha na construo do comando: " + "\n\tSGA ["
        + sga.getName() + "] - id: " + cmdId + ".\n\tErro: " + errorMsg
        + ".\n");
      sga.failCommand(cmdId, FailureFinalizationType.CSFS_SERVICE_UNAVAILABLE);
    }
    else {
      try {
        Thread.sleep(csfsUpdateProjectAreaDelay);
      }
      catch (InterruptedException e) {
        Server.logSevereMessage("Tempo de espera do CSFS interrompido!");
      }
      String csfsHost = sga.getCSFSHost();
      int csfsPort = sga.getCSFSPort();
      String[] sandbox = command.getSandbox();
      // obtem o conjunto de arquivos no sandbox (depois da execucao do
      // algoritmo)
      Set<Entry<String[], Long>> entrySet = sandboxFinal.entrySet();
      // percorre todos os arquivos verificando sua data de modificacao
      for (Entry<String[], Long> entry : entrySet) {
        String[] remoteFilePath = entry.getKey();
        // recupera a data de modificacao do arquivo
        long finalTime = entry.getValue().longValue();
        // procura pela data de modificacao antes da execucao do comando
        Long originalTimeLong = sandboxInit.get(remoteFilePath);
        // caso o arquivo tenha sido criado ou alterado
        if (originalTimeLong == null
          || finalTime > originalTimeLong.longValue()) {
          // caminho relativo do arquivo (sem o caminho at a sandbox)
          String[] relativeFilePath =
            Arrays.copyOfRange(remoteFilePath, sandbox.length,
              remoteFilePath.length);
          String[] localFilePath =
            combinePaths(csfsService.getCSFSProjectsRootPath(), command
              .getProjectPath(), relativeFilePath);
          // copia o arquivo do sandbox para a area de projetos
          csfsService.copyFrom(csfsHost, csfsPort, remoteFilePath,
            localFilePath);
        }
        // remove o arquivo do snapshot inicial, apenas para controle
        sandboxInit.remove(remoteFilePath);
      }

      if (!sandboxInit.isEmpty()) {
        // BUG WARNING
        // eventualmente, tratar a remocao de arquivos por parte do algoritmo...
        // ou seja, remover os arquivos da area de projeto.
        Server.logWarningMessage("Ooops! sandbox shrunk! command.getId()="
          + command.getId());
      }
    }
  }

  /**
   * Retorna o nome do diretrio de persistncia para comandos.
   *
   * @return nome do diretrio de persistncia para comandos.
   *
   * @throws OperationFailureException em caso de erro na validao do diretrio
   *         no sistema de arquivos.
   */
  public String getCommandsPersistencyDirectoryName()
    throws OperationFailureException {
    try {
      Server srv = Server.getInstance();
      String directory =
        srv.getPersistencyRootDirectoryName() + File.separator + COMMANDS_DIR;
      Server.checkDirectory(directory);
      return directory;
    }
    catch (ServerException e) {
      throw new OperationFailureException(e);
    }
  }

  /**
   * Constri a instncia do servio de gerncia de SGAs
   *
   * @throws ServerException se houver erro na instanciao.
   */
  public static void createService() throws ServerException {
    new SGAService();
  }

  /**
   * Construtor.
   *
   * @throws ServerException se houver erro na instanciao.
   */
  protected SGAService() throws ServerException {
    super(SERVICE_NAME);
    ClientRemoteLocator.sgaService = this;
    registeredSGAs = new Hashtable<String, SGA>();
    lockSGAs = Collections.synchronizedSet(new HashSet<String>());

    for (CommandEvent.EventType type : CommandEvent.EventType.values()) {
      finishedCmdsCount.put(type, new AtomicLong(0));
    }
  }

  /**
   * Thread responsvel por atualizar as taxas de transferncia de dados na rede
   * de cada SGA.
   */
  private class TransferRateAcquisitionThread extends Thread {
    /**
     * Coleta taxas de transferncia a cada perodo determinado pelo padro da
     * propriedade processing.interval do SchedulerService a qual corresponde ao
     * dobro do valor da propriedade sgaUpdateIntervalSeconds do SGAService.
     */
    @Override
    public void run() {
      while (true) {
        FindTransferRate transferThread;
        Enumeration<SGA> sgas = registeredSGAs.elements();
        while (sgas.hasMoreElements()) {
          SGA sga = sgas.nextElement();
          try {
            if (sga.execNetBench() == SGAInfo.CALC_CAPACITY) {
              transferThread = new FindTransferRate(sga, netBenchDuration);
              transferThread.setName(this.getClass().getSimpleName() + "::"
                + "FindTransferRateThread");
              transferThread.start();
              try {
                transferThread.join(iperfTimeout);
              }
              catch (InterruptedException e) {
                // Thread j foi interrompida por outra thread.
                Server.logSevereMessage(
                  "Falha na coleta da taxa de transferncia de "
                    + "dados na rede para o SGA: " + sga.getName(), e);
                sga.setTransferRate(SGAInfo.NO_CAPACITY);
                continue;
              }
            }
          }
          catch (ServerException e) {
            Server.logSevereMessage(
              "Falha na determinao se o benchmark de rede "
                + "deve ser executado para o SGA: " + sga.getName() + ".", e);
            sga.setTransferRate(SGAInfo.NO_CAPACITY);
          }
        }

        try {
          Thread.sleep(updateDataTransferRate);
        }
        catch (InterruptedException ex) {
        }

        transferThread = null;
        sgas = null;
      }
    }
  }

  class ParametersReportMaker {

    private final String emptyValueTemplate =
      getString("SGAService.mail.content.param.value.0");

    private final String singleValueTemplate =
      getString("SGAService.mail.content.param.value.1");

    private final String multiValueTemplate =
      getString("SGAService.mail.content.param.value.n");
    private final String valueTemplate = getString("SGAService.value");

    public String makeReport(List<ParameterGroup> groups) {
      final StringBuilder reportBuilder = new StringBuilder();
      report(reportBuilder, groups);
      return reportBuilder.toString();
    }

    private void report(StringBuilder reportBuilder,
      Collection<ParameterGroup> groups) {
      for (ParameterGroup group : groups) {
        report(reportBuilder, group);
      }
    }

    private void report(StringBuilder reportBuilder, Parameter<?> param) {
      if (param instanceof ParameterGroup) {
        report(reportBuilder, (ParameterGroup) param);
      }
      else if (param instanceof BooleanParameter) {
        report(reportBuilder, (BooleanParameter) param);
      }
      else if (param instanceof FileParameter) {
        report(reportBuilder, (FileParameter) param);
      }
      else if (param instanceof ListParameter<?>) {
        report(reportBuilder, (ListParameter<?>) param);
      }
      else if (param instanceof InputFileListParameter) {
        report(reportBuilder, (InputFileListParameter) param);
      }
      else if (param instanceof TableParameter) {
        report(reportBuilder, (TableParameter) param);
      }
      else if (param instanceof SimpleParameter<?>) {
        report(reportBuilder, (SimpleParameter<?>) param);
      }
    }

    private void report(StringBuilder reportBuilder, ParameterGroup group) {
      for (Parameter<?> param : group.getParameters()) {
        report(reportBuilder, param);
        report(reportBuilder, group.getGroups());
      }
    }

    private void report(StringBuilder reportBuilder, BooleanParameter param) {
      String value = getString("SGAService.param.value." + param.getValue());
      writeParamValue(reportBuilder, param, value);
    }

    private void report(StringBuilder reportBuilder, FileParameter param) {
      FileURLValue file = param.getValue();
      String value = null == file ? null : file.getPath();
      writeParamValue(reportBuilder, param, value);
    }

    private void report(StringBuilder reportBuilder, ListParameter<?> param) {

      List<?> paramValues = param.getValue();
      if (paramValues == null) {
        writeParamValue(reportBuilder, param, null);
      }
      else {
        String valueSeparator = ", ";
        StringBuilder sb = new StringBuilder();
        for (Object value : paramValues) {
          if (null == value) {
            continue;
          }

          if (0 < sb.length()) {
            sb.append(valueSeparator);
          }
          sb.append('\"').append(value).append('\"');
        }

        String value = sb.toString();
        writeParamValue(reportBuilder, param, value);
      }
    }

    private void report(StringBuilder reportBuilder,
      InputFileListParameter param) {
      List<String> values = new ArrayList<String>();

      List<FileURLValue> files = param.getValue();
      if (files != null) {
        for (FileURLValue file : files) {
          if (null != file) {
            values.add(file.getPath());
          }
        }
      }

      writeParamValues(reportBuilder, param, values);
    }

    private void report(StringBuilder reportBuilder, TableParameter param) {
      final String valueSeparator = ", ";
      final String emptyValue = "";

      List<String> values = new ArrayList<String>();

      StringBuilder sb = new StringBuilder();
      for (int i = 0; i < param.getRowCount(); i++) {
        sb.append(i + 1).append(") ");
        for (int j = 0; j < param.getColumnCount(); j++) {
          if (0 < j) {
            sb.append(valueSeparator);
          }

          Object itemValue = param.getItemValue(i, j);
          if (itemValue instanceof Boolean) {
            boolean booleanValue = ((Boolean) itemValue).booleanValue();
            String value = getString("SGAService.param.value." + booleanValue);
            sb.append(value);
          }
          else {
            Column column = param.getColumns().get(j);
            String value = column.getItemValueAsText(itemValue);
            value = null == value ? emptyValue : value;
            sb.append('\"').append(value).append('\"');
          }

        }
        values.add(sb.toString());
        sb.delete(0, sb.length());
      }

      writeParamValues(reportBuilder, param, values);
    }

    private void report(StringBuilder reportBuilder, SimpleParameter<?> param) {
      String value = param.getValueAsText();
      writeParamValue(reportBuilder, param, value);
    }

    private void writeParamValues(StringBuilder reportBuilder,
      SimpleParameter<?> param, List<String> values) {

      String label = param.getLabel();
      if (null == values || 0 == values.size()) {
        reportBuilder.append(String.format(emptyValueTemplate, label));
      }
      else {
        reportBuilder.append(String.format(multiValueTemplate, label));
        for (String value : values) {
          reportBuilder.append(String.format(valueTemplate, value));
        }
      }
      reportBuilder.append("\n");
    }

    private void writeParamValue(StringBuilder reportBuilder,
      SimpleParameter<?> param, String value) {

      String label = param.getLabel();
      if (null == value) {
        reportBuilder.append(String.format(emptyValueTemplate, label));
      }
      else {
        reportBuilder.append(String.format(singleValueTemplate, label, value));
      }
      reportBuilder.append("\n");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<CommandFinalizationType, Integer> getExeResultsStats(
    boolean flowResults) throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    // DEBUG
    // Map<CommandFinalizationType, Integer> result =
    // new TreeMap<CommandFinalizationType, Integer>();
    // if (!flowResults) {
    // result.put(CommandFinalizationType.FAILED, 20);
    // result.put(CommandFinalizationType.KILLED, 10);
    // result.put(CommandFinalizationType.SUCCESS, 1000);
    // }
    // else {
    // result.put(CommandFinalizationType.FAILED, 5);
    // result.put(CommandFinalizationType.KILLED, 30);
    // result.put(CommandFinalizationType.SUCCESS, 500);
    // }
    // return result;
    return Collections.unmodifiableMap(flowResults ? flowResultCounter
      : exeResultCounter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Integer> getSGAsStats() throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return Collections.unmodifiableMap(sgasCounter);
  }
}
