/**
 * $Id$
 */

package csbase.server.services.projectservice;

import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import csbase.exception.InvalidRequestException;
import csbase.logic.FileLockListenerInterface;
import csbase.logic.SecureKey;
import csbase.remote.ProjectServiceInterface;
import csbase.server.Server;
import csbase.server.services.projectservice.FileLockInterface.LockStatus;

/**
 * Classe responsvel por gerenciar os locks de arquivos do servio de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
class FileLockManager {
  /**
   * Tempo mximo que a thread de processamento da fila dorme: 1 minuto.
   */
  private final static long MAX_SLEEP_INTERVAL = 60000;

  /**
   * Instncia nica do <code>FileLockManager</code>.
   */
  private static FileLockManager instance;

  /**
   * Mapa com os locks para arquivos. Os locks possuem referncias fracas para
   * as chaves das sesses; se elas expiram, os locks so liberados.
   */
  private final Map<String, FileLockInterface> lockedFiles;

  /**
   * Thread responsvel por escalonar os pedidos de lock quando um lock 
   * liberado ou quando o tempo de espera (timeout) da solicitao de lock
   * expira.
   */
  private Thread managerThread;

  /** Indica o trmino do processamento da fila de locks */
  private boolean exitManagerThread;

  /**
   * Representa as filas de requisies de lock.
   */
  private class LockQueue {
    /**
     * Mapa com as filas de requisies de locks.
     */
    private final Map<String, List<LockRequest>> lockRequests;

    /**
     * Requisies ordenadas pela data de expirao.
     */
    private final List<LockRequest> sortedLockRequests;

    /**
     * Constri a fila de requisio de locks.
     */
    LockQueue() {
      this.lockRequests = new Hashtable<String, List<LockRequest>>();
      this.sortedLockRequests = new ArrayList<LockRequest>();
    }

    /**
     * Obtm o prximo pedido de lock a expirar
     * 
     * @return o prximo pedido de lock a expirar
     */
    LockRequest getNextRequest() {
      synchronized (FileLockManager.this) {
        if (sortedLockRequests.size() > 0) {
          return sortedLockRequests.get(0);
        }
        return null;
      }
    }

    /**
     * Adiciona a requisio o pedido de lock na fila.
     * 
     * @param request pedido de lock
     */
    void addRequest(LockRequest request) {
      synchronized (FileLockManager.this) {
        List<LockRequest> requests =
          lockRequests.get(request.projectFile.getAbsolutePath());
        if (requests == null) {
          requests = new ArrayList<LockRequest>();
          lockRequests.put(request.projectFile.getAbsolutePath(), requests);
        }
        requests.add(request);
        addSortedRequest(request);
        FileLockManager.this.notifyAll();
      }
    }

    /**
     * Adiciona uma requisio  fila ordenada por prioridade.
     * 
     * @param request requisio de lock
     */
    private void addSortedRequest(LockRequest request) {
      if (sortedLockRequests.size() <= 0) {
        sortedLockRequests.add(request);
        return;
      }
      for (int i = 0; i < sortedLockRequests.size(); i++) {
        if (request.expirationDate < sortedLockRequests.get(i).expirationDate) {
          sortedLockRequests.add(i, request);
          return;
        }
      }
      /*
       * Se chegou aqui, a data de expirao  maior do que todos que j est na
       * fila.
       */
      sortedLockRequests.add(request);
    }

    /**
     * Obtm as requisies de lock para o um determinado arquivo.
     * 
     * @param filePath caminho do arquivo
     * @return as requisies de lock para o um determinado arquivo
     */
    LockRequest[] getRequests(String filePath) {
      synchronized (FileLockManager.this) {
        List<LockRequest> requests = lockRequests.get(filePath);
        return requests.toArray(new LockRequest[requests.size()]);
      }
    }

    /**
     * Remove as requisies de um determinado arquivo.
     * 
     * @param filePath caminho do arquivo
     * @param requests requisies a serem removidas
     */
    void remove(String filePath, List<LockRequest> requests) {
      synchronized (FileLockManager.this) {
        List<LockRequest> queue = lockRequests.get(filePath);
        if (queue == null) {
          return;
        }
        queue.removeAll(requests);
        if (queue.isEmpty()) {
          lockRequests.remove(filePath);
        }
        sortedLockRequests.removeAll(requests);
      }
    }

    /**
     * Remove a requisio de lock de um determinado arquivo.
     * 
     * @param filePath caminho do arquivo
     * @param request requisio a ser removidas
     * @return true se o pedido de lock estava na fila
     */
    boolean remove(String filePath, LockRequest request) {
      synchronized (FileLockManager.this) {
        List<LockRequest> queue = lockRequests.get(filePath);
        if (queue == null) {
          return false;
        }
        queue.remove(request);
        if (queue.isEmpty()) {
          lockRequests.remove(filePath);
        }
        return sortedLockRequests.remove(request);
      }
    }

    /**
     * Obtm uma requisio de lock com o caminho de arquivo e identificador
     * especificados
     * 
     * @param filePath caminho do arquivo cujo lock foi pedido
     * @param lockId identificador da requisio de lock
     * @return uma requisio de lock
     */
    LockRequest getRequest(String filePath, Object lockId) {
      synchronized (FileLockManager.this) {
        List<LockRequest> requests = lockRequests.get(filePath);
        if (requests == null) {
          return null;
        }
        for (LockRequest request : requests) {
          if (request.id.equals(lockId)) {
            return request;
          }
        }
        return null;
      }
    }

    /**
     * Obtm as chaves (filePath) das filas de lock. Este mtodo chama o mtodo
     * <code>wait</code> caso no haja requisies de lock. O
     * <code>notify</code>  chamado quando um lock  requirido pelo mtodo
     * <code>addRequest</code> ou quando um lock  liberado.
     * 
     * @return as chaves (filePath) das filas de lock
     */
    String[] getAndWaitForQueuesKeys() {
      synchronized (FileLockManager.this) {
        while (size() <= 0) {
          try {
            FileLockManager.this.wait();
          }
          catch (InterruptedException e) {
          }
        }
        return lockRequests.keySet().toArray(new String[lockRequests.size()]);
      }
    }

    /**
     * Obtm o nmero de filas de requisies de lock.
     * 
     * @return o nmero de filas de requisies de lock
     */
    int size() {
      synchronized (FileLockManager.this) {
        return lockRequests.size();
      }
    }

    /**
     * Dorme durante o tempo especificado ou at que seja acordada por outra
     * thread que invoque <code>notify</code>.
     * 
     * @param interval tempo mximo de interrupo da execuo
     */
    void sleep(long interval) {
      synchronized (FileLockManager.this) {
        try {
          FileLockManager.this.wait(interval);
        }
        catch (InterruptedException e) {
        }
      }
    }

    /**
     * Mtodo chamado quando um lock  liberado.
     */
    void lockReleased() {
      synchronized (FileLockManager.this) {
        FileLockManager.this.notifyAll();
      }
    }
  }

  /**
   * Fila de requisies de lock.
   */
  private final LockQueue lockQueue;

  /**
   * Representa uma requisio de lock de arquivo.
   */
  private class LockRequest {
    /**
     * Identificador do pedido de lock.  usado para identificar que o
     * desbloqueio  feito por quem pediu o lock.
     */
    SecureKey id;
    /**
     * Sesso do usurio que solitou o lock. Pode ser null se o pedido de lock
     * veio de uma chamada que no foi originada pela thread do cliente.
     */
    SecureKey sessionKey;
    /** Observador a ser notificado quando lock for obtido ou expirado. */
    FileLockListenerInterface listener;
    /** Data limite para que o lock seja obtido */
    long expirationDate;
    /** Indica se o lock  compartilhado */
    boolean shared;
    /** Arquivo sobre o qual o lock foi solicitado. */
    ServerProjectFile projectFile;

    /**
     * Constri uma requisio de lock.
     * 
     * @param sessionKey sesso do usurio
     * @param listener observador a ser notificado quando for obtido ou expirado
     * @param expirationDate data limite para que o lock seja obtido
     * @param shared indica se o lock  compartilhado
     * @param projectFile arquivo sobre o qual foi solicitado o lock
     */
    LockRequest(SecureKey sessionKey, FileLockListenerInterface listener,
      long expirationDate, boolean shared, ServerProjectFile projectFile) {
      this.id = new SecureKey();
      this.sessionKey = sessionKey;
      this.listener = listener;
      this.expirationDate = expirationDate;
      this.shared = shared;
      this.projectFile = projectFile;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      StringBuilder str = new StringBuilder();
      str.append("\nId: " + id);
      str.append("Arquivo pedido para lock: " + projectFile.getAbsolutePath());
      str.append("\nSesso do Usurio: " + sessionKey);
      str.append("Data de expirao: " + new Date(expirationDate));
      str.append("\nCompartilhado: " + shared);
      return str.toString();
    }
  }

  /**
   * Constri o gerenciador de locks.
   */
  private FileLockManager() {
    this.lockedFiles = new Hashtable<String, FileLockInterface>();
    this.lockQueue = new LockQueue();
  }

  /**
   * Retorna a instncia nica do gerenciador de locks de arquivos.
   * 
   * @return o gerenciado de locks
   */
  synchronized static FileLockManager getInstance() {
    if (instance == null) {
      instance = new FileLockManager();
    }
    return instance;
  }

  /**
   * Obtm o lock de um arquivo a partir do seu path. Pode retornar null se no
   * existir lock para o arquivo especificado.
   * 
   * @param filePath caminho do arquivo
   * @return o lock de um arquivo
   */
  FileLockInterface getLock(String filePath) {
    return lockedFiles.get(filePath);
  }

  /**
   * Cria um lock compartilhado para um arquivo. Se j existir um lock exclusivo
   * para o arquivo, a operao resultar em falha.
   * 
   * @param prjFile referncia para o arquivo
   * @param sessionKey sesso do usurio
   * @return indicador se o lock foi obtido com sucesso, ou null caso contrrio
   */
  Object acquireSharedLock(ServerProjectFile prjFile, SecureKey sessionKey) {
    SecureKey lockId = new SecureKey();
    if (acquireSharedLock(prjFile, sessionKey, lockId)) {
      return lockId;
    }
    return null;
  }

  /**
   * Cria um lock compartilhado para um arquivo. Se j existir um lock exclusivo
   * para o arquivo, a operao resultar em falha.
   * 
   * @param prjFile - referncia para o arquivo
   * @param sessionKey sesso do usurio
   * @param lockId identificador do pedido de lock
   * @return flag indicando se o lock foi obtido com sucesso
   */
  private synchronized boolean acquireSharedLock(ServerProjectFile prjFile,
    SecureKey sessionKey, Object lockId) {
    if (prjFile == null) {
      return false;
    }
    String absolutePath = prjFile.getAbsolutePath();
    FileLockInterface lockRef = lockedFiles.get(absolutePath);
    if (lockRef != null) {
      // existe um lock para o arquivo
      if (!lockRef.isShared()) {
        //  um lock exclusivo
        return false;
      }
      lockRef.newLocker(sessionKey, lockId);
      /*
       * bloqueamos o path at o arquivo com locks compartilhados, para bloquear
       * operaes destrutivas (move, remove, rename etc.)
       */
      acquireSharedLock(prjFile.getParent(), sessionKey, lockId);
      return true;
    }
    lockedFiles.put(absolutePath, new SharedFileLock(sessionKey, lockId));
    /*
     * bloqueamos o path at o arquivo com locks compartilhados, para bloquear
     * operaes destrutivas (move, remove, rename etc.)
     */
    acquireSharedLock(prjFile.getParent(), sessionKey, lockId);
    return true;
  }

  /**
   * Tenta obter um lock exclusivo para um arquivo. Caso este se encontre
   * bloqueado, uma requisio  cadastrada na fila.
   * 
   * @param prjFile arquivo em que deseja o lock
   * @param sessionKey sesso do usurio
   * @param listener observador a ser notificado quando o lock  obtido ou o
   *        timeout atingido
   * @param timeout tempo de espera pelo lock
   * @return identificador do lock
   */
  Object acquireExclusiveLock(ServerProjectFile prjFile, SecureKey sessionKey,
    FileLockListenerInterface listener, long timeout) {
    return registerLockRequest(prjFile, listener, timeout, false, sessionKey);
  }

  /**
   * Tenta obter um lock compartilhado para um arquivo. Caso este se encontre
   * bloqueado, uma requisio  cadastrada na fila.
   * 
   * @param prjFile arquivo em que deseja o lock
   * @param sessionKey sesso do usurio
   * @param listener observador a ser notificado quando o lock  obtido ou o
   *        timeout atingido
   * @param timeout tempo de espera pelo lock
   * @return identificador do lock
   */
  Object acquireSharedLock(ServerProjectFile prjFile, SecureKey sessionKey,
    FileLockListenerInterface listener, long timeout) {
    return registerLockRequest(prjFile, listener, timeout, true, sessionKey);
  }

  /**
   * Tenta obter um lock para um arquivo. Caso este se encontre bloqueado, uma
   * requisio  cadastrada na fila.
   * 
   * @param prjFile arquivo em que deseja o lock
   * @param listener observador a ser notificado quando o lock  obtido ou o
   *        timeout atingido
   * @param timeout tempo de espera pelo lock
   * @param shared tipo de lock
   * @param sessionKey chave da sesso do usurio
   * @return identificador do pedido de lock
   */
  private synchronized Object registerLockRequest(ServerProjectFile prjFile,
    FileLockListenerInterface listener, long timeout, boolean shared,
    SecureKey sessionKey) {
    long expirationDate =
      (timeout == ProjectServiceInterface.INFINITE_TIMEOUT) ? timeout : System
        .currentTimeMillis()
        + timeout;
    LockRequest request =
      new LockRequest(sessionKey, listener, expirationDate, shared, prjFile);
    if (isLocked(prjFile)) {
      lockQueue.addRequest(request);
    }
    else {
      if (shared) {
        acquireSharedLock(prjFile, sessionKey, request.id);
      }
      else {
        acquireExclusiveLock(prjFile, sessionKey, request.id);
      }
      notifyFileLocked(request);
    }
    return request.id;
  }

  /**
   * Cria um lock exclusivo para o arquivo. Caso este j se encontre bloqueado,
   * a operao resultar em falha.
   * 
   * @param prjFile referncia para o arquivo
   * @param sessionKey sesso do usurio
   * @return identificador do lock se o lock foi obtido com sucesso, ou null
   *         caso contrrio
   */
  Object acquireExclusiveLock(ServerProjectFile prjFile, SecureKey sessionKey) {
    SecureKey lockId = new SecureKey();
    if (acquireExclusiveLock(prjFile, sessionKey, lockId)) {
      return lockId;
    }
    return null;
  }

  /**
   * Cria um lock exclusivo para o arquivo. Caso este j se encontre bloqueado,
   * a operao resultar em falha.
   * 
   * @param prjFile referncia para o arquivo
   * @param sessionKey sesso do usurio
   * @param lockId identificador do pedido de lock
   * @return flag indicando se o lock foi obtido com sucesso
   */
  private synchronized boolean acquireExclusiveLock(ServerProjectFile prjFile,
    SecureKey sessionKey, Object lockId) {
    if (prjFile == null) {
      return false;
    }
    String absolutePath = prjFile.getAbsolutePath();
    FileLockInterface lockRef = lockedFiles.get(absolutePath);
    if (lockRef != null) {
      // existe um lock para o arquivo
      if (lockRef.isShared()) {
        // era um lock compartilhado, no podemos prosseguir
        return false;
      }
      if (!lockRef.hasExpired()) {
        /*
         * algum usurio cuja sesso ainda est ativa est bloquando o arquivo
         * para escrita (pode inclusive ser o mesmo usurio que est tentando
         * escrever agora)
         */
        return false;
      }
      else {
      }
    }
    /*
     * ou o arquivo ainda no havia sido bloqueado ou a sesso do usurio que o
     * bloqueou j expirou. Neste caso, criamos outro lock para a sesso
     * corrente.
     */
    lockedFiles.put(absolutePath, new ExclusiveFileLock(sessionKey, lockId));

    /*
     * bloqueamos o path at o arquivo com locks compartilhados, para bloquear
     * operaes destrutivas (move, remove, rename etc.)
     */
    acquireSharedLock(prjFile.getParent(), sessionKey, lockId);
    return true;
  }

  /**
   * Remove o lock para um path. Se o lock  exclusivo, remove apenas se o
   * identificador do lock  igual ao id do lock do arquivo. Se o lock 
   * compartilhado e o usurio  um dos que solicitou o lock, decrementa o
   * contador de referncias para o lock; se o a contagem chegar a zero, remove
   * o lock.
   * <p>
   * O lock s  removido se o identificador do lock  igual ao parmetro
   * <code>lockId</code>
   * 
   * @param absolutePath path
   * @param lockId identificador do lock
   * @return contagem de referncias para o lock
   */
  synchronized int releaseLock(String absolutePath, Object lockId) {
    if (lockId == null) {
      throw new InvalidRequestException("lockId == null");
    }
    if (absolutePath == null) {
      return 0;
    }
    // Tenta remover da fila
    LockRequest request = lockQueue.getRequest(absolutePath, lockId);
    if (request != null) {
    }

    FileLockInterface lockRef = lockedFiles.get(absolutePath);
    if (lockRef != null) {
      /*
       * j existe um lock para o path especificado, precisamos verificar se a
       * sesso associada est vlida
       */
      lockRef.removeLocker(lockId);
      releaseLock(getParentPath(absolutePath), lockId);
      if (lockRef.getLockRefCount() == 0) {
        lockedFiles.remove(absolutePath);
        lockQueue.lockReleased();
        return 0;
      }
      else {
        return lockRef.getLockRefCount();
      }
    }
    // o lock no existia, voltamos como se tivesse sido removido com sucesso
    return 0;
  }

  /**
   * Remove o lock para um arquivo. Se o lock  exclusivo, remove apenas se o
   * identificador do lock  o mesmo do lock obtido. Se o lock  compartilhado e
   * o usurio  um dos que solicitou o lock, decrementa o contador de
   * referncias para o lock; se o a contagem chegar a zero, remove o lock.
   * <p>
   * O lock s  removido se o identificador do lock  igual ao parmetro
   * <code>lockId</code>.
   * 
   * @param prjFile - referncia para o arquivo
   * @param lockId identificador do lock
   * @return contagem de referncias para o lock
   */
  int releaseLock(ServerProjectFile prjFile, Object lockId) {
    if (prjFile == null) {
      return 0;
    }
    int lockRefCount = releaseLock(prjFile.getAbsolutePath(), lockId);
    return lockRefCount;
  }

  /**
   * Fora a remoo do lock do arquivo. S deve ser usado pelo administrador se
   * acontencer alguma situao de ter um lock "preso". Situaes onde as sesso
   * do usurio que foi finalizadas por falha na comunicao, ou usurios que
   * terminam o cliente abruptamente j esto previstas e no devem precisar
   * deste mecanismo de segurana.
   * 
   * @param prjFile arquivo sobre o qual se deseja remover o lock
   * @return nmero de referncias para o lock
   */
  synchronized int forceReleaseLock(ServerProjectFile prjFile) {
    if (prjFile == null) {
      return 0;
    }
    FileLockInterface fileLock = lockedFiles.get(prjFile.getAbsolutePath());
    if (fileLock == null) {
      return 0;
    }
    int lockRefCount = 0;
    if (fileLock.isShared()) {
      Object[] lockIds = ((SharedFileLock) fileLock).getIds();
      for (Object id : lockIds) {
        lockRefCount = releaseLock(prjFile.getAbsolutePath(), id);
      }
    }
    else {
      lockRefCount =
        releaseLock(prjFile.getAbsolutePath(), ((ExclusiveFileLock) fileLock)
          .getId());
    }
    return lockRefCount;
  }

  /**
   * Retorna a contagem de referncias para o lock de um determinado arquivo.
   * 
   * @param prjFile - arquivo que se deseja verificar
   * @return contagem de referncias para o lock do arquivo. Se o arquivo no
   *         possuir lock, retorna 0.
   */
  synchronized int getLockRefCount(ServerProjectFile prjFile) {
    String absolutePath = prjFile.getAbsolutePath();
    FileLockInterface lock = lockedFiles.get(absolutePath);
    if (lock == null) {
      return 0;
    }
    return lock.getLockRefCount();
  }

  /**
   * Verifica se existe um lock (exclusivo ou compartilhado) para um arquivo e,
   * se existe, se est associado a uma sesso especfica.
   * 
   * @param prjFile - arquivo que se deseja verificar
   * @param sessionKey - chave para a sesso
   * @return flag indicando se existe um lock para o arquivo associado  sesso
   */
  synchronized LockStatus isLocked(ServerProjectFile prjFile, String sessionKey) {
    String absolutePath = prjFile.getAbsolutePath();
    FileLockInterface lock = lockedFiles.get(absolutePath);
    if (lock != null) {
      LockStatus lockStatus = lock.checkLockStatus(sessionKey);
      if (lockStatus == LockStatus.EXPIRED) {
        lockedFiles.remove(absolutePath);
      }
      return lockStatus;
    }
    return LockStatus.UNLOCKED;
  }

  /**
   * Verifica se existe um lock (exclusivo ou compartilhado) para um arquivo
   * especfico.
   * 
   * @param prjFile - arquivo que se deseja verificar
   * @return flag indicando se existe um lock para o arquivo
   */
  boolean isLocked(ServerProjectFile prjFile) {
    return isLocked(prjFile.getAbsolutePath());
  }

  /**
   * Verifica se existe um lock (exclusivo ou compartilhado) para um arquivo
   * especfico.
   * 
   * @param absolutePath - path absoluto para o arquivo
   * @return flag indicando se existe um lock para o arquivo
   */
  synchronized boolean isLocked(String absolutePath) {
    FileLockInterface lock = lockedFiles.get(absolutePath);
    if (lock == null) {
      return false;
    }
    /*
     * j existe um lock para o path especificado, precisamos verificar se a
     * sesso associada est vlida
     */
    if (lock.hasExpired()) {
      lockedFiles.remove(absolutePath);
      return false;
    }
    else {
      return true;
    }
  }

  /**
   * Obtm o caminho do diretrio pai.
   * 
   * @param path caminho do arquivo
   * @return o caminho do diretrio pai
   */
  private String getParentPath(String path) {
    int lastSeparator = path.lastIndexOf(File.separator);
    if (lastSeparator == -1) {
      return null;
    }
    return path.substring(0, lastSeparator);
  }

  /**
   * Notifica o listener que o lock foi expirado.
   * 
   * @param lockRequest requisio de lock
   */
  private synchronized void notifyFileLockExpired(final LockRequest lockRequest) {
    Thread thread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          lockRequest.listener.fileLockExpired(lockRequest.id);
        }
        catch (RemoteException e) {
          Server
            .logSevereMessage("Falha na notificao de lock expirado. Pedido de lock: "
              + lockRequest);
        }
      }
    });
    thread.start();
  }

  /**
   * Notifica o listener que o lock foi obtido.
   * 
   * @param lockRequest requisio de lock
   */
  private synchronized void notifyFileLocked(final LockRequest lockRequest) {
    Thread thread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          lockRequest.listener.fileLocked(lockRequest.id);
        }
        catch (RemoteException e) {
          Server.logSevereMessage(
            "Falha na notificao de lock obtido. Pedido de lock: "
              + lockRequest, e);
        }
      }
    });
    thread.start();
  }

  /**
   * Remove da tabela que arquivos bloqueados aqueles que expiraram pela sesso
   * do usurio.
   */
  private synchronized void clearExpiredLockSessions() {
    Set<Entry<String, FileLockInterface>> locks = lockedFiles.entrySet();
    Set<String> expiredLocks = new HashSet<String>();
    for (Entry<String, FileLockInterface> lock : locks) {
      if (lock.getValue().hasExpired()) {
        expiredLocks.add(lock.getKey());
      }
    }
    // Remove da tabela de locks
    for (String filePath : expiredLocks) {
      lockedFiles.remove(filePath);
    }
  }

  /**
   * Processa as filas de requisies de locks. Antes de obter o lock, verifica
   * se a requisio expirou. No final, notifica os observadores os locks que
   * foram obtidos ou expirados.
   */
  private void processRequests() {
    String[] filePaths = lockQueue.getAndWaitForQueuesKeys();
    for (String filePath : filePaths) {
      List<LockRequest> acquiredLocks = new ArrayList<LockRequest>();
      List<LockRequest> expiredLocks = new ArrayList<LockRequest>();
      LockRequest[] requests = lockQueue.getRequests(filePath);
      long currentDate = System.currentTimeMillis();
      // Percorre a fila de solicities de lock. Se no consegue obter o lock,
      // verifica se expirou
      for (LockRequest lockRequest : requests) {
        boolean locked = false;
        if (lockRequest.shared) {
          locked =
            acquireSharedLock(lockRequest.projectFile, lockRequest.sessionKey,
              lockRequest.id);
        }
        else {
          locked =
            acquireExclusiveLock(lockRequest.projectFile,
              lockRequest.sessionKey, lockRequest.id);
        }
        if (locked) {
          acquiredLocks.add(lockRequest);
          notifyFileLocked(lockRequest);
        }
        else {
          if ((lockRequest.expirationDate != ProjectServiceInterface.INFINITE_TIMEOUT)
            && (currentDate >= lockRequest.expirationDate)) {
            expiredLocks.add(lockRequest);
            notifyFileLockExpired(lockRequest);
          }
        }
      }
      // Remove as requisies que obtiveram o lock
      lockQueue.remove(filePath, acquiredLocks);
      // Remove as requisies expiradas
      lockQueue.remove(filePath, expiredLocks);
    }
  }

  /**
   * Inicia a thread de processamento das requisies de lock.
   */
  void startThread() {
    managerThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitManagerThread) {
          clearExpiredLockSessions();
          processRequests();
          LockRequest next = lockQueue.getNextRequest();
          if (next != null) {
            long now = System.currentTimeMillis();
            /*
             * O intervalo  tempo que falta para o prximo comando expirar
             * menos 1 segundo. Fazemos isto para que a fila seja acordada antes
             * do comando expirar.
             */
            long nextTimeout = next.expirationDate - now;
            long interval =
              ((nextTimeout < 0) || (MAX_SLEEP_INTERVAL < nextTimeout)) ? MAX_SLEEP_INTERVAL
                : nextTimeout;
            lockQueue.sleep(interval);
          }
          else {
            // espera at que chegue um pedido de lock
          }
        }
      }

    }, "FileLockManager");
    managerThread.start();
  }

  /**
   * Interrompe a execuo da <code>Thread</code>.
   */
  void stopThread() {
    this.exitManagerThread = true;
  }

}
