package csbase.server.services.schedulerservice;

import java.io.Serializable;
import java.util.Comparator;
import java.util.List;

import csbase.logic.CapacityType;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;

/**
 * Comparador dos SGAs. Calcula o melhor SGA disponvel considerando os
 * critrios de capacidade de processamento, leitura/escrita em disco e taxa de
 * transferncia na rede. A capacidade de processamento contribui com 50% dessa
 * medida, enquanto a capacidade de leitura/escrita no disco juntamente com
 * capacidade de transferncia em rede com os 50% restante. Essa distribuio
 * consiste em uma heurstica que assume que um algoritmo passa metade do seu
 * tempo de execuo realizando processamento, isto , ele ele  considerado
 * igualmente limitado por CPU e I/O. Dessa maneira, essa poltica no escalonar
 * bem para algoritmos com grande quantidade de entrada/sada!!! Isso somente
 * ser possvel com a insero de um histrico de execues que informar ao
 * escalonador o tempo estimado de execuo de um algoritmo.
 */
public class SGASetCPUAndIOComparator implements Comparator<SGASet>,
  Serializable {
  private long minWritingCap;
  private long maxWritingCap;
  private long minSendCap;
  private long maxSendCap;
  private double minProcCap;
  private double maxProcCap;

  /**
   * Construtor do comparador.
   * 
   * @param serversNames Servidores disponveis para execuo
   * @param minWritingCap Capacidade de escrita em disco mnima entre os SGAs
   * @param maxWritingCap Capacidade de escrita em disco mxima entre os SGAs
   * @param minProcCap Capacidade de processamento mnima entre os SGAs
   * @param maxProcCap Capacidade de processamento mxima entre os SGAs
   * @param minSendCap Capacidade de transmisso em rede mnima entre os SGAs
   * @param maxSendCap Capacidade de transmisso em rede mxima entre os SGAs
   */
  public SGASetCPUAndIOComparator(List<String> serversNames,
    long minWritingCap, long maxWritingCap, double minProcCap,
    double maxProcCap, long minSendCap, long maxSendCap) {
    this.minWritingCap = minWritingCap;
    this.maxWritingCap = maxWritingCap;
    this.minProcCap = minProcCap;
    this.maxProcCap = maxProcCap;
    this.minSendCap = minSendCap;
    this.maxSendCap = maxSendCap;
  }

  /**
   * Compara as capacidades de dois SGAs.
   * 
   * @param sga1
   * @param sga2
   * @return -1 quando a capacidade do SGA1 for maior que a capacidade do SGA2.
   *         0 quando a capacidade do SGA1 for igual  capacidade do SGA2. 1
   *         quando a capacidade do SGA1 for menor que a capacidade do SGA2.
   */
  public int compare(SGASet sga1, SGASet sga2) {
    double sgaWriting1;
    double sgaWriting2;
    double sgaTransfer1;
    double sgaTransfer2;
    double cpuCap1;
    double cpuCap2;

    if (minWritingCap == maxWritingCap) {
      sgaWriting1 = 1;
      sgaWriting2 = 1;
    }
    else {
      sgaWriting1 = sga1.getCapacity(CapacityType.DISK_WRITE);
      sgaWriting2 = sga2.getCapacity(CapacityType.DISK_WRITE);

      if (sgaWriting1 == SGAInfo.NO_CAPACITY)
        sgaWriting1 = minWritingCap;
      if (sgaWriting2 == SGAInfo.NO_CAPACITY)
        sgaWriting2 = maxWritingCap;

      sgaWriting1 =
        (sgaWriting1 - minWritingCap) / (maxWritingCap - minWritingCap);
      sgaWriting2 =
        (sgaWriting2 - minWritingCap) / (maxWritingCap - minWritingCap);
    }

    if (minSendCap == maxSendCap) {
      sgaTransfer1 = 1;
      sgaTransfer2 = 1;
    }
    else {
      sgaTransfer1 = sga1.getCapacity(CapacityType.NET);
      sgaTransfer2 = sga2.getCapacity(CapacityType.NET);

      if (sgaTransfer1 == SGAInfo.NO_CAPACITY)
        sgaTransfer1 = minSendCap;
      else if (sgaTransfer1 == SGAInfo.ALL_CAPACITY)
        sgaTransfer1 = maxSendCap;
      if (sgaTransfer2 == SGAInfo.NO_CAPACITY)
        sgaTransfer2 = minSendCap;
      else if (sgaTransfer2 == SGAInfo.ALL_CAPACITY)
        sgaTransfer2 = maxSendCap;

      sgaTransfer1 = (sgaTransfer1 - minSendCap) / (maxSendCap - minSendCap);
      sgaTransfer2 = (sgaTransfer2 - minSendCap) / (maxSendCap - minSendCap);
    }

    if (minProcCap == maxProcCap) {
      cpuCap1 = 1;
      cpuCap2 = 1;
    }
    else {
      cpuCap1 = sga1.getCapacity(CapacityType.CPU);
      cpuCap2 = sga2.getCapacity(CapacityType.CPU);

      if (cpuCap1 == SGAInfo.NO_CAPACITY)
        cpuCap1 = 1;
      if (cpuCap2 == SGAInfo.NO_CAPACITY)
        cpuCap2 = 1;

      cpuCap1 = (1 / (sga1.getCPULoad1() + 1)) * cpuCap1;
      cpuCap1 = (cpuCap1 - minProcCap) / (maxProcCap - minProcCap);

      cpuCap2 = (1 / (sga2.getCPULoad1() + 1)) * cpuCap2;
      cpuCap2 = (cpuCap2 - minProcCap) / (maxProcCap - minProcCap);
    }

    double execCap1 =
      (0.25 * sgaWriting1) + (0.25 * sgaTransfer1) + (0.5 * cpuCap1);
    double execCap2 =
      (0.25 * sgaWriting2) + (0.25 * sgaTransfer2) + (0.5 * cpuCap2);

    if (execCap1 > execCap2) {
      return -1;
    }
    if (execCap1 < execCap2) {
      return 1;
    }
    return 0;
  }
}
