package csbase.server.services.eventlogservice;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import tecgraf.javautils.parsers.csv.CsvParser;
import tecgraf.javautils.parsers.exception.MalformedInputException;
import csbase.logic.User;
import csbase.logic.eventlogservice.LogsInfo;

/**
 * Parser para os arquivos de log de eventos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class EventLogParser {

  /**
   * Referncia para parser de arquivos csv.
   */
  private CsvParser csvParser;

  /**
   * Informaes logs lidas.
   */
  private LogsInfo info;

  /**
   * Formato para armazenar as datas nas tabelas de resultado.
   */
  protected static final DateFormat dateWriteFormat = new SimpleDateFormat(
    "yyyy/MM/dd");

  /**
   * Construtor.
   * 
   * @param startDate data inicial
   * @param endDate data final
   */
  public EventLogParser(Date startDate, Date endDate) {
    this.info = new LogsInfo(startDate, endDate);
    this.csvParser = new CsvParser();
  }

  /**
   * Faz o parse do stream fornecido e adiciona a lista de informaes de logs.
   * 
   * @param stream stream do arquivo de log
   * @param charset charset do servidor
   * 
   * @throws IOException exceo ao tentar fazer parser csv
   * @throws MalformedInputException exceo ao tentar fazer parser csv
   */
  public void parse(FileInputStream stream, Charset charset)
    throws IOException, MalformedInputException {

    String[][] lines = csvParser.parse(stream, charset);

    if (lines == null) {
      return;
    }

    for (int i = 0; i < lines.length; i++) {

      String[] line = lines[i];

      if (line[0].startsWith("#") || line[0].trim().length() == 0) {
        // ignorar linhas de cabealho ou invlidas *
        // * incorretamente idiom escrito em arquivos Desptop e Applications
        continue;
      }
      else {

        try {

          String user = line[2].trim();
          if (!User.isAdmin(user)) {
            long time = Long.parseLong(line[0].trim());
            String strDate = dateWriteFormat.format(time);
            String operation = line[3].trim();
            if (operation.equals("Aplicativo nativo lanado")) {
              info.addLineApplication(strDate, line[6].trim(), line[4].trim());
            }
            else if (operation.equals("login")) {
              info.addLineLogin(strDate, line[2].trim());
            }
            else if (operation.startsWith(user + "@")) {
              info.addLineExecution(strDate, line[4].trim(), user);
            }
          }
        }
        catch (Exception e) {
          continue;
        }
      }
    }

  }

  /**
   * Consulta as informaes lidas.
   * 
   * @return informaes dos logs
   */
  public LogsInfo getInfo() {
    return info;
  }

}
