/**
 * $Id: SchedulerService.java 174851 2016-07-07 17:52:00Z isabella $
 */

package csbase.server.services.schedulerservice;

import java.io.File;
import java.io.IOException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.Vector;

import csbase.exception.InitFailureException;
import csbase.exception.OperationFailureException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.algorithms.AlgorithmValidationException;
import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.CommandFailedNotification;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandKilledNotification;
import csbase.logic.CommandSubmission;
import csbase.logic.FailureFinalizationType;
import csbase.logic.Priority;
import csbase.logic.SchedulerStateChangedEvent;
import csbase.logic.SimpleCommandFinalizationInfo;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.Node;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.remote.RemoteObserver;
import csbase.remote.SchedulerServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.algorithmservice.AlgorithmService;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.sgaservice.SGAService;
import csbase.util.messages.Message;

/**
 * Servio escalonador de comandos.
 *
 * @author Tecgraf/PUC-Rio
 *
 */
public class SchedulerService extends Service
  implements SchedulerServiceInterface {

  /**
   * Gerador de identificadores de comandos.
   */
  private CommandIdGeneratorInterface commandIdGenerator;
  /**
   * Nmero de vezes que o scheduler tentar gerar um novo comando, caso um
   * comando com o mesmo id j exista.
   */
  private int commandIdRetries;
  /** Fila que controla a prioridade dos processos */
  private PriorityQueue commandQueue;
  /** Thread do escalonador */
  private Scheduler scheduler;

  /** Propriedade que define o tempo de quarentena do SGA */
  private static final String QUARANTINE_PROPERTY = "sga.quarantine.time";
  /** Propriedade que define o tempo de processamento da fila (milisegundos) */
  private static final String PROCESSING_INTERVAL_PROPERTY =
    "processing.interval";
  /** Propriedade que define a poltica de escalonamento a ser empregada */
  private static final String SCHED_POLICY_PROPERTY = "sga.schedule.policy";
  /** Propriedade que define se deve fazer o controle de recursos de cada sga */
  private static final String RESOURCES_CONTROL_PROPERTY =
    "sga.resources.control";
  /** Propriedade que define o gerador de identificadores de comandos */
  private static final String COMMAND_ID_GENERATOR_PROPERTY =
    "command.id.generator";
  /**
   * Propriedade que define o nmero de vezes que o scheduler tentar gerar um
   * novo comando, caso um comando com o mesmo id j exista.
   */
  private static final String COMMAND_ID_RETRIES_PROPERTY =
    "command.id.retries";

  /** Nome do arquivo de backup de recursos */
  private static final String RESOURCES_BACKUP_FILE_NAME =
    "resources-backup.dat";
  /** Nome do arquivo de backup */
  private static final String BACKUP_FILE_NAME = "queue-backup.dat";

  /** Valor padro do campo de identificao do algoritmo. */
  private static final String ABBREV_DEFAULT = "abbr";
  /** Tamanho do campo de identificao do algoritmo. */
  private static final int ABBREV_LENGTH = 4;
  /** Valor padro do campo de identificao do sistema. */
  private static final String SYSNAME_DEFAULT = "C";
  /** Tamanho do campo de identificao do sistema. */
  private static final int SYSNAME_LENGTH = 1;
  /**
   * Valor padro do campo de identificao do usuario que disparou o comando.
   */
  private static final String USERID_DEFAULT = "user";
  /** Tamanho do campo de identificao do usuario que disparou o comando. */
  private static final int USERID_LENGTH = 4;

  /** Referncia para o servio de mensagens */
  private MessageService messageService;

  /**
   * Mapa para contabilizar as submisses de comandos por tipo de execuo.
   * Usado na coleta de estatsticas.
   */
  private final Map<ExecutionType, Integer> exeTypeCounter =
    new TreeMap<ExecutionType, Integer>();

  /**
   * Mapa para contabilizar as submisses de execues simples, por algoritmo.
   * Usado na coleta de estatsticas.
   *
   * @see #flowAlgoNameCounter
   */
  private final Map<String, Integer> algoNameCounter =
    new TreeMap<String, Integer>();

  /**
   * Mapa para contabilizar as submisses de fluxos, por algoritmo. Usado na
   * coleta de estatsticas.
   *
   * @see #algoNameCounter
   */
  private final Map<String, Integer> flowAlgoNameCounter =
    new TreeMap<String, Integer>();

  /**
   * Mapa para contabilizar as execues por usurio, independente do tipo das
   * mesmas. Usado na coleta de estatsticas.
   */
  private final Map<String, Integer> userExeCounter =
    new TreeMap<String, Integer>();

  /**
   * Contador de execues de fluxos.
   */
  private int flowExecutionCounter;

  /**
   * Construtor da classe.
   *
   * @throws ServerException se houver erro na inicializao.
   */
  protected SchedulerService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Deprecated
  @Override
  public Set<CommandInfo> submitCommand(CommandSubmission commandSubmission,
    RemoteObserver... observers) throws RemoteException {
    return submitCommand(commandSubmission);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<CommandInfo> submitCommand(CommandSubmission submission)
    throws RemoteException {
    Set<CommandInfo> commands = new HashSet<CommandInfo>();
    switch (submission.getExecutionType()) {
      case SIMPLE:
        CommandInfo commandInfo = createCommand(submission);
        if (submission.isManual()) {
          String sgaName = submission.getSGANames().get(0);
          commandInfo.setSGAName(sgaName);
        }
        commands.add(commandInfo);
        commandQueue.add(commandInfo);
        break;
      case MULTIPLE:
        if (submission.isManual()) {
          int executionCountPerSGA =
            submission.getExecutionCountPerSGAForMultipleExecution();
          for (String sgaName : submission.getSGANames()) {
            for (int i = 0; i < executionCountPerSGA; i++) {
              CommandInfo command = createCommand(submission);
              command.setSGAName(sgaName);
              commands.add(command);
              commandQueue.add(command);
            }
          }
        }
        else {
          int executionCount =
            submission.getExecutionCountForMultipleExecution();
          for (int i = 0; i < executionCount; i++) {
            CommandInfo command = createCommand(submission);
            commands.add(command);
            commandQueue.add(command);
          }
        }
        break;
      default:
        throw new IllegalArgumentException("Tipo de execuo invlida.");
    }
    return commands;
  }

  /**
   * Cria um comando.
   * <p>
   * Persiste o comando utilizando o {@link CommandPersistenceService servio de
   * persistncia de comandos}. O servio de persistncia de comandos pode no
   * estar disponvel em todos os sistemas.
   * </p>
   * Registra os observadores do comando criado.
   *
   * @param submission dados da submisso do comando.
   *
   * @return o comando.
   * @throws ServiceFailureException Se no for possvel criar o comando com
   *         sucesso.
   */
  private CommandInfo createCommand(CommandSubmission submission) {
    Object userId = Service.getUser().getId();
    // TODO REMOVER APS TESTE
    // Service.setSystemId("VGE");
    // Service.setSystemId(null);
    String systemId = Service.getSystemId();
    String submissionInfo = submission + "usurio: " + userId;
    if (systemId != null) {
      submissionInfo += " - Sistema origem do comando: " + systemId;
    }

    try {
      CommandInfo command;
      Server.logInfoMessage(submissionInfo);
      AlgorithmConfigurator configurator =
        submission.createAlgorithmConfigurator();
      checkExecutionAlgorithmPermission(configurator);
      checkParameterValues(configurator, submission.getProjectId(), userId);
      /*
       * incrementamos as estatsticas
       */
      collectExeTypeStats(submission, configurator);
      collectAlgoNameStats(configurator);
      incrCounter(userExeCounter, Service.getUser().getLogin());

      String serverName = Server.getInstance().getSystemName();
      /* Obtm a abreviao do algoritmo. */
      String abbrev = configurator.getAbbreviation();
      if (abbrev == null) {
        ProjectService projectService = ProjectService.getInstance();
        String projectName =
          projectService.getProjectName(submission.getProjectId());
        /* Usa o nome do projeto como abreviao padro. */
        abbrev = projectName;
      }

      String user = userId.toString();
      CommandPersistenceService service =
        CommandPersistenceService.getInstance();
      String commandId = null;
      /*
       * A createCommandId garante que um servidor no ir gerar 2 ids iguais,
       * porm, estamos considerando o caso em que 2 ou mais servidores podem
       * escrever na mesma rea de projetos. Por isso, o mtodo
       * createUniqueCommandDirectory  chamado, de modo a garantir que o
       * identificador do comando seja nico.
       */
      int retries = commandIdRetries;
      boolean createNewCommandId = true;
      while (createNewCommandId) {
        commandId = createCommandId(user, abbrev, serverName);
        try {
          service.createUniqueCommandDirectory(submission.getProjectId(),
            commandId);
          createNewCommandId = false;
        }
        catch (ServiceFailureException sfe) {
          retries--;
          if (retries == 0) {
            Server.logSevereMessage(
              "Ocorreu um erro ao criar o diretrio do comando no servio de persistncia. O diretrio j existe.",
              sfe);
            throw new ServiceFailureException(
              "Falha na submisso de comando. Ocorreu um erro ao criar o diretrio do comando no servio de persistncia"
                + submissionInfo,
              sfe);
          }
          Server.logFineMessage(String.format(
            "Diretrio do comando %s j existe no servio de persistncia. Criando novo identificador...",
            commandId));
        }
      }
      command = new CommandInfo(commandId, configurator, submission, user);

      service.saveCommandInfo(command);

      return command;
    }
    catch (AlgorithmValidationException e) {
      Validation validationResult = e.getValidationResult();
      String message =
        "Falha na submisso de comando. Ocorreu um erro ao validar os parmetros do comando. ";
      if (validationResult != null) {
        message += "\n" + validationResult.getMessage();
      }
      Server.logSevereMessage(message, e);
      throw e;
    }
    catch (OperationFailureException e) {
      Server.logSevereMessage(
        "Ocorreu um erro ao salvar o comando no servio de persistncia", e);
      throw new ServiceFailureException(
        "Falha na submisso de comando. Ocorreu um erro ao salvar o comando no servio de persistncia"
          + submissionInfo,
        e);
    }
    catch (PermissionException pe) {
      Server
        .logWarningMessage("Ocorreu um erro de permisso. " + pe.getMessage());
      throw pe;
    }
    catch (AlgorithmConfigurationSerializerException e) {
      Server.logSevereMessage(
        "Falha na submisso do comando. Ocorreu um erro na criao do configurador a partir do comando.",
        e);
      throw new ServiceFailureException(
        "Falha na submisso do comando. \nOcorreu um erro na criao do configurador a partir do comando. \nVerifique o log para maiores detalhes.",
        e);
    }
    catch (IOException e) {
      Server.logSevereMessage(
        "Falha na submisso de comando. Ocorreu um erro de I/O na recuperao do configurador a partir do comando.",
        e);
      throw new ServiceFailureException(
        "Falha na submisso de comando.  \nOcorreu um erro de I/O na recuperao do configurador a partir do comando. \nVerifique o log para maiores detalhes.");
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Falha na submisso do comando: \n" + submissionInfo, e);
      throw new ServiceFailureException(
        "Falha na submisso do comando. \n" + submissionInfo, e);
    }
  }

  /**
   * Verifica se os parmetros configurados tm valor vlido.
   *
   * @param configurator o configurador.
   * @param projectId o projeto onde o comando ser executado.
   * @param userId o identificador do usurio que fez a submisso.
   *
   * @throws AlgorithmValidationException em caso de erro na validao dos
   *         parmetros.
   */
  private void checkParameterValues(AlgorithmConfigurator configurator,
    Object projectId, Object userId) throws AlgorithmValidationException {

    try {
      ValidationContext context =
        new ValidationContext(ValidationMode.FULL, projectId, userId);
      Validation result = configurator.validate(context);
      if (!result.isWellSucceded()) {
        throw new AlgorithmValidationException(result);
      }
    }
    catch (RemoteException e) {
      throw new AlgorithmValidationException(e);
    }
  }

  /**
   * @param submission
   * @param configurator
   */
  private void collectExeTypeStats(CommandSubmission submission,
    AlgorithmConfigurator configurator) {
    incrCounter(exeTypeCounter, submission.getExecutionType());
    if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      flowExecutionCounter++;
    }
  }

  /**
   * Coleta as estatsticas dos nomes dos algoritmos. Para fluxos, contabiliza
   * os algoritmos de cada um dos ns.
   *
   * @param configurator configurador dos algoritmos (pode ser um fluxo)
   */
  private void collectAlgoNameStats(AlgorithmConfigurator configurator) {
    ConfiguratorType configuratorType = configurator.getConfiguratorType();
    if (configuratorType == ConfiguratorType.FLOW) {
      /*
       * estamos em um fluxo, vamos coletar os nomes de cada um dos algoritmos
       * individualmente
       */
      FlowAlgorithmConfigurator fconf =
        (FlowAlgorithmConfigurator) configurator;
      Set<Node> nodes = fconf.getNodes();
      for (Node node : nodes) {
        String algoName = node.getConfigurator().getAlgorithmName();
        incrCounter(flowAlgoNameCounter, algoName);
      }
    }
    else {
      incrCounter(algoNameCounter, configurator.getAlgorithmName());
    }
  }

  /**
   * Criao de um identificador de comando nico.
   *
   * @param userId Identificador do usurio
   * @param abbrev Abreviao que identifica o algoritmo.
   * @param sysName Nome do sistema (WebSintesi, InfoGrid, por exemplo)
   * @return uma string que representa um nmero nico.
   */
  private String createCommandId(String userId, String abbrev, String sysName) {
    return String.format("%s@%s.%s%s",
      formatField(userId, USERID_DEFAULT, USERID_LENGTH, '0'),
      formatField(abbrev, ABBREV_DEFAULT, ABBREV_LENGTH, '0'),
      formatField(sysName, SYSNAME_DEFAULT, SYSNAME_LENGTH, '0'),
      commandIdGenerator.generateId());
  }

  /**
   * Formata o campo que ser usado no identificador
   *
   * @param field o campo que ser formatado
   * @param defaultValue usado no caso de field ser null
   * @param defaultLength Tamanho que ser usado para 'truncar' o campo
   * @param defaultChar Caracter que ser usado para completar o campo se ele
   *        for menor que defaultLength.
   * @return field
   */

  private String formatField(String field, String defaultValue,
    int defaultLength, char defaultChar) {
    StringBuilder result =
      new StringBuilder(field != null ? field : defaultValue);
    // Se o tamanho do campo for menor que defaultLength,
    // completa o campo com defaultChar.
    int length = result.length();
    if (length < defaultLength) {
      for (int i = defaultLength - length; i > 0; i--) {
        result.append(defaultChar);
      }
    }
    // Se o tamanho do campo for maior que defaultLength, trunca
    // o campo.
    else {
      result.setLength(defaultLength);
    }
    return result.toString();
  }

  /**
   * Verifica se o usurio tem permisso para execuo do algoritmo (se for um
   * algoritmo simples), ou verifica se o usurio possui permisso para executar
   * todos os algoritmos que fazem parte do configurador (no caso de ser um
   * fluxo).
   *
   * Se o usurio for administrador, ele tem permisso sempre.
   *
   * @param configurator configurador do algoritmo ou do fluxo
   *
   * @throws PermissionException usurio no tem permisso para executar um ou
   *         mais algoritmos que fazem parte do configurador (pode ser um fluxo)
   * @throws Exception qualquer outro tipo de erro ocorrido
   */
  private void checkExecutionAlgorithmPermission(
    AlgorithmConfigurator configurator) throws PermissionException, Exception {
    User user = Service.getUser();
    if (user.isAdmin()) {
      return;
    }

    String algorithmName = configurator.getAlgorithmName();
    CategorySet categorySet = AlgorithmService.getInstance().getAllCategories();
    List<String> categoriesFullNames =
      categorySet.getAlgorithmCategoriesFullNames(algorithmName);

    String systemId = Service.getSystemId();
    String errorMsgFormatter = "Sem permisso para executar{0} {1}: {2}";
    String systemMsg = (systemId == null) ? "" : " pelo sistema " + systemId;
    String algoMsg = "";
    String errorMsg = "";

    if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      // Mantm uma lista com os nomes dos algoritmos que o usurio no tem
      // permisso para execuo
      List<String> noPermissionAlgorithms = new Vector<String>();
      FlowAlgorithmConfigurator flowConfigurator =
        (FlowAlgorithmConfigurator) configurator;
      Set<Node> nodes = flowConfigurator.getNodes();
      for (Node node : nodes) {
        algorithmName = node.getConfigurator().getAlgorithmName();
        if (!AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
          Service.getUser(), systemId, algorithmName)) {
          // Verifica se tem alguma permisso para as categorias em que o
          // algoritmo faz parte
          categoriesFullNames =
            categorySet.getAlgorithmCategoriesFullNames(algorithmName);
          if (!CategoryAlgorithmsExecutionPermission
            .checkSystemAndCategoriesExecPermission(Service.getUser(), systemId,
              categoriesFullNames)) {
            noPermissionAlgorithms.add(algorithmName);
          }
        }
      }
      int nAlgo = noPermissionAlgorithms.size();

      if (noPermissionAlgorithms.isEmpty()) {
        return;
      }

      String algoErrorNames = "";
      for (String algoName : noPermissionAlgorithms) {
        algoErrorNames += "\n" + algoName;
      }

      algoMsg = (nAlgo == 1) ? "o algoritmo" : " os algoritmos";
      errorMsg = MessageFormat.format(errorMsgFormatter,
        new Object[] { systemMsg, algoMsg, algoErrorNames });
    }
    else {
      if (AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
        Service.getUser(), systemId, algorithmName)) {
        return;
      }
      // Verifica se tem alguma permisso para as categorias em que o
      // algoritmo faz parte
      if (CategoryAlgorithmsExecutionPermission
        .checkSystemAndCategoriesExecPermission(Service.getUser(), systemId,
          categoriesFullNames)) {
        return;
      }
      algoMsg = "o algoritmo";
      errorMsg = MessageFormat.format(errorMsgFormatter,
        new Object[] { systemMsg, algoMsg, "\n" + algorithmName });
    }
    throw new PermissionException(errorMsg);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setPosition(Object commandId, int position) {
    Server.logFineMessage(
      "Alterao da posicao do comando " + commandId + " para " + position);
    if (position < 0 || position >= commandQueue.size()) {
      return false;
    }
    return commandQueue.setPosition(commandId, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setPriority(Object commandId, Priority priority) {
    Server.logFineMessage(
      "Alterao da prioridade do comando " + commandId + " para " + priority);
    return commandQueue.setPriority(commandId, priority);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateCommandDescription(String commandId, String description) {
    for (CommandInfo scheduledCommand : commandQueue.getCommands()) {
      if (scheduledCommand.getId().equals(commandId)) {
        scheduledCommand.setDescription(description);
        return;
      }
    }
  }

  /**
   * Indica a falha na submisso de um comando.
   *
   * @param cmd o comando que falhou
   * @param cause a causa da falha no comando
   */
  protected void failCommand(CommandInfo cmd, FailureFinalizationType cause) {
    // Atribui o status de morto
    cmd.setFinished(
      new SimpleCommandFinalizationInfo(CommandFinalizationType.FAILED, cause));
    CommandPersistenceService service = CommandPersistenceService.getInstance();
    if (service != null && cause != FailureFinalizationType.PROJECT_NOT_FOUND) {
      try {
        service.saveCommandInfo(cmd);
      }
      catch (OperationFailureException e) {
        Server.logSevereMessage("Ocorreu um erro ao salvar o comando "
          + cmd.getId() + "com o status FAILED no servio de persistncia", e);
      }
    }
    notifyCommandFailed(cmd, cause);
  }

  /**
   * Notificao de falha de comando para um usurio e para os observadores
   * registrados no servio
   *
   * @param command comando.
   * @param cause a causa da falha no comando.
   */
  private void notifyCommandFailed(CommandInfo command,
    FailureFinalizationType cause) {
    /* Envia notificao para os usurios */
    Object userId = null;
    try {
      userId = command.getUserId();
      final CommandFailedNotification notData =
        new CommandFailedNotification(getSenderName(), command.getId(),
          command.getDescription(), command.getTip(),
          command.getSubmittedDate().getTime(), null, command.getProjectId(),
          new SimpleCommandFinalizationInfo(CommandFinalizationType.FAILED,
            cause));
      messageService.send(new Message(notData), (String) userId);
      messageService.sendToServer(new Message(notData));
    }
    catch (Exception e) {
      Server.logSevereMessage("Impossvel notificar o usurio (" + userId + ") "
        + "sobre a falha do comando (" + command.getId() + ")", e);
    }
  }

  /**
   * Notifica que um comando foi interrompido.
   *
   * @param command
   */
  private void notifyCommandKilled(CommandInfo command) {
    final CommandKilledNotification notData = new CommandKilledNotification(
      getSenderName(), command.getId(), command.getDescription(),
      command.getTip(), command.getSubmittedDate().getTime(), null,
      command.getProjectId(), command.getFinalizationInfo());

    messageService.sendToServer(new Message(notData));

    /* Envia notificao para os usurios */
    Object userId = null;
    try {
      userId = command.getUserId();
      messageService.send(new Message(notData), (String) userId);
    }
    catch (Exception e) {
      Server.logSevereMessage("Impossvel notificar o usurio (" + userId + ") "
        + "sobre a remoo do comando (" + command.getId() + ")", e);
    }

    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(command.getProjectId());
      messageService.send(new Message(notData), users);
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao notificar os usurios do projeto " + command.getProjectId()
          + " o fim da execuo do comando (" + command.getId() + ")",
        e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeCommand(Object commandId) {
    Server.logFineMessage("Remoo do comando : " + commandId);
    try {
      // Remove o comando da fila
      CommandInfo cmd = commandQueue.remove(commandId);
      if (null == cmd) {
        return false;
      }
      ProjectService projectService = ProjectService.getInstance();
      Object projectId = cmd.getProjectId();
      Object commandOwnerId = cmd.getUserId();
      Object projectOwnerId = projectService.getOwnerId(projectId);
      User loggedUser = Service.getUser();
      Object loggedUserId = loggedUser.getId();
      if (!loggedUser.isAdmin() && !loggedUserId.equals(commandOwnerId)
        && !loggedUserId.equals(projectOwnerId)) {
        throw new PermissionException();
      }
      // Atribui o status de morto
      cmd.setFinished(
        new SimpleCommandFinalizationInfo(CommandFinalizationType.KILLED,
          false));
      CommandPersistenceService service =
        CommandPersistenceService.getInstance();
      if (service != null) {
        service.saveCommandInfo(cmd);
      }
      notifyCommandKilled(cmd);

      /**
       * Alm dos observadores, aqui deveria notificar os usurios do projeto,
       * de que o comando foi terminado. <br>
       * A exemplo do SGAService#notifyCommandEnd(Command,CommandEvent)<br>
       * <code>
       * notifyCommandEndToProject(command, event);
       * mailCommandEndToProject(command, event);
       * </code>
       */
      return true;
    }
    catch (OperationFailureException e) {
      Server.logSevereMessage(
        "Ocorreu um erro ao salvar o comando com o status KILLED no servio de persistncia",
        e);
      throw new ServiceFailureException(
        "Falha na remoo de comando. Ocorreu um erro ao salvar o comando com o status KILLED no servio de persistncia",
        e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setBlocked(boolean blocked) {
    Server.logFineMessage("Bloqueio da fila chamado: " + blocked);
    commandQueue.setBlocked(blocked);
    messageService
      .sendToAll(new Message(new SchedulerStateChangedEvent(blocked)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isBlocked() {
    return commandQueue.isBlocked();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandInfo[] getQueuedCommands() {
    Object userId = Service.getUser().getId();
    boolean isAdmin = Service.getUser().isAdmin();
    CommandInfo cmds[] = commandQueue.getCommands();
    ArrayList<CommandInfo> commandsArray = new ArrayList<CommandInfo>();
    for (int i = 0; i < cmds.length; i++) {
      if (isAdmin || cmds[i].getUserId().equals(userId)) {
        commandsArray.add(cmds[i]);
        cmds[i].setGlobalPosition(i + 1);
      }
    }
    return commandsArray.toArray(new CommandInfo[0]);
  }

  /**
   * Constri a instncia do servio de escalonamento.
   *
   * @throws ServerException se houver erro na inicializao.
   */
  public static void createService() throws ServerException {
    new SchedulerService();
  }

  /**
   * * {@inheritDoc}
   */
  @Override
  protected Service[] getInitializationDependencies() {
    return new Service[] { AlgorithmService.getInstance(),
        CommandPersistenceService.getInstance(), MessageService.getInstance(),
        ProjectService.getInstance(), SGAService.getInstance() };
  }

  /**
   * Retorno da instncia do servio.
   *
   * @return o servio.
   */
  final public static SchedulerService getInstance() {
    return (SchedulerService) getInstance(SERVICE_NAME);
  }

  /**
   * Inicializao do servio de escalonamento.
   *
   * @throws ServerException caso ocorra um erro na inicializao
   */
  @Override
  final public void initService() throws ServerException {
    SGAService sgaService = SGAService.getInstance();
    messageService = MessageService.getInstance();
    long queueProcessingInterval;
    int sgaQuarantine;
    String schedPolicyName;
    boolean resourcesControl;

    /*
     * Determina quem  o gerador de identificadores de comandos.
     */
    String converterClassName =
      getStringProperty(COMMAND_ID_GENERATOR_PROPERTY);
    try {
      Class<?> commandIdGeneratorClass = Class.forName(converterClassName);
      commandIdGenerator =
        (CommandIdGeneratorInterface) commandIdGeneratorClass.newInstance();
    }
    catch (Exception e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao do gerador de identificadores de comandos.",
        e));
    }

    /*
     * Faz a leitura do intervalo de processamento da fila. Quando no 
     * definido no arquivo de propriedades, fixamos em 2 vezes o intervalo de
     * atualizao das informaes do SGA.
     */
    if (isPropertyNull(PROCESSING_INTERVAL_PROPERTY)) {
      queueProcessingInterval = sgaService.getUpdateInterval() * 1000 * 2;
    }
    else {
      queueProcessingInterval = getIntProperty(PROCESSING_INTERVAL_PROPERTY);
    }
    Server.logInfoMessage("Intervalo de processamento da fila de comandos: "
      + queueProcessingInterval + " milisegundo(s).");

    /*
     * Faz a leitura do nmero de vezes que o scheduler tentar gerar um novo
     * comando. Quando no  definido no arquivo de propriedades, fixamos em 1
     * vez.
     */
    if (isPropertyNull(COMMAND_ID_RETRIES_PROPERTY)) {
      commandIdRetries = 1;
    }
    else {
      commandIdRetries = getIntProperty(COMMAND_ID_RETRIES_PROPERTY);
    }
    Server.logInfoMessage("Intervalo de processamento da fila de comandos: "
      + queueProcessingInterval + " milisegundo(s).");

    try {
      /*
       * Obtm o tempo de quarentena que o SGA permanece depois que um comando 
       * enviado para ele.
       */
      sgaQuarantine = getIntProperty(QUARANTINE_PROPERTY);
      Server.logInfoMessage(
        "Intervalo de quarentena dos SGAs: " + sgaQuarantine + " segundo(s).");
    }
    catch (Exception e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao do tempo de quarentena do SGA.", e));
    }

    /*
     * Estabelece a poltica de escalonamento do SchedulerService.
     */
    try {
      schedPolicyName = getStringProperty(SCHED_POLICY_PROPERTY);
      Server.logInfoMessage(
        "Poltica de escalonamento a ser ativada: " + schedPolicyName + ".");
    }
    catch (Exception e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao da poltica de escalonamento.", e));
    }

    /*
     * Estabelece o caminho para o arquivo de backup.
     */
    String backupFilename;
    try {
      backupFilename = sgaService.getCommandsPersistencyDirectoryName()
        + File.separator + BACKUP_FILE_NAME;
    }
    catch (OperationFailureException e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao do arquivo para backup.", e));
    }
    Server.logInfoMessage("Arquivo de backup: " + backupFilename);

    /*
     * Cria a fila e carrega o seu contedo do arquivo de backup.  comum
     * ocorrer erro na leitura antes que o primeiro backup seja feito, por isso
     * o log apenas como warning.
     */
    commandQueue = new PriorityQueue(backupFilename);
    File backupFile = new File(backupFilename);
    if (backupFile.exists()) {
      commandQueue.loadFromBackup();
    }
    else {
      Server.logWarningMessage("Arquivo de backup no encontrado."
        + "Fila no recuperada na inicializao do servidor.");
    }

    /*
     * Estabelece se deve fazer o controle de recursos de cada sga.
     */
    String resourcesBackupFilename = null;
    try {
      resourcesControl = getBooleanProperty(RESOURCES_CONTROL_PROPERTY);
      if (resourcesControl) {
        Server.logInfoMessage("Controle de recursos ativo.");

        /*
         * Estabelece o caminho para o arquivo de backup de recursos.
         */
        resourcesBackupFilename =
          sgaService.getCommandsPersistencyDirectoryName() + File.separator
            + RESOURCES_BACKUP_FILE_NAME;
        Server.logInfoMessage("Arquivo de backup: " + resourcesBackupFilename);
      }
      else {
        Server.logInfoMessage("Controle de recursos inativo.");
      }
    }
    catch (OperationFailureException e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao do arquivo para backup.", e));
    }
    catch (Exception e) {
      throw new ServerException(new InitFailureException(
        "Falha na inicializao do controle de recursos de cada SGA.", e));
    }
    scheduler =
      new Scheduler(commandQueue, queueProcessingInterval, sgaQuarantine,
        schedPolicyName, resourcesControl, resourcesBackupFilename);
    scheduler.startScheduler();
  }

  /**
   * Mtodo invocado no trmino do servio.
   */
  @Override
  public void shutdownService() {
    if (scheduler != null) {
      scheduler.stopScheduler();
    }
    Server.logInfoMessage("Servio encerrado.");
  }

  /**
   * Mtodo para indicar se o observador deve ser notificado.
   *
   * @param arg Argumento do observador.
   * @param event Evento gerado para notificar o observador.
   *
   * @return <code>true</code> quando o observador deve ser notificado.
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<ExecutionType, Integer> getExeTypeStats() throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return Collections.unmodifiableMap(exeTypeCounter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Integer> getAlgoStats(boolean flowAlgorithms)
    throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return Collections
      .unmodifiableMap(flowAlgorithms ? flowAlgoNameCounter : algoNameCounter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getFlowExecutionStats() throws RemoteException {
    return flowExecutionCounter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Integer> getUserStats() throws RemoteException {
    return Collections.unmodifiableMap(userExeCounter);
  }
}
