/*
 * $Id: ProjectDataService.java 128667 2012-05-03 20:40:27Z mjulia $
 */
package csbase.server.services.projectservice.v1_02;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.omg.CORBA.Any;
import org.omg.CORBA.ORB;

import csbase.exception.ServiceFailureException;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.logic.filters.ProjectFileNameFilter;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.projectservice.ServerProject;
import csbase.server.services.projectservice.UpdatableFileInfo;
import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.MaxClientsReachedException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.ftc.utils.Utils;
import tecgraf.openbus.data_service.core.v1_02.AbsentViews;
import tecgraf.openbus.data_service.core.v1_02.DataAccessDenied;
import tecgraf.openbus.data_service.core.v1_02.DataAlreadyExist;
import tecgraf.openbus.data_service.core.v1_02.DataDescription;
import tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper;
import tecgraf.openbus.data_service.core.v1_02.DataNotFound;
import tecgraf.openbus.data_service.core.v1_02.DataView;
import tecgraf.openbus.data_service.core.v1_02.DefaultView;
import tecgraf.openbus.data_service.core.v1_02.IDataService;
import tecgraf.openbus.data_service.core.v1_02.IDataServiceOperations;
import tecgraf.openbus.data_service.core.v1_02.InvalidDataKey;
import tecgraf.openbus.data_service.core.v1_02.Metadata;
import tecgraf.openbus.data_service.core.v1_02.ServiceFailure;
import tecgraf.openbus.data_service.core.v1_02.UnavailableDataService;
import tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView;
import tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewHelper;
import tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewImpl;
import tecgraf.openbus.data_service.core.v1_02.UnsupportedView;
import tecgraf.openbus.data_service.hierarchical.v1_02.HierarchicalNodeDataView;
import tecgraf.openbus.data_service.hierarchical
  .v1_02.HierarchicalNodeDataViewHelper;
import tecgraf.openbus.data_service.hierarchical
  .v1_02.HierarchicalNodeDataViewImpl;
import tecgraf.openbus.data_service.hierarchical
  .v1_02.IHierarchicalManagementDataServiceOperations;
import tecgraf.openbus.data_service.hierarchical
  .v1_02.IHierarchicalTransferDataServiceOperations;
import tecgraf.openbus.data_service.hierarchical.v1_02.InvalidContainer;
import tecgraf.openbus.data_service.hierarchical.v1_02.InvalidPrototype;
import tecgraf.openbus.data_service.hierarchical.v1_02.UnsupportedOperation;
import tecgraf.openbus.data_service.project
  .v1_02.IProjectNavigationDataServiceOperations;
import tecgraf.openbus.data_service.project.v1_02.InvalidOwner;
import tecgraf.openbus.data_service.project.v1_02.PATH_SEPARATOR;
import tecgraf.openbus.data_service.project.v1_02.ProjectDataView;
import tecgraf.openbus.data_service.project.v1_02.ProjectDataViewFactory;
import tecgraf.openbus.data_service.project.v1_02.ProjectDataViewHelper;
import tecgraf.openbus.data_service.project.v1_02.ProjectDataViewImpl;
import tecgraf.openbus.data_service.project.v1_02.ProjectItemDataView;
import tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewFactory;
import tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewHelper;
import tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewImpl;

/**
 * Representa um Servio de Dados de Projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectDataService implements
  IHierarchicalManagementDataServiceOperations,
  IHierarchicalTransferDataServiceOperations,
  IProjectNavigationDataServiceOperations, IDataServiceOperations {

  /**
   * Array vazio de descritores de dados. Utilizado principalmente para retorno.
   */
  private static final DataDescription[] EMPTY_DATA_DESCRIPTION_ARRAY =
    new DataDescription[0];
  /**
   * O nome do metadado que representa o caminho absoluto de um dado.
   */
  private static final String ABSOLUTE_PATH_METADATUM_NAME = "ABSOLUTE_PATH";
  /**
   * O valor do campo SourceId na gerao das chaves
   */
  protected String SourceId = null;

  /**
   * A instncia nica do Servio de Dados de Projeto.
   */
  private static ProjectDataService instance;

  /**
   * Cria um Servico de Dados de Projeto.
   */
  private ProjectDataService() {
    SourceId = ProjectService.getInstance().getSourceId();
    org.omg.CORBA_2_3.ORB orb =
      (org.omg.CORBA_2_3.ORB) OpenBusService.getInstance().getORB();
    orb.register_value_factory(ProjectDataViewHelper.id(),
      new ProjectDataViewFactory());
    orb.register_value_factory(ProjectItemDataViewHelper.id(),
      new ProjectItemDataViewFactory());
  }

  /**
   * Obtm a instncia nica do Servio de Dados de Projeto.
   * 
   * @return A instncia nica do Servio de Dados de Projeto.
   */
  public static ProjectDataService getInstance() {
    if (instance == null) {
      instance = new ProjectDataService();
    }
    return instance;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription[] getRoots() throws ServiceFailure, DataAccessDenied {
    Object userId = Service.getUser().getId();
    ProjectService projectService = ProjectService.getInstance();
    List<UserProjectInfo> projects = new ArrayList<UserProjectInfo>();
    try {
      projects.addAll(projectService.getProjectsFromUser(userId));
      projects.addAll(projectService.getProjectsSharedWithUser(userId));
    }
    catch (ServiceFailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao obter as razes dos projetos: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    if (projects.size() == 0) {
      return EMPTY_DATA_DESCRIPTION_ARRAY;
    }
    List<DataDescription> roots =
      new ArrayList<DataDescription>(projects.size());
    for (UserProjectInfo projectInfo : projects) {
      String projectName = projectInfo.getProjectName();
      try {
        CommonClientProject project =
          projectService.openProject(projectInfo.getProjectId(), false);
        ClientProjectFile rootFile = project.getRoot();
        try {
          roots.add(this.createDataDescription(rootFile));
        }
        catch (InvalidDataKey e) {
          String userLogin = Service.getUser().getLogin();
          String msg =
            MessageFormat
              .format(
                "Ocorreu um erro na solicitao do usurio {0}: erro ao criar descrio da raz do projeto {1}: {2}",
                new Object[] { userLogin, projectName, e.getMessage() });
          Server.logSevereMessage(msg, e);
          throw new ServiceFailure(msg);
        }
      }
      catch (ServiceFailureException e) {
        String userLogin = Service.getUser().getLogin();
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: erro ao abrir o projeto {1} do usurio {2}: {3}",
              new Object[] { userLogin, projectName, projectInfo.getOwnerId(),
                  e.getMessage() });
        Server.logSevereMessage(msg, e);
        throw new ServiceFailure(msg);
      }
    }

    return roots.toArray(EMPTY_DATA_DESCRIPTION_ARRAY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription[] getChildren(byte[] fKey) throws ServiceFailure,
    DataAccessDenied, InvalidDataKey, DataNotFound {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    ClientProjectFile parent = getProjectFile(dataKey);
    if (!parent.isDirectory()) {
      return EMPTY_DATA_DESCRIPTION_ARRAY;
    }

    ClientProjectFile[] children = null;
    try {
      children = parent.getChildren();
    }
    catch (RemoteException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(parent.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao obter os fihos do dado {1} no projeto {2}: {3}",
            new Object[] { userLogin, parent.getName(), projectName,
                e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    if (children == null) {
      return EMPTY_DATA_DESCRIPTION_ARRAY;
    }

    List<DataDescription> nodes =
      new ArrayList<DataDescription>(children.length);
    for (int i = 0; i < children.length; i++) {
      nodes.add(this.createDataDescription(children[i]));
    }
    return nodes.toArray(EMPTY_DATA_DESCRIPTION_ARRAY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription getParent(byte[] fKey) throws ServiceFailure,
    DataAccessDenied, InvalidDataKey, DataNotFound {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    ClientProjectFile file = getProjectFile(dataKey);
    if (file == null) {
      throw new DataNotFound("Dado no encontrado.", new byte[][] { fKey });
    }
    ClientProjectFile parent = file.getParent();
    if (parent == null) {
      return createEmptyDataDescription();
    }
    return this.createDataDescription(parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectDataView[] getProject(String fOwner) throws DataAccessDenied,
    ServiceFailure, InvalidOwner {
    Object owner;
    try {
      owner = User.getUserByLogin(fOwner).getId();
    }
    catch (Exception e) {
      throw new InvalidOwner("Erro ao obter informaes do usurio dono.");
    }

    if (owner == null) {
      throw new InvalidOwner("Usurio dono no existe.");
    }

    ProjectService projectService = ProjectService.getInstance();

    Object userId = Service.getUser().getId();

    List<UserProjectInfo> projects = new ArrayList<UserProjectInfo>();

    if (userId.equals(owner)) {
      projects.addAll(projectService.getProjectsFromUser(userId));
    }
    else {
      projects.addAll(projectService.getProjectsSharedWithUser(userId));
    }

    List<ProjectDataView> projectsView = new ArrayList<ProjectDataView>();

    for (UserProjectInfo prjInfo : projects) {
      if (prjInfo.getOwnerId().equals(fOwner)) {
        try {
          projectsView.add(createProjectDataView(prjInfo.getProjectId()));
        }
        catch (InvalidDataKey e) {
          String userLogin = Service.getUser().getLogin();
          String projectName =
            ServerProject.getProjectName(prjInfo.getProjectId());
          String msg =
            MessageFormat
              .format(
                "Ocorreu um erro na solicitao do usurio {0}: erro ao criar a viso ProjectDataView do projeto {1}: {2}",
                new Object[] { userLogin, projectName, e.getMessage() });
          Server.logSevereMessage(msg, e);
          throw new ServiceFailure(msg);
        }
      }
    }
    return projectsView.toArray(new ProjectDataView[0]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription getDataDescription(byte[] fKey) throws ServiceFailure,
    InvalidDataKey, DataNotFound, DataAccessDenied {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    ClientProjectFile file = getProjectFile(dataKey);
    return this.createDataDescription(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataView getDataView(byte[] fKey, String fViewInterface)
    throws ServiceFailure, InvalidDataKey, DataNotFound, UnsupportedView,
    DataAccessDenied {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    ClientProjectFile file = getProjectFile(dataKey);

    if (file == null) {
      throw new DataNotFound("Dado no encontrado.", new byte[][] { dataKey
        .getKey() });
    }
    //    try {
    return this.createDataView(file, fViewInterface);
    //    }
    //    catch (UnsupportedView e) {
    //      e.fMessage = ("Viso " + fViewInterface + " no suportada.");
    //      e.fKeys = new byte[][] { dataKey.getKey() };
    //      throw e;
    //    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataView[] getDataViewSeq(byte[][] fKeys, String fViewInterface)
    throws ServiceFailure, InvalidDataKey, DataNotFound, UnsupportedView,
    DataAccessDenied {
    List<DataView> viewList = new LinkedList<DataView>();

    List<byte[]> InvalidDataKeyList = new LinkedList<byte[]>();
    List<byte[]> DataNotFoundList = new LinkedList<byte[]>();
    List<byte[]> UnsupportedViewList = new LinkedList<byte[]>();
    List<byte[]> DataAccessDeniedList = new LinkedList<byte[]>();

    for (byte[] key : fKeys) {
      try {
        viewList.add(this.getDataView(key, fViewInterface));
      }
      catch (InvalidDataKey e) {
        for (byte[] k : e.fKeys) {
          InvalidDataKeyList.add(k);
        }
      }
      catch (DataNotFound e) {
        for (byte[] k : e.fKeys) {
          DataNotFoundList.add(k);
        }
      }
      catch (UnsupportedView e) {
        for (byte[] k : e.fKeys) {
          UnsupportedViewList.add(k);
        }
      }
      catch (DataAccessDenied e) {
        for (byte[] k : e.fKeys) {
          DataAccessDeniedList.add(k);
        }
      }
    }

    if (!InvalidDataKeyList.isEmpty()) {
      throw new InvalidDataKey("Chaves dos dados so invlidas.",
        InvalidDataKeyList.toArray(new byte[0][]));
    }
    else if (!DataNotFoundList.isEmpty()) {
      throw new DataNotFound("Dados no encontrados.", DataNotFoundList
        .toArray(new byte[0][]));
    }
    else if (!UnsupportedViewList.isEmpty()) {
      throw new UnsupportedView("Viso no suportada pelos dados.",
        UnsupportedViewList.toArray(new byte[0][]));
    }
    else if (!DataAccessDeniedList.isEmpty()) {
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso aos dados.",
        DataAccessDeniedList.toArray(new byte[0][]));
    }

    return viewList.toArray(new DataView[0]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] createData(DataDescription fPrototype, byte[] fParentKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    DataAlreadyExist, InvalidPrototype, InvalidContainer, UnsupportedOperation {
    DataKeyWrapper parentKey = new DataKeyWrapper(fParentKey);
    DataValidation.checkDataKey(parentKey);
    ClientProjectFile parent = getProjectFile(parentKey);
    if (parent == null) {
      throw new DataNotFound("Dado no encontrado.", new byte[][] { parentKey
        .getKey() });
    }
    if (!parent.isDirectory()) {
      String msg =
        MessageFormat.format("Dado no  um diretrio.", new Object[] { parent
          .getName() });
      throw new InvalidContainer(msg, new byte[][] { fParentKey });
    }

    if ((fPrototype.fName == null) || (fPrototype.fName.length() == 0)) {
      throw new InvalidPrototype(
        "O prottipo no tem o campo fName preenchido. Na criao de dado este  um campo obrigatrio.");
    }

    if (fPrototype.fDefaultView == null) {
      throw new InvalidPrototype(
        "O prottipo no tem viso padro. Na criao de dado este  um campo obrigatrio.");
    }

    if (!fPrototype.fDefaultView.fInterfaceName
      .equals(ProjectItemDataViewHelper.id())) {
      throw new InvalidPrototype(
        "O prottipo no tem a viso padro do tipo ProjectItemDataView. Na criao de dado esta deve ser a viso padro.");
    }

    ProjectItemDataView prototypeProjItemDataView =
      ProjectItemDataView.class.cast(fPrototype.fDefaultView.fValue);

    try {
      if (parent.getChild(fPrototype.fName) != null) {
        String msg =
          MessageFormat.format("Dado {0} j existe no diretrio {1}.",
            new Object[] { fPrototype.fName, parent.getName() });
        throw new DataAlreadyExist(msg);
      }

      if (prototypeProjItemDataView.fIsContainer) {
        parent.createFile(fPrototype.fName, ProjectFileType.DIRECTORY_TYPE);
      }
      else {
        parent.createFile(fPrototype.fName, prototypeProjItemDataView.fType);
      }
    }
    catch (RemoteException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(parent.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao criar o dado {1} no projeto {2}: {3}",
            new Object[] { userLogin, fPrototype.fName, projectName,
                e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (ServiceFailureException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(parent.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao criar o dado {1} no projeto {2}: {3}",
            new Object[] { userLogin, fPrototype.fName, projectName,
                e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    String dataId = generateDataId(parent, fPrototype.fName);
    DataKeyWrapper key1 = this.generateDataKey(dataId);
    DataKeyWrapper key = key1;
    return key.getKey();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] copyData(byte[] fSourceKey, byte[] fParentKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    DataAlreadyExist, InvalidContainer, UnsupportedOperation {
    DataKeyWrapper parentKey = new DataKeyWrapper(fParentKey);
    DataValidation.checkDataKey(parentKey);
    ClientProjectFile parent = getProjectFile(parentKey);
    DataKeyWrapper sourceKey = new DataKeyWrapper(fSourceKey);
    DataValidation.checkDataKey(sourceKey);
    ClientProjectFile source = getProjectFile(sourceKey);

    if (!parent.isDirectory()) {
      String msg =
        MessageFormat.format("Dado no  um diretrio.", new Object[] { parent
          .getName() });
      throw new InvalidContainer(msg, new byte[][] { fParentKey });
    }

    try {
      if (parent.getChild(source.getName()) == null) {
        source.copy(parent);
      }
      else {
        String msg =
          MessageFormat.format("Dado {0} j existe no diretrio {1}.",
            new Object[] { source.getName(), parent.getName() });
        throw new DataAlreadyExist(msg);
      }
    }
    catch (RemoteException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(parent.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado {1} para o diretrio {2} dentro do projeto {3}: {4}",
            new Object[] { userLogin, source.getName(), parent.getName(),
                projectName, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (csbase.exception.PermissionException e) {
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso ao dado.",
        new byte[][] { fParentKey });
    }

    String dataId = generateDataId(parent, source.getName());
    DataKeyWrapper dataKey = this.generateDataKey(dataId);
    return dataKey.getKey();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void moveData(byte[] fKey, byte[] fNewParentKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    DataAlreadyExist, InvalidContainer, UnsupportedOperation {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    DataKeyWrapper newParentKey = new DataKeyWrapper(fNewParentKey);
    DataValidation.checkDataKey(newParentKey);
    ClientProjectFile file = getProjectFile(dataKey);
    ClientProjectFile newParent = getProjectFile(newParentKey);

    if (!newParent.isDirectory()) {
      String msg =
        MessageFormat.format("Dado no  um diretrio.",
          new Object[] { newParent.getName() });
      throw new InvalidContainer(msg, new byte[][] { fNewParentKey });
    }

    try {
      if (newParent.getChild(file.getName()) == null) {
        file.move(newParent);
      }
      else {
        String msg =
          MessageFormat.format("Dado {0} j existe no diretrio {1}.",
            new Object[] { file.getName(), newParent.getName() });
        throw new DataAlreadyExist(msg);
      }
    }
    catch (RemoteException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName =
        ServerProject.getProjectName(newParent.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao mover o dado {1} para o diretrio {2} dentro do projeto {3}: {4}",
            new Object[] { userLogin, file.getName(), newParent.getName(),
                projectName, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (csbase.exception.PermissionException e) {
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso ao dado.",
        new byte[][] { fNewParentKey });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateData(byte[] fKey, byte[] fSourceKey) throws ServiceFailure,
    DataAccessDenied, InvalidDataKey, DataNotFound, AbsentViews {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    DataKeyWrapper sourceKey = new DataKeyWrapper(fSourceKey);
    DataValidation.checkDataKey(sourceKey);
    ClientProjectFile destFile = getProjectFile(dataKey);
    ClientProjectFile sourceFile = getProjectFile(sourceKey);

    //Origem e destino iguais no h nada a fazer
    if (sourceFile.equals(destFile)) {
      return;
    }

    Object projectId = destFile.getProjectId();
    String[] path = destFile.getPath();
    try (OutputStream out = new BufferedOutputStream(
      ProjectService.getInstance().getOutputStream(projectId, path))) {
      sourceFile.download(out, 8 * 1024, null);
    }
    catch (FileLockedException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(destFile.getProjectId());
      String msg = MessageFormat.format(
        "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o " +
          "dado {1} dentro do projeto {2}: o dado est bloqueado: {3}",
        new Object[] { userLogin, destFile.getName(), projectName, e
          .getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (IOException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(destFile.getProjectId());
      String msg = MessageFormat.format(
        "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o " +
          "dado {1} dentro do projeto {2}: {3}",
        new Object[] { userLogin, destFile.getName(), projectName, e
          .getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (csbase.exception.PermissionException e) {
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso ao dado.",
        new byte[][] { fKey });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteData(byte[] fKey) throws ServiceFailure, DataAccessDenied,
    InvalidDataKey, DataNotFound, UnsupportedOperation {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    ClientProjectFile file = getProjectFile(dataKey);

    if (file.getParent() == null) {
      ProjectService.getInstance().removeProject(file.getProjectId());
    }
    else {
      try {
        file.remove();
      }
      catch (RemoteException e) {
        String userLogin = Service.getUser().getLogin();
        String projectName = ServerProject.getProjectName(file.getProjectId());
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: error ao remover o dado {1} dentro do projeto {2}: {3}",
              new Object[] { userLogin, file.getName(), projectName,
                  e.getMessage() });
        Server.logSevereMessage(msg, e);
        throw new ServiceFailure(msg);
      }
      catch (csbase.exception.PermissionException e) {
        throw new DataAccessDenied(
          "Usurio solicitante no tem permisso de acesso ao dado.",
          new byte[][] { fKey });
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] copyDataFrom(byte[] fSourceKey, byte[] fParentKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    AbsentViews, UnavailableDataService, DataAlreadyExist, InvalidContainer,
    UnsupportedOperation {
    DataKeyWrapper parentKey = new DataKeyWrapper(fParentKey);
    DataValidation.checkDataKey(parentKey);

    DataKeyWrapper sourceDataKey = new DataKeyWrapper(fSourceKey);
    try {
      DataValidation.checkDataKey(sourceDataKey);
      //Se passou da validao, significa que o dado de origem  local, ento 
      //deve ser feita uma cpia local.
      return copyData(fSourceKey, fParentKey);
    }
    catch (DataNotFound dnf) {
      //Se o dado de origem no  local, ento  remoto, deve seguir a lgica
      //de cpia remota mesmo.
    }

    OpenBusService.getInstance().joinChain();

    try {
      DataDescription parentDescription = this.getDataDescription(fParentKey);
      String parentAbsolutePath = null;
      for (int i = 0; i < parentDescription.fMetadata.length; i++) {
        if (parentDescription.fMetadata[i].fName
          .equals(ABSOLUTE_PATH_METADATUM_NAME)) {
          Any any = parentDescription.fMetadata[i].fValue;
          parentAbsolutePath = any.extract_string();
        }
      }

      IDataService sourceDataService = FindService.find(sourceDataKey);
      if (sourceDataService == null) {
        String userLogin = Service.getUser().getLogin();
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: servio de dados da origem no disponvel: {1} {2}",
              new Object[] { userLogin, sourceDataKey.getSystemDeploymentId(),
                  sourceDataKey.getDataSourceId() });
        Server.logWarningMessage(msg);
        throw new UnavailableDataService(msg);
      }

      DataDescription sourceDataDescription =
        sourceDataService.getDataDescription(fSourceKey);

      ProjectItemDataView sourceProjItemDataView =
        ProjectItemDataView.class.cast(sourceDataService.getDataView(
          fSourceKey, ProjectItemDataViewHelper.id()));

      String absolutePath =
        parentAbsolutePath + PATH_SEPARATOR.value + sourceDataDescription.fName;
      Metadata[] metadata = createMetadataList(absolutePath);
      long date = System.currentTimeMillis();

      String userLogin = Service.getUser().getLogin();

      ProjectItemDataView prototypeProjItemDataView =
        new ProjectItemDataViewImpl(null, userLogin,
          sourceProjItemDataView.fDescription, absolutePath,
          sourceProjItemDataView.fType, 0, sourceProjItemDataView.fIsContainer,
          sourceProjItemDataView.fCanRead, sourceProjItemDataView.fCanWrite,
          date, date);

      DefaultView defaultView =
        new DefaultView(ProjectItemDataViewHelper.id(),
          prototypeProjItemDataView);

      DataDescription prototype =
        new DataDescription(null, sourceDataDescription.fName, defaultView,
          null, metadata);

      byte[] key = this.createData(prototype, fParentKey);

      this.updateDataFrom(key, fSourceKey);

      return key;
    }
    catch (ServiceFailure e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: {1}",
            new Object[] { userLogin, e.fMessage });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (InvalidPrototype e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: prottipo invlido: {1}",
            new Object[] { userLogin, e.fMessage });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (UnsupportedView e) {
      throw new AbsentViews("Viso no suportada pelo dado.", e.fKeys,
        new String[] { ProjectItemDataViewHelper.id() });
    }
    finally {
      OpenBusService.getInstance().exitChain();
    }
  }

  /**
   * Cria um conjunto com as vises suportadas pelos dados oferecidos por este
   * servio.
   * 
   * @param container Indica se o dado  um continer.
   * @param root Indica se o dado  a raiz da hierarquia (um projeto)
   * 
   * @return O conjunto de vises.
   */
  private static String[] createViewSet(boolean container, boolean root) {
    Set<String> views = new HashSet<String>();
    if (!container) {
      views.add(UnstructuredDataViewHelper.id());
    }

    if (root) {
      views.add(ProjectItemDataViewHelper.id());
    }

    views.add(HierarchicalNodeDataViewHelper.id());

    return views.toArray(new String[0]);
  }

  /**
   * Cria uma lista de metadados para os dados oferecidos por este servio.
   * 
   * @param absolutePath O caminho absoluto do dado.
   * 
   * @return A lista de metadados.
   */
  private static Metadata[] createMetadataList(String absolutePath) {
    List<Metadata> metadataList = new ArrayList<Metadata>();

    ORB orb = OpenBusService.getInstance().getORB();
    Any any;

    if (absolutePath != null) {
      any = orb.create_any();
      any.insert_string(absolutePath);
      metadataList.add(new Metadata(ABSOLUTE_PATH_METADATUM_NAME, any));
    }

    return metadataList.toArray(new Metadata[0]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateDataFrom(byte[] fKey, byte[] fSourceKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    UnavailableDataService, AbsentViews {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);
    DataKeyWrapper sourceDataKey = new DataKeyWrapper(fSourceKey);
    ClientProjectFile file = getProjectFile(dataKey);

    OpenBusService.getInstance().joinChain();

    //Origem e destino iguais no h nada a fazer
    if (dataKey.getSystemDeploymentId().equals(
      sourceDataKey.getSystemDeploymentId())) {
      if (dataKey.getDataSourceId().equals(sourceDataKey.getDataSourceId())) {
        if (dataKey.getDataId().equals(sourceDataKey.getDataId())) {
          return;
        }
      }
    }
    try {
      DataValidation.checkDataKey(sourceDataKey);
      //Se a origem  local, deve-se fazer apenas uma cpia local.
      updateData(fKey, fSourceKey);
      return;
    }
    catch (DataNotFound dnf) {
      //Se o arquivo de origem no foi identificado como local,  remoto.
    }

    IDataService sourceDataService = FindService.find(sourceDataKey);
    if (sourceDataService == null) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: servio de dados da origem no disponvel: {1} {2}",
            new Object[] { userLogin, sourceDataKey.getSystemDeploymentId(),
                sourceDataKey.getDataSourceId() });
      Server.logWarningMessage(msg);
      throw new UnavailableDataService(msg);
    }

    UnstructuredDataView sourceView;
    try {
      sourceView =
        UnstructuredDataView.class.cast(sourceDataService.getDataView(
          sourceDataKey.getKey(), UnstructuredDataViewHelper.id()));
    }
    catch (UnsupportedView e) {
      throw new AbsentViews("Viso no suportada pelo dado.", e.fKeys,
        new String[] { UnstructuredDataViewHelper.id() });
    }

    UnstructuredDataView view;
    try {
      view =
        UnstructuredDataView.class.cast(this.createDataView(file,
          UnstructuredDataViewHelper.id()));
    }
    catch (UnsupportedView e) {
      throw new AbsentViews("Viso no suportada pelo dado.", e.fKeys,
        new String[] { UnstructuredDataViewHelper.id() });
    }

    SyncRemoteFileChannel channel;
    try {
      channel =
        new SyncRemoteFileChannel(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: erro ao converter o identificador do dado de destino ({1}) para bytes.",
            new Object[] { userLogin, dataKey.getDataId() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    SyncRemoteFileChannel sourceChannel;
    try {
      sourceChannel =
        new SyncRemoteFileChannel(sourceDataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), sourceView.fWritable, sourceView.fHost,
          sourceView.fPort, sourceView.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: erro ao converter o identificador do dado de origem ({1}) para bytes.",
            new Object[] { userLogin, sourceDataKey.getDataId() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      channel.open(false);
    }
    catch (PermissionException e) {
      Server.logWarningMessage(e.getMessage());
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso ao dado.",
        new byte[][] { fKey });
    }
    catch (FileNotFoundException e) {
      Server.logWarningMessage(e.getMessage());
      throw new DataNotFound("Dado no encontrado.", new byte[][] { fKey });
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (MaxClientsReachedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: quantidade mxima de clientes atingida: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      sourceChannel.open(true);
    }
    catch (PermissionException e) {
      Server.logWarningMessage(e.getMessage());
      throw new DataAccessDenied(
        "Usurio solicitante no tem permisso de acesso ao dado.",
        new byte[][] { fSourceKey });
    }
    catch (FileNotFoundException e) {
      Server.logWarningMessage(e.getMessage());
      throw new DataNotFound("Dado no encontrado.",
        new byte[][] { fSourceKey });
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (MaxClientsReachedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: quantidade mxima de clientes atingida: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      ProjectItemDataView sourceProjectView =
        (ProjectItemDataView) sourceDataService.getDataView(sourceDataKey
          .getKey(), ProjectItemDataViewHelper.id());

      channel.syncTransferFrom(sourceChannel, 0, sourceProjectView.fSize);
      channel.setSize(sourceProjectView.fSize);
    }
    catch (PermissionException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: arquivo aberto somente para escrita: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: arquivo bloqueado para escrita: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (UnsupportedView e) {
      throw new AbsentViews("Viso no suportada pelo dado.", e.fKeys,
        new String[] { ProjectItemDataViewHelper.id() });
    }
    finally {
      try {
        OpenBusService.getInstance().exitChain();
        channel.close();
      }
      catch (FailureException e) {
      }
      finally {
        try {
          sourceChannel.close();
        }
        catch (FailureException e) {
        }
      }
    }

    if (!ProjectService.getInstance().setUpdatableFileInfo(
      file,
      new UpdatableFileInfo(OpenBusProjectFileUpdater.class, sourceDataKey
        .getKey()))) {
      Server.logWarningMessage(MessageFormat.format(
        "A informao sobre atualizao do arquivo {0} no foi definida",
        new Object[] { file.getStringPath() }));
    }
  }

  /**
   * Atualiza um arquivo a partir de um dado localizado em um outro servidor.
   * 
   * @param projectId O identificador do projeto.
   * @param pathArray O caminho para o arquivo.
   * @param fSourceKey A chave unvoca do dado de origem, ou seja, o dado
   *        armazenado no outro servidor.
   * 
   * @throws ServiceFailure Caso ocorra uma falha no servio.
   * @throws DataAccessDenied Caso o usurio solicitante no tenha acesso ao
   *         dado de origem
   * @throws InvalidDataKey Caso a chave do dado de origem no seja vlida.
   * @throws DataNotFound Caso o dado de origem no exista.
   * @throws UnavailableDataService Caso o servio de dados da origem esteja
   *         indisponvel
   * @throws AbsentViews Caso as vises oferecidas pelo dado de origem no sejam
   *         conhecidas deste servidor.
   */
  public void updateDataFrom(Object projectId, String[] pathArray,
    byte[] fSourceKey) throws ServiceFailure, DataAccessDenied, InvalidDataKey,
    DataNotFound, UnavailableDataService, AbsentViews {
    String dataId = generateDataId(projectId, pathArray);
    DataKeyWrapper dataKey = this.generateDataKey(dataId);
    this.updateDataFrom(dataKey.getKey(), fSourceKey);
  }

  /**
   * Obtm informaes sobre o arquivo de projeto representado por uma chave.
   * 
   * @param dataKey A chave.
   * 
   * @return As informaes sobre o arquivo de projeto.
   * 
   * @throws InvalidDataKey Caso a chave no represente um arquivo.
   * @throws ServiceFailure Caso ocorra uma falha no servio.
   * @throws DataNotFound Caso o arquivo de projeto no exista.
   * @throws DataAccessDenied Caso o usurio solicitante no tenha acesse ao
   *         arquivo
   */
  public static ClientProjectFile getProjectFile(DataKeyWrapper dataKey)
    throws InvalidDataKey, ServiceFailure, DataNotFound, DataAccessDenied {
    String dataId = dataKey.getDataId();

    String[] splittedKey = dataId.split(ProjectService.FILE_ID_SEPARATOR);
    if (splittedKey.length < ProjectService.MINIMUM_FILE_ID_SIZE) {
      throw new InvalidDataKey("Chave do dado  invlida.",
        new byte[][] { dataKey.getKey() });
    }

    Object projectId = splittedKey[0];

    if (!ProjectService.getInstance().existsProject(projectId)) {
      throw new DataNotFound("Dado no encontrado.", new byte[][] { dataKey
        .getKey() });
    }

    Object userId = Service.getUser().getId();

    if (!ProjectService.getInstance().userHasAccess(projectId, userId)) {
      throw new DataAccessDenied("Usurio solicitante " + userId
        + " no tem acesso ao dado.", new byte[][] { dataKey.getKey() });
    }

    CommonClientProject project;
    try {
      project = ProjectService.getInstance().openProject(projectId, false);
    }
    catch (ServiceFailureException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(projectId);
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao abrir o projeto {1}: {2}",
            new Object[] { userLogin, projectName, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    if (splittedKey.length == ProjectService.MINIMUM_FILE_ID_SIZE) {
      ClientProjectFile file = project.getRoot();
      if (file == null) {
        throw new DataNotFound("Dado no encontrado.", new byte[][] { dataKey
          .getKey() });
      }
      return file;
    }
    else {
      String[] filePath =
        new String[splittedKey.length - ProjectService.MINIMUM_FILE_ID_SIZE];
      for (int i = 0; i < filePath.length; i++) {
        filePath[i] = splittedKey[i + ProjectService.MINIMUM_FILE_ID_SIZE];
      }
      try {
        ClientProjectFile file = project.getFile(filePath);
        if (file == null) {
          throw new DataNotFound("Dado no encontrado.", new byte[][] { dataKey
            .getKey() });
        }
        return file;
      }
      catch (RemoteException e) {
        String userLogin = Service.getUser().getLogin();
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: erro ao obter o arquivo {1}: {2}",
              new Object[] { userLogin, filePath, e.getMessage() });
        Server.logSevereMessage(msg, e);
        throw new ServiceFailure(msg);
      }
    }
  }

  /**
   * Gera um identificador nico para um arquivo.
   * 
   * @param file O arquivo.
   * 
   * @return O identificador nico.
   */
  private static String generateDataId(ClientProjectFile file) {
    return generateDataId(file.getProjectId(), file.getPath());
  }

  /**
   * Gera um identificador para um arquivo de projeto.
   * 
   * @param projectId O identificador do arquivo.
   * @param pathArray O caminho para o arquivo de projeto.
   * @return O identificador.
   */
  private static String generateDataId(Object projectId, String[] pathArray) {
    String dataId = (String) projectId;
    for (String path : pathArray) {
      dataId += ProjectService.FILE_ID_SEPARATOR + path;
    }
    return dataId;
  }

  /**
   * Gera um identificador para um arquivo de projeto a partir do seu diretrio
   * de origem.
   * 
   * @param parent O diretrio de origem do arquivo.
   * @param childName O nome do arquivo.
   * 
   * @return O identificador.
   */
  private static String generateDataId(ClientProjectFile parent,
    String childName) {
    String dataId = generateDataId(parent);
    dataId += ProjectService.FILE_ID_SEPARATOR + childName;
    return dataId;
  }

  /**
   * Gera um identificador unvoco a partir de um identificador de um arquivo de
   * projeto.
   * 
   * @param dataId O identificador do arquivo de projeto.
   * 
   * @return O identificador unvoco.
   * 
   * @throws InvalidDataKey Caso ocorra algum erro ao criar o identificador
   *         unvoco.
   */
  private DataKeyWrapper generateDataKey(String dataId) throws InvalidDataKey {
    String systemDeploymentId = OpenBusService.getInstance().getEntityName();
    DataKeyWrapper dataKey =
      new DataKeyWrapper(systemDeploymentId, SourceId, dataId);
    return dataKey;
  }

  /**
   * Cria uma descrio de dado para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * 
   * @return A descrio do dado.
   * 
   * @throws InvalidDataKey Caso no seja possvel criar a chave unvoca do
   *         dado.
   * @throws ServiceFailure
   */
  private DataDescription createDataDescription(ClientProjectFile file)
    throws InvalidDataKey, ServiceFailure {
    DefaultView defaultView;
    try {
      defaultView = createDefaultDataView(file);
    }
    catch (UnsupportedView e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(file.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao criar descrio do dado {1} do projeto {2}: {3}",
            new Object[] { userLogin, file.getName(), projectName,
                e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    String[] othersViews = createViewSet(file.isDirectory(), file.isRoot());
    Metadata[] metadata = createMetadataList(file.getStringPath());

    return new DataDescription(defaultView.fValue.getKey(), file.getName(),
      defaultView, othersViews, metadata);
  }

  /**
   * Cria uma viso para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * @param viewInterface O nome da interface da viso.
   * 
   * @return A viso.
   * 
   * @throws UnsupportedView Caso a viso solicitada no seja oferecida pelo
   *         dado.
   * @throws InvalidDataKey Caso ocorra erro ao criar a chave unvoca do arquivo
   *         de projeto.
   * @throws ServiceFailure Caso ocorra alguma falha inesperada durante a
   *         criao da viso.
   */
  private DataView createDataView(ClientProjectFile file, String viewInterface)
    throws UnsupportedView, ServiceFailure, InvalidDataKey {
    if (viewInterface.equals(UnstructuredDataViewHelper.id())) {
      return this.createUnstructuredDataView(file);
    }
    else if (viewInterface.equals(HierarchicalNodeDataViewHelper.id())) {
      return this.createHierarchicalNodeDataView(file);
    }
    else if (viewInterface.equals(ProjectItemDataViewHelper.id())) {
      return this.createProjectItemDataView(file);
    }
    else if (viewInterface.equals(ProjectDataViewHelper.id())) {
      return this.createProjectDataView(file);
    }
    String dataIdString = generateDataId(file);
    DataKeyWrapper dataKey = this.generateDataKey(dataIdString);
    throw new UnsupportedView("Viso no suportada.", new byte[][] { dataKey
      .getKey() });
  }

  /**
   * Cria uma viso no-estruturada para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * 
   * @return A viso no-estruturada.
   * 
   * @throws UnsupportedView Caso a viso solicitada no seja oferecida pelo
   *         dado.
   * @throws InvalidDataKey Caso ocorra erro ao criar a chave unvoca do arquivo
   *         de projeto.
   * @throws ServiceFailure Caso ocorra alguma falha inesperada durante a
   *         criao da viso.
   */
  private UnstructuredDataView createUnstructuredDataView(ClientProjectFile file)
    throws UnsupportedView, ServiceFailure, InvalidDataKey {
    String dataIdString = generateDataId(file);
    DataKeyWrapper dataKey = this.generateDataKey(dataIdString);
    if (file.isDirectory()) {
      throw new UnsupportedView("Viso no suportada por diretrios.",
        new byte[][] { dataKey.getKey() });
    }
    ProjectService prjSrv = ProjectService.getInstance();
    try {
      RemoteFileChannelInfo accessInfo = prjSrv.createFileChannelInfo(file);
      return new UnstructuredDataViewImpl(dataKey.getKey(), accessInfo
        .getHost(), accessInfo.getPort(), accessInfo.getKey(), true);
    }
    catch (csbase.exception.PermissionException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(file.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao criar viso no estruturada do dado {1} do projeto {2}: usurio sem permisso.",
            new Object[] { userLogin, file.getName(), projectName });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (Exception e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(file.getProjectId());
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao criar viso no estruturada do dado {1} do projeto {2}: {3}",
            new Object[] { userLogin, file.getName(), projectName,
                e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
  }

  /**
   * Cria uma viso hierquica para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * 
   * @return A viso hierquica.
   * 
   * @throws InvalidDataKey Caso ocorra erro ao criar a chave unvoca do arquivo
   *         de projeto.
   */
  private HierarchicalNodeDataView createHierarchicalNodeDataView(
    ClientProjectFile file) throws InvalidDataKey {
    String dataIdString = generateDataId(file);
    DataKeyWrapper dataKey = this.generateDataKey(dataIdString);

    ClientProjectFile parent = file.getParent();
    if (parent != null) {
      String parentDataIdString = generateDataId(parent);
      DataKeyWrapper parentDataKey = this.generateDataKey(parentDataIdString);

      return new HierarchicalNodeDataViewImpl(dataKey.getKey(), parentDataKey
        .getKey(), file.isDirectory());
    }
    else {
      return new HierarchicalNodeDataViewImpl(dataKey.getKey(), new byte[0],
        file.isDirectory());
    }
  }

  /**
   * Cria a viso padro de um arquivo
   * 
   * @param file arquivo
   * 
   * @return viso padro do arquivo: ProjectDataView para arquivos que so root
   *         de projetos e ProjectItemDataView para os demais
   * 
   * @throws InvalidDataKey caso ocorra erro na gerao da chave do arquivo
   * @throws UnsupportedView caso tente-se cirar a viso ProjectDataView para
   *         arquivos que no sejam o root do projeto
   * @throws ServiceFailure caso ocorra erro na recuperao do tamanho total do
   *         projeto
   */
  private DefaultView createDefaultDataView(ClientProjectFile file)
    throws InvalidDataKey, UnsupportedView, ServiceFailure {
    if (file.isRoot()) {
      return new DefaultView(ProjectDataViewHelper.id(),
        createProjectDataView(file));
    }
    else {
      return new DefaultView(ProjectItemDataViewHelper.id(),
        createProjectItemDataView(file));
    }
  }

  /**
   * Cria uma viso do tipo ProjectItemDataView para um arquivo
   * 
   * @param file arquivo
   * 
   * @return viso do tipo ProjectItemDataView que representa um arquivo
   * 
   * @throws InvalidDataKey caso ocorra erro na gerao da chave do arquivo
   */
  private ProjectItemDataView createProjectItemDataView(ClientProjectFile file)
    throws InvalidDataKey {
    String description = null;
    try {
      description = file.getDescription();
    }
    catch (RemoteException e) {
      Server.logWarningMessage(MessageFormat.format(
        "Erro ao obter a descrio do arquivo {0}.", new Object[] { file
          .getStringPath() }));
    }

    String dataIdString = generateDataId(file);
    DataKeyWrapper dataKey = this.generateDataKey(dataIdString);

    String[] pathArray = file.getPath();
    String path = (String) file.getProjectId();
    for (int i = 0; i < pathArray.length; i++) {
      path += PATH_SEPARATOR.value + pathArray[i];
    }

    String owner = (String) file.whoCreated();
    if (owner == null) {
      owner = "";
    }
    if (description == null) {
      description = "";
    }

    return new ProjectItemDataViewImpl(dataKey.getKey(), owner, description,
      path, file.getType(), file.size(), file.isDirectory(), file.canRead(),
      file.canWrite(), file.getCreationDate(), file.getModificationDate());
  }

  /**
   * Cria uma viso do tipo ProjectDataView para um projeto (via seu arquivo
   * root).
   * 
   * @param file arquivo root do projeto
   * 
   * @return viso do tipo ProjectDataView que representa um projeto
   * 
   * @throws InvalidDataKey caso ocorra erro na gerao da chave do projeto
   * @throws UnsupportedView caso o arquivo informado no seja o root do projeto
   * @throws ServiceFailure caso ocorra erro na recuperao do tamanho total do
   *         projeto
   */
  private ProjectDataView createProjectDataView(ClientProjectFile file)
    throws InvalidDataKey, UnsupportedView, ServiceFailure {
    if (!file.isRoot()) {
      String dataIdString = generateDataId(file);
      DataKeyWrapper dataKey = this.generateDataKey(dataIdString);
      throw new UnsupportedView(
        "Viso somente suportada por dados razes de projetos.",
        new byte[][] { dataKey.getKey() });
    }

    Object projectId = file.getProjectId();
    return createProjectDataView(projectId);
  }

  /**
   * Cria uma viso do tipo ProjectDataView para um projeto.
   * 
   * @param projectId identificador do projeto
   * 
   * @return viso do tipo ProjectDataView que representa um projeto
   * 
   * @throws InvalidDataKey caso ocorra erro na gerao da chave do projeto
   * @throws ServiceFailure caso ocorra erro na recuperao do tamanho total do
   *         projeto
   */
  private ProjectDataView createProjectDataView(Object projectId)
    throws InvalidDataKey, ServiceFailure {
    ProjectService prjService = ProjectService.getInstance();

    CommonClientProject clientProject =
      prjService.openProject(projectId, false);

    Set<Object> userRO = clientProject.getUsersRO();
    Set<Object> userRW = clientProject.getUsersRW();

    Set<String> readingUsers = new HashSet<String>(userRO.size());
    for (Object user : userRO) {
      readingUsers.add(String.class.cast(user));
    }

    Set<String> writingUsers = new HashSet<String>(userRW.size());
    for (Object user : userRW) {
      writingUsers.add(String.class.cast(user));
    }

    String dataId = generateDataId(clientProject.getRoot());
    DataKeyWrapper dataKey = generateDataKey(dataId);

    String path = (String) clientProject.getId();

    long projectSize;
    try {
      projectSize =
        clientProject.getRoot().getTotalSize(new ProjectFileNameFilter(""));
    }
    catch (RemoteException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(projectId);
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: erro ao recuperar o tamanho total do projeto {1}: {2}",
            new Object[] { userLogin, projectName, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    return new ProjectDataViewImpl(dataKey.getKey(), //key
      (String) clientProject.getUserId(), //owner
      (clientProject.getDescription() != null) ? clientProject.getDescription()
        : "", //description
      path, //path
      clientProject.getOwnerServerName(), //software
      projectSize, //size
      readingUsers, //readingUsers 
      writingUsers, //writingUsers
      clientProject.getCreationDate(), //creationDate 
      clientProject.getLastModificationDate() //modificationDate
    );
  }

  /**
   * Cria uma representao de descrio de dado vazia.
   * 
   * @return a descrio de dado
   */
  private DataDescription createEmptyDataDescription() {
    DataDescription emptyDataDescription = new DataDescription();
    emptyDataDescription.fKey = new byte[0];
    emptyDataDescription.fName = "";
    emptyDataDescription.fDefaultView = new DefaultView();
    emptyDataDescription.fOthersViews = new String[0];
    emptyDataDescription.fMetadata = new Metadata[0];

    return emptyDataDescription;
  }
}
