/*
 * $Id$
 */
package csbase.server.services.administrationservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;
import java.util.Map.Entry;

import csbase.logic.IdFactory;
import csbase.logic.UserGroup;
import csbase.logic.UserGroupInfo;
import csbase.server.Server;

/**
 * A classe <code>UserGroupIO</code>  responsvel pela persistncia dos
 * registros dos UserGroups. Todos os objetos so mantidos em um arquivo texto
 * que  atualizado sempre que as funes UserGroup.createUserGroup,
 * UserGroup.modifyUserGroup e UserGroup.deleteUserGroup so invocadas. 
 * responsvel tambm pela gerao do identificador de um UserGroup. O
 * identificador  um inteiro sequencial. O algoritmo reutiliza os valores de
 * identificadores liberados pela remoo de UserGroups.
 */
public class UserGroupIO {
  /** O gerador de identificadores de UserGroups */
  protected IdFactory idFactory;
  /** Chave do nome do UserGroup na Hashtable persistida */
  private static final String NAME = "name";
  /** Chave da descrio do UserGroup na Hashtable persistida */
  private static final String DESCRIPTION = "description";
  /** Nome da propriedade que possui o valor do nome do arquivo de UserGroups */
  private static String FILENAME_PROPERTY = "UserGroupIO.filename";
  /**
   * Nome do arquivo default de UserGroups, usado no caso da propriedade acima
   * no existir
   */
  private static String FILENAME = "usergroup.dat";
  /** Nome do arquivo de UserGroups no existir. */
  private static String fileName = FILENAME;

  /**
   * Constri um objeto responsvel pela persistncia de UserGroups. O nome do
   * arquivo de dados  obtido da propriedade <b>UserGroupIO.filename</b> de
   * <code>Environment</code>.
   */
  public UserGroupIO() {
    final Server server = Server.getInstance();
    AdministrationService service = AdministrationService.getInstance();
    fileName = server.getPersistencyRootDirectoryName() + File.separator
        + service.getStringProperty(FILENAME_PROPERTY);
    Server.logInfoMessage("Arquivo de UserGroups: " + fileName);
  }

  /**
   * Imprime na tela o contedo do arquivo de dados
   * 
   * @param args argumentos
   * 
   * @throws IOException em caso de falha de I/O.
   */
  public static void main(String[] args) throws IOException {
    new UserGroupIO().printDataFile();
  }

  /**
   * Imprime na tela o contedo do arquivo de grupos de usurios.
   * 
   * @throws IOException erro de IO
   */
  public void printDataFile() throws IOException {
    try {
      for (UserGroup userGroup : readAll()) {
        System.out.println(userGroup);
      }
    }
    catch (Exception e) {
      System.out.println(e);
    }
  }

  /**
   * Cria um novo UserGroup e grava no arquivo.
   * 
   * @param info os dados a serem usados na criao do UserGroup
   * 
   * @return um novo UserGroup
   * 
   * @throws IOException erro de I/O
   */
  public synchronized UserGroup writeNew(UserGroupInfo info) throws IOException {
    try {
      List<UserGroup> userGroups = UserGroup.getAllUserGroups();
      if (idFactory == null) {
        idFactory = new IdFactory(userGroups);
      }
      Object id = idFactory.next();
      UserGroup userGroup = new UserGroup(id, info);
      userGroups.add(userGroup);
      writeAll(userGroups);
      return userGroup;
    }
    catch (Exception e) {
      throw new IOException("Arquivo de UserGroups corrompido");
    }
  }

  /**
   * Modifica um UserGroup e grava no arquivo.
   * 
   * @param id o identificador do UserGroup a ser gravado
   * @param info os dados do UserGroup a ser gravado
   * 
   * @return o UserGroup gravado
   * 
   * @throws IOException se esse UserGroup no existir no arquivo ou se ocorrer
   *         algum erro de gravao
   */
  public synchronized UserGroup write(Object id, UserGroupInfo info)
    throws IOException {
    try {
      UserGroup oldUserGroup = UserGroup.getUserGroup(id);
      List<UserGroup> userGroups = UserGroup.getAllUserGroups();
      if (!userGroups.remove(oldUserGroup)) {
        throw new IOException("UserGroup no existe");
      }
      UserGroup userGroup = new UserGroup(id, info);
      userGroups.add(userGroup);
      writeAll(userGroups);
      return userGroup;
    }
    catch (Exception e) {
      throw new IOException(e.getMessage());
    }
  }

  /**
   * Remove um UserGroup do arquivo.
   * 
   * @param id o identificador do UserGroup a ser removido
   * 
   * @throws IOException se esse UserGroup no existe no arquivo ou se ocorrer
   *         algum erro de gravao
   */
  public synchronized void delete(Object id) throws IOException {
    List<UserGroup> userGroups;
    UserGroup userGroup;
    try {
      userGroup = UserGroup.getUserGroup(id);
      userGroups = UserGroup.getAllUserGroups();
      if (idFactory == null) {
        idFactory = new IdFactory(userGroups);
      }
    }
    catch (Exception e) {
      throw new IOException();
    }
    if (!userGroups.remove(userGroup)) {
      throw new IOException("UserGroup no existe");
    }
    writeAll(userGroups);
    idFactory.free(userGroup.getId());
  }

  /**
   * Devolve um UserGroup que possua um determinado identificador.
   * 
   * @param id o identificador do UserGroup procurado
   * 
   * @return o grupo procurado ou null caso esse UserGroup no exista
   * 
   * @throws IOException erro de I/O
   */
  public synchronized UserGroup read(Object id) throws IOException {
    try {
      List<UserGroup> userGroups = UserGroup.getAllUserGroups();
      for (UserGroup userGroup : userGroups) {
        if (userGroup.getId().equals(id)) {
          return userGroup;
        }
      }
      return null;
    }
    catch (Exception e) {
      throw new IOException("Arquivo de UserGroups corrompido");
    }
  }

  /**
   * L e retorna todos os UserGroups existentes no arquivo de dados.
   * 
   * @return um vetor com todos os UserGroups existentes no sistema
   * 
   * @throws IOException erro de I/O
   */
  public synchronized List<UserGroup> readAll() throws IOException {
    List<UserGroup> userGroupsList = new Vector<UserGroup>();
    File userGroupsFile = new File(fileName);
    if (!userGroupsFile.exists()) {
      /*
       * arquivo de permisses ainda no existe, registramos e retornamos uma
       * lista vazia
       */
      AdministrationService srv = AdministrationService.getInstance();
      Server.logInfoMessage("Criando novo arquivo de grupos de usurios "
        + fileName);
      return userGroupsList;
    }
    ObjectInputStream in = null;
    try {
      in =
        new ObjectInputStream(new DataInputStream(new BufferedInputStream(
          new FileInputStream(userGroupsFile))));
      Hashtable<Object, Hashtable<String, String>> userGroupsHash =
        (Hashtable<Object, Hashtable<String, String>>) in.readObject();
      for (Entry<Object, Hashtable<String, String>> ugEntry : userGroupsHash
        .entrySet()) {
        Object id = ugEntry.getKey();
        Hashtable<String, String> userGroupHash = ugEntry.getValue();
        String name = userGroupHash.get(NAME);
        String desc = userGroupHash.get(DESCRIPTION);
        userGroupsList.add(new UserGroup(id, new UserGroupInfo(name, desc)));
      }
      return userGroupsList;
    }
    catch (Exception e) {
      return userGroupsList;
    }
    finally {
      if (in != null) {
        in.close();
      }
    }
  }

  /**
   * Salva todos os UserGroups no arquivo de dados.
   * 
   * @param userGroups os UserGroups a serem gravadas no arquivo
   * 
   * @throws IOException a operao no pode ser efetuada
   */
  protected synchronized void writeAll(List<UserGroup> userGroups)
    throws IOException {
    Hashtable<Object, Hashtable<String, String>> userGroupsHash =
      new Hashtable<Object, Hashtable<String, String>>();
    for (int i = 0; i < userGroups.size(); i++) {
      UserGroup userGroup = userGroups.get(i);
      Hashtable<String, String> userGroupHash = new Hashtable<String, String>();
      userGroupHash.put(NAME, userGroup.getName());
      userGroupHash.put(DESCRIPTION, userGroup.getDesc());
      userGroupsHash.put(userGroup.getId(), userGroupHash);
    }
    ObjectOutputStream out =
      new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
        new FileOutputStream(fileName))));
    try {
      out.writeObject(userGroupsHash);
    }
    finally {
      out.close();
    }
  }

}
