package csbase.server.services.ftcservice;

import java.io.File;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.ftc.server.FileServerConfig;

import csbase.remote.FTCServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Servio FTC para transferncia de arquivos entre o servidor e os clientes.
 * Essa deve ser a nica forma de transferir arquivos.
 * 
 * @author Tecgraf
 */
public class FTCService extends Service implements FTCServiceInterface {
  /**
   * O controlador do FTC, que faz o papel de owner.
   */
  private FTCController controller = null;

  /**
   * Cria a instncia do servio FTC.
   * 
   * @throws ServerException em caso de erro na construo.
   */
  public FTCService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    try {
      configController();
      controller.start();
    }
    catch (Exception e) {
      throw new ServerException(e);
    }
  }

  /**
   * Cria e configura o controlador, usando as propriedades de configurao.
   */
  private void configController() throws ServerException {
    controller = new FTCController(this);
    FileServerConfig config = controller.getConfig();
    config.setAcceptMaxPossible(getBooleanProperty("acceptMaxClients"));
    config.setClientBufferSize(getIntProperty("clientBufferSize"));
    config.setClientTimeout(getLongProperty("clientTimeout"));
    if (isPropertyNull("hostAddr")) {
      config.setHostName(Server.getInstance().getHostAddr());
    }
    else {
      config.setHostName(getStringProperty("hostAddr"));
    }
    config.setMaxClients(getIntProperty("maxClients"));
    config.setPort(getIntProperty("port"));
    config.setSelectTimeout(getLongProperty("selectTimeout"));
    config.setTestMode(getBooleanProperty("testMode"));
    controller
      .setRemoteFileChannelInfoTimeout(getLongProperty("remoteFileChannelInfoTimeout"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
    if (controller != null) {
      controller.stop();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException caso ocorra falha na inicializao
   */
  public static void createService() throws ServerException {
    new FTCService();
  }

  /**
   * Obtm a instncia do servio
   * 
   * @return a instncia do servio
   */
  public static FTCService getInstance() {
    return (FTCService) getInstance(SERVICE_NAME);
  }

  /**
   * Cria uma descrio de canal de arquivo. Essa descrio contm todas as
   * informaes necessrias para que um cliente se conecte ao servidor e acesse
   * o arquivo.
   * 
   * @param file arquivo a ser acessado.
   * @param readOnly indica se o arquivo deve ser acessado apenas para leitura
   *        (true) ou se para leitura e escrita (false).
   * 
   * @return A descrio de canal de arquivo.
   * @throws Exception em caso de falha na operao.
   */
  public RemoteFileChannelInfo createFileChannelInfo(File file,
    boolean readOnly) throws Exception {
    return controller.createFileChannelInfo(new FTCRequesterImpl(), file,
      readOnly, getLogMessage());
  }

  /**
   * Cria uma descrio de canal de arquivo. Essa descrio contm todas as
   * informaes necessrias para que um cliente se conecte ao servidor e acesse
   * o arquivo. O requisitante do canal deve responder pelo arquivo, fornecendo
   * o o FileChannel quando necessrio e informando sobre a existncia ou no de
   * lock que no permita a escrita.
   * 
   * @param requester requisitante do canal.
   * @param file arquivo a ser acessado.
   * @param readOnly indica se o arquivo deve ser acessado apenas para leitura
   *        (true) ou se para leitura e escrita (false).
   * 
   * @return A descrio de canal de arquivo.
   * @throws Exception em caso de falha na operao.
   */
  public RemoteFileChannelInfo createFileChannelInfo(FTCRequester requester,
    File file, boolean readOnly) throws Exception {
    return controller.createFileChannelInfo(requester, file, readOnly,
      getLogMessage());
  }

  /**
   * @return mensagem de log com dados do usurio.
   */
  private String getLogMessage() {
    String realUser = (String) getUser().getAttribute("realUser");
    return String.format("user: %s, name: %s", getUser().getLogin(), getUser()
      .getName(), realUser != null ? ", realUser: " + realUser : "");
  }
}
