/**
 * SchedulerResourcesController.java
 * 
 * $Id$
 * $author$ $date$ $revision$
 */
package csbase.server.services.schedulerservice;

import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.server.Server;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.sgaservice.SGAService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;

/**
 * Classe que controla os recursos usados por cada comando executado. Para
 * habilitar o controle, o administrador deve definir como true a propriedade
 * RESOURCES_CONTROL_PROPERTY. A persistncia  feita no arquivo
 * RESOURCES_BACKUP_FILE_NAME.
 * 
 * @see SchedulerService#RESOURCES_CONTROL_PROPERTY
 * @see SchedulerService#RESOURCES_BACKUP_FILE_NAME
 * 
 *      Os recursos so contabilizados antes do comando entrar em execuo. A
 *      remoo dos recursos  feita atravs de um observador, registrado no
 *      SGAService.
 * 
 * @author ururahy
 */

public class SchedulerResourcesController {
  /** Referncia para o servio. */
  private SchedulerService service;
  /** SGAService */
  private final SGAService sgaService;
  /** Indica se deve controlar os recursos dos sgas. */
  private boolean control;
  /** Caminho do arquivo para backup */
  private String backupFilePath;
  /** Estrutura que representa os recursos que cada comando necessita. */
  private ConcurrentHashMap<String, CommandResources> commands;
  /** Estrutura que representa os recursos em uso de cada sga. */
  private ConcurrentHashMap<String, SGAResources> sgas;

  /**
   * Controla os recursos usados por cada comando executado.
   * 
   * @param control Indica se deve fazer o controle
   * @param backupFilePath Caminho do arquivo de backup
   */
  SchedulerResourcesController(boolean control, String backupFilePath) {
    this.control = control;
    this.backupFilePath = backupFilePath;
    this.service = SchedulerService.getInstance();
    this.sgaService = SGAService.getInstance();
    this.commands = new ConcurrentHashMap<String, CommandResources>();
    this.sgas = new ConcurrentHashMap<String, SGAResources>();
    registerListener();
    loadUsedResources();
  }

  /**
   * Registra os ouvintes de mensagens.
   */
  private void registerListener() {
    MessageService messageService = MessageService.getInstance();

    messageService.setServerMessageListener(new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          CommandNotification notification =
            (CommandNotification) message.getBody();
          String cmdId = (String) notification.getCommandId();
          removeCommandResources(cmdId);
        }
      }
    }, new BodyTypeFilter(CommandNotification.class));
  }

  /**
   * Carrega da persistncia uma estrutura que representa os recursos de cada
   * comando em execuo.
   */
  private synchronized void loadUsedResources() {
    if (!control) {
      return;
    }

    File backupFile = new File(backupFilePath);
    if (!backupFile.exists()) {
      Server
        .logWarningMessage("Arquivo de backup no encontrado."
          + " Recursos de comandos em execuo no recuperados na inicializao do servidor.");
      return;
    }

    ObjectInputStream input = null;
    try {
      input = new ObjectInputStream(new FileInputStream(backupFilePath));

      Object obj = null;
      while ((obj = input.readObject()) != null) {
        String cmdId = (String) obj;

        Map<String, Object> resourcesContents =
          (Map<String, Object>) input.readObject();
        addCommandResources(cmdId, resourcesContents);
      }
    }
    catch (EOFException eofe) {
      //This exception will be caught when EOF is reached
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao carregar os recursos de comandos.", e);
    }
    finally {
      if (input != null) {
        try {
          input.close();
        }
        catch (IOException ioe) {
          Server.logSevereMessage(
            "Erro ao fechar stream de leitura dos recursos de comandos.", ioe);
        }
      }
    }
  }

  /**
   * Persiste uma estrutura que representa os recursos de cada comando em
   * execuo.
   */
  private synchronized void saveUsedResources() {
    ObjectOutput output = null;
    try {
      output =
        new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(
          backupFilePath)));
      for (String cmdId : commands.keySet()) {
        output.writeObject(cmdId);
        CommandResources resources = commands.get(cmdId);
        output.writeObject(resources.getContent());
      }
    }
    catch (IOException ioe) {
      Server.logSevereMessage("Erro ao gravar os recursos de comandos.", ioe);
    }
    finally {
      try {
        if (output != null) {
          output.close();
        }
      }
      catch (IOException ioe) {
        Server.logSevereMessage(
          "Erro ao fechar stream de gravao dos recursos de comandos.", ioe);
      }
    }
  }

  /**
   * Inclui recursos utilizados por um comando.
   * 
   * @param cmd o comando cujos recursos devem ser reservados.
   */
  void makeReservation(CommandInfo cmd) {
    if (!control) {
      return;
    }
    try {
      /* Obtm os recursos necessrios para a execuo do comando. */
      AlgorithmConfigurator configurator = cmd.getConfigurator();
      float cpu = configurator.getCpuAmount();
      float mem = configurator.getMemoryAmount();
      /* Inclui os recursos utilizados pelo comando na lista de recursos do sga. */
      addCommandResources(cmd.getId(), cmd.getSGAName(), cpu, mem);
    }
    catch (RemoteException re) {
      Server.logSevereMessage(String.format(
        "Erro ao adicionar o comando %s  lista de recursos do sga %s.", cmd
          .getId(), cmd.getSGAName()), re);
    }
  }

  /**
   * Usado quando o comando  adicionado pela persistncia.
   * 
   * @param cmdId o identificador do comando cuja reserva ser adicionada.
   * @param resources mapa com os recursos que o comando necessita.
   */
  private void addCommandResources(String cmdId, Map<String, Object> resources) {
    addCommandResources(cmdId, new CommandResources(resources));
  }

  /**
   * Usado quando o comando  adicionado pelo escalonador.
   * 
   * @param cmdId o identificador do comando cuja reserva ser adicionada.
   * @param sgaName o nome do sga no qual o comando ser executado.
   * @param cpu o percentual de memria que o comando necessita para executar.
   * @param mem a quantidade de memria RAM (em MB) que o comando necessita para
   *        executar.
   */
  private void addCommandResources(String cmdId, String sgaName, float cpu,
    float mem) {
    addCommandResources(cmdId, new CommandResources(cpu, mem, sgaName));
    saveUsedResources();
  }

  /**
   * @param cmdId
   * @param resources
   */
  private void addCommandResources(String cmdId, CommandResources resources) {
    String sgaName = resources.getSGAName();
    Server.logFineMessage("Adicionando recursos " + cmdId + " ao sga "
      + sgaName);
    commands.put(cmdId, resources);

    SGAResources sgaResources = sgas.get(sgaName);
    if (sgaResources != null) {
      sgaResources.addCpuAmount(resources.getCpuAmount());
      sgaResources.addMemoryAmount(resources.getMemoryAmount());
    }
  }

  /**
   * Remove os recursos utilizados por um comando.
   * 
   * @param cmdId o identificador do comando cuja reserva ser removida.
   */
  private void removeCommandResources(String cmdId) {
    Server.logFineMessage("Removendo recursos " + cmdId);

    if (!commands.containsKey(cmdId)) {
      return;
    }

    /** Exclui o comando da lista de recursos. */
    CommandResources resources = commands.remove(cmdId);
    if (resources == null) {
      Server.logSevereMessage(String.format(
        "Erro ao remover o comando %s da lista de recursos.", cmdId));
      return;
    }
    saveUsedResources();

    /**
     * Exclui os recursos utilizados pelo comando da lista de recursos do sga.
     */
    SGAResources sgaResources = sgas.get(resources.getSGAName());
    if (sgaResources != null) {
      sgaResources.subCpuAmount(resources.getCpuAmount());
      sgaResources.subMemoryAmount(resources.getMemoryAmount());
    }
  }

  /**
   * Verifica se o SGA possui CPU e memria suficientes para a execuo,
   * considerando os recursos que esto sendo controlados.
   * 
   * @param sgaSet SGA
   * @param cpuAmount Quantidade de CPU requerida
   * @param memoryAmount Quantidade de memria requerida.
   * @return <code>true</code> se houver recursos para executar o algoritmo,
   *         <code>false</code> caso contrrio.
   */
  boolean checkControllerResources(SGASet sgaSet, float cpuAmount,
    float memoryAmount) {
    if (!control) {
      return true;
    }

    if (cpuAmount <= 0 && memoryAmount <= 0) {
      return true;
    }

    Server.logFineMessage("Verificando recursos:");
    String sgaName = sgaSet.getName();
    SGAResources sgaResources = sgas.get(sgaName);
    if (sgaResources == null) {
      sgaResources = new SGAResources();
      sgas.put(sgaName, sgaResources);
      for (String cmdId : commands.keySet()) {
        CommandResources resources = commands.get(cmdId);
        if (sgaName.equals(resources.getSGAName())) {
          CommandInfo command = sgaService.getSGACommand(sgaName, cmdId);
          if (command != null) {
            sgaResources.addCpuAmount(resources.getCpuAmount());
            sgaResources.addMemoryAmount(resources.getMemoryAmount());
          }
        }
      }
      return true;
    }
    float freeCPU =
      sgaSet.getNumProcessors() - sgaResources.getCpuTotalAmount();
    float freeMem =
      sgaSet.getRAMMemoryInfoMb() - sgaResources.getMemoryTotalAmount();

    Server.logFineMessage("=======================\nController");
    Server.logFineMessage("=======================");
    Server.logFineMessage("cpuAmount =" + cpuAmount);
    Server.logFineMessage("freeCPU =" + freeCPU);
    Server.logFineMessage("memoryAmount =" + memoryAmount);
    Server.logFineMessage("freeMem =" + freeMem);

    return (cpuAmount <= freeCPU) && (memoryAmount <= freeMem);
  }

  /**
   * Informaes do uso de recursos de um comando.
   */
  class CommandResources {
    /** Nome do recurso referente a cpu necessria para a execuo do comando. */
    private final static String CPU_AMOUNT = "CPU_AMOUNT";
    /** Nome do recurso referente a cpu necessria para a execuo do comando. */
    private final static String MEMORY_AMOUNT = "MEMORY_AMOUNT";
    /** Nome do recurso referente a cpu necessria para a execuo do comando. */
    private final static String SGA_NAME = "SGA_NAME";

    /** Percentual de cpu necessria para a execuo do comando. */
    private float cpuAmount;
    /** Quantidade de memria RAM (em MB) necessria para a execuo do comando. */
    private float memoryAmount;
    /** SGA onde o comando ser executado. */
    private String sgaName;

    /**
     * Informaes do uso de recursos de um comando.
     * 
     * @param cpuAmount Percentual de cpu necessria para a execuo do comando.
     * @param memoryAmount Quantidade de memria RAM (em MB) necessria para a
     *        execuo do comando.
     * @param sgaName SGA onde o comando ser executado.
     */
    CommandResources(float cpuAmount, float memoryAmount, String sgaName) {
      this.cpuAmount = cpuAmount;
      this.memoryAmount = memoryAmount;
      this.sgaName = sgaName;
    }

    /**
     * Informaes do uso de recursos de um comando.
     * 
     * @param map Mapa com os valores dos recursos.
     */
    CommandResources(Map<String, Object> map) {
      this.cpuAmount = (Float) map.get(CPU_AMOUNT);
      this.memoryAmount = (Float) map.get(MEMORY_AMOUNT);
      this.sgaName = (String) map.get(SGA_NAME);
    }

    /**
     * Obtm o percentual de cpu necessria para a execuo do comando.
     * 
     * @return o percentual de cpu necessria para a execuo do comando.
     */
    float getCpuAmount() {
      return cpuAmount;
    }

    /**
     * Obtm a quantidade de memria RAM (em MB) necessria para a execuo do
     * comando.
     * 
     * @return a quantidade de memria RAM (em MB) necessria para a execuo do
     *         comando.
     */
    float getMemoryAmount() {
      return memoryAmount;
    }

    /**
     * Obtm o SGA onde o comando ser executado.
     * 
     * @return oSGA onde o comando ser executado.
     */
    String getSGAName() {
      return sgaName;
    }

    /**
     * Obtm um mapa com os valores dos recursos para persistir.
     * 
     * @return um mapa com os valores dos recursos
     */
    Map<String, Object> getContent() {
      Map<String, Object> map = new HashMap<String, Object>();
      map.put(CPU_AMOUNT, cpuAmount);
      map.put(MEMORY_AMOUNT, memoryAmount);
      map.put(SGA_NAME, sgaName);
      return map;
    }
  }

  /**
   * Informaes do uso de recursos de um sga.
   */
  class SGAResources {
    /** Total do percentual de cpu utilizada pelos comandos em execuo. */
    private float cpuTotalAmount;
    /** Total de memria RAM (em MB) utilizada pelos comandos em execuo. */
    private float memoryTotalAmount;

    /**
     * Informaes do uso de recursos de um sga.
     */
    SGAResources() {
      this.cpuTotalAmount = 0;
      this.memoryTotalAmount = 0;
    }

    /**
     * Obtm o total do percentual de cpu utilizada pelos comandos em execuo
     * neste sga.
     * 
     * @return o total do percentual de cpu utilizada pelos comandos em execuo
     *         neste sga.
     */
    float getCpuTotalAmount() {
      return cpuTotalAmount;
    }

    /**
     * Adiciona o parmetro cpuAmount ao total do percentual de cpu utilizada
     * pelos comandos em execuo neste sga.
     * 
     * @param cpuAmount uso de cpu a ser adicionado ao total do percentual
     *        utilizado.
     */
    void addCpuAmount(float cpuAmount) {
      this.cpuTotalAmount += cpuAmount;
      Server.logFineMessage("ADD cpuTotalAmount =" + cpuTotalAmount);

    }

    void subCpuAmount(float cpuAmount) {
      this.cpuTotalAmount -= cpuAmount;
      Server.logFineMessage("SUB cpuTotalAmount =" + cpuTotalAmount);
    }

    /**
     * Obtm o total de memria RAM (em MB) utilizada pelos comandos em
     * execuo.
     * 
     * @return o total de memria RAM (em MB) utilizada pelos comandos em
     *         execuo.
     */
    float getMemoryTotalAmount() {
      return memoryTotalAmount;
    }

    /**
     * Adiciona um valor de memria (em MB) ao total de memria utilizada pelos
     * comandos em execuo.
     * 
     * @param memoryAmount o valor de memria (em MB) a ser adicionado.
     */
    void addMemoryAmount(float memoryAmount) {
      this.memoryTotalAmount += memoryAmount;
      Server.logFineMessage("ADD memoryTotalAmount =" + memoryTotalAmount);
    }

    /**
     * Subtrai um valor de memria (em MB) do total de memria utilizada pelos
     * comandos em execuo.
     * 
     * @param memoryAmount o valor de memria (em MB) a ser subtrado.
     */
    void subMemoryAmount(float memoryAmount) {
      this.memoryTotalAmount -= memoryAmount;
      Server.logFineMessage("SUB memoryTotalAmount =" + memoryTotalAmount);
    }
  }
}
