/**
 * $Id: ServerProject.java 178884 2017-01-19 19:55:21Z isabella $
 */

package csbase.server.services.projectservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import csbase.exception.InfoException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.CloseProjectNotification;
import csbase.logic.CommonProjectInfo;
import csbase.logic.Notification;
import csbase.logic.OpenProjectNotification;
import csbase.logic.ProjectAttribute;
import csbase.logic.ProjectEvent;
import csbase.logic.ProjectFileInfo;
import csbase.logic.ProjectPermissions;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.ProjectRecoveryFailureNotification;
import csbase.logic.ProjectRecoverySuccessNotification;
import csbase.logic.User;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.mailservice.MailService;
import csbase.server.services.messageservice.MessageService;
import csbase.util.messages.Message;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Modela um projeto no servidor. Essa classe faz os acessos ao sistema de
 * arquivos para fornecer/atualizar as informaes relativas aos projetos.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ServerProject {
  /**
   * Extenso dos arquivos de configurao dos projetos.
   */
  public static final String INFO_EXTENSION = ".csbase_project_info";

  /**
   * Guarda todos os projetos abertos, para no termos mais de um objeto
   * representando o mesmo projeto. Isso facilita a construo mais eficiente de
   * observadores de projetos. Essa tabela faz um mapeamento entre
   * identificadores de projetos e suas respectivas instncias de
   * {@link ServerProject}.
   */
  private static Hashtable<Object, ServerProject> projects =
    new Hashtable<Object, ServerProject>();

  /**
   * Informaes do projeto.
   */
  private CommonProjectInfo info;

  /**
   * Raiz (diretrio-raiz) do projeto.
   */
  private ServerProjectFile tree;

  /**
   * Armazena o diretrio raiz do projeto.
   */
  private File dir;

  /**
   * Indica quantos clientes esto usando este projeto.
   */
  private int count;

  /**
   * Caminho de onde esse projeto est instalado, a partir da raiz de todos os
   * projetos.
   */
  private String[] path;

  /**
   * Lista de templates do projeto.
   */
  private final List<ProjectTemplate> projectTemplates;

  /**
   * Fornece o identificador do projeto.
   *
   * @return Identificador do projeto.
   */
  public Object getId() {
    return info.projectId;
  }

  /**
   * Fornece o caminho onde o projeto est instalado, a partir da raiz de todos
   * os projetos.
   *
   * @return Caminho a partir da raiz dos projetos at este projeto
   */
  public String[] getPath() {
    if (path != null) {
      return path;
    }
    User user;
    try {
      user = User.getUser(getUserId());
    }
    catch (Exception e) {
      return path;
    }
    path = new String[2];
    path[0] = user.getLogin();
    path[1] = getName();
    return path;
  }

  /**
   * Retorna o caminho absoluto para a raiz desse projeto.
   *
   * @return o caminho absoluto desse projeto
   */
  public String getAbsolutePath() {
    return tree.getAbsolutePath();
  }

  /**
   * Dado um usurio e um nome de projeto, obtm o identificador desse projeto.
   *
   * @param userId Identificador do usurio dono do projeto.
   * @param projectName Nome do projeto.
   * @return O identificador do projeto.
   *
   * @see #splitProjectId(Object)
   */
  public static Object getId(Object userId, String projectName) {
    try {
      User user = User.getUser(userId);
      String sep = File.separator;
      return user.getLogin() + sep + projectName;
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Informa o caminho para o diretrio raiz do projeto.
   *
   * @param projectId Identificador do projeto.
   * @return O caminho para o diretrio raiz do projeto.
   */
  public static String getAbsolutePath(Object projectId) {
    ProjectService projectService = ProjectService.getInstance();
    String path = FileUtils.joinPath(projectService.getProjectRepositoryPath(),
      (String) projectId);
    String absolutePath = null;
    try {
      absolutePath = (new File(path)).getCanonicalPath();
    }
    catch (IOException e) {
      absolutePath = (new File(path)).getAbsolutePath();
    }
    return absolutePath;
  }

  /**
   * Dado um identificador de um projeto, obtm o nome desse projeto.
   *
   * @param projectId O identificador do projeto.
   * @return O nome do projeto.
   */
  public static String getProjectName(Object projectId) {
    String[] elements = splitProjectId(projectId);
    return elements[1];
  }

  /**
   * Obtm o login do dono do projeto.
   *
   * @param projectId identificador do projeto
   * @return login do dono do projeto
   */
  public static String getOwnerLogin(Object projectId) {
    String[] elements = splitProjectId(projectId);
    return elements[0];
  }

  /**
   * Decompe o identificador do projeto em um array de strings (inverso do
   * mtodo {@link #getId(Object, String)}).
   *
   * @param projectId identificador do projeto
   * @return array de strings com os "componentes" do identificador do projeto
   *
   * @see #getId(Object, String)
   */
  private static String[] splitProjectId(Object projectId) {
    String[] elements = FileUtils.splitPath((String) projectId);
    if (elements.length < 2) {
      String message =
        String.format(
          "O texto informado (%s) no est no formato esperado para "
            + "um identificador de projeto.", projectId);
      throw new ServiceFailureException(message);
    }
    return elements;
  }

  /**
   * Dado um identificador de um projeto, obtm o usurio desse projeto.
   *
   * @param projectId O indentificador do projeto.
   * @return O usurio dono do projeto.
   */
  public static Object getOwnerId(Object projectId) {
    return getOwner(projectId).getId();
  }

  /**
   * Dado um identificador de um projeto, obtm o nome do usurio desse projeto.
   *
   * @param projectId O indentificador do projeto.
   * @return O nome usurio dono do projeto.
   */
  public static String getOwnerName(Object projectId) {
    return getOwner(projectId).getName();
  }

  /**
   * Obtm o usurio (Objeto User) a partir do project Id
   *
   * @param projectId id do projeto.
   * @return o usurio (objeto User) associado a este projectId
   */
  public static User getOwner(Object projectId) {
    String userLogin = getOwnerLogin(projectId);
    User user;
    try {
      user = User.getUserByLogin(userLogin);
    }
    catch (Exception e) {
      String message =
        String.format("Erro ao obter o usurio cujo login  %s.\n", userLogin);
      throw new ServiceFailureException(message, e);
    }
    if (user == null) {
      throw new ServiceFailureException("Usurio invlido: " + userLogin);
    }
    return user;
  }

  /**
   * Obtm o nome do servidor dono do projeto. Este nome est salvo na lista de
   * propriedades de um projeto.
   *
   * @return O nome do servidor dono do projeto.
   */
  public String getOwnerServerName() {
    String ownerServerName =
      (String) info
        .getAttribute(ProjectAttribute.SERVER_NAME.getAttributeKey());
    if (ownerServerName == null) {
      return "";
    }
    return ownerServerName;
  }

  /**
   * Obtem o arquivo de configurao do projeto. Esse arquivo est associado ao
   * diretrio raiz do projeto.
   *
   * @param dir .
   *
   * @return .
   */
  protected static File getConfigFile(File dir) {
    File configFile =
      new File(dir.getParentFile(), dir.getName() + INFO_EXTENSION);
    return configFile;
  }

  /**
   * Indica que um diretrio ({@link File})  valido para estar presente como
   * diretrio de projeto ou de usurio.
   *
   * @param file diretrio
   * @return indicativo
   */
  static boolean isValidBaseDirectory(File file) {
    return file.isDirectory() && !file.isHidden();
  }

  /**
   * Retorna os identificadores de todos os projetos de um usurio (apenas
   * aqueles dos quais  dono, no inclui projetos aos quais apenas tem acesso).
   *
   * @param userId Identificador do usurio.
   *
   * @return Array de identificadores dos projetos dos quais o usurio  dono.
   */
  public static List<Object> getAllProjectIds(final Object userId) {
    String dirPath = getBasePrjDirForUser(userId);
    File dir = new File(dirPath);
    File[] prjs = dir.listFiles(new FileFilter() {
      @Override
      public boolean accept(File file) {
        if (!isValidBaseDirectory(file)) {
          return false;
        }
        File configFile = getConfigFile(file);
        CommonProjectInfo cpi =
          readProjectInfoFromConfigFile(configFile, userId);
        return cpi != null;
      }
    });
    List<Object> ids = new ArrayList<Object>();
    if (prjs != null) {
      for (int i = 0; i < prjs.length; i++) {
        ids.add(ServerProject.getId(userId, prjs[i].getName()));
      }
    }
    return ids;
  }

  /**
   * Verifica se o usurio possui algum projeto.
   *
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio possui algum projeto prprio
   */
  public static boolean userHasHisOwnProjects(Object userId) {
    String dirPath = getBasePrjDirForUser(userId);
    File dir = new File(dirPath);
    File[] prjDirs = dir.listFiles(new FileFilter() {
      boolean foundOne = false;

      @Override
      public boolean accept(File file) {
        if (!foundOne && file.isDirectory()) {
          /*
           * s queremos retornar true para o 1o diretrio encontrado
           */
          foundOne = true;
          return true;
        }
        return false;
      }
    });
    /*
     * o array ser null se houve algum erro ou se dir no  um diretrio, ser
     * vazio se no existe nenhum projeto, ou ter apenas 1 elemento caso haja
     * algum projeto
     */
    return prjDirs != null && prjDirs.length > 0;
  }

  /**
   * Retorna o diretrio-base dos projetos de um determinado usurio.
   *
   * @param userId id do usurio
   * @return diretrio base para os projetos do usurio
   */
  private static String getBasePrjDirForUser(Object userId) {
    try {
      User user = User.getUser(userId);
      ProjectService projectService = ProjectService.getInstance();
      String projectRepositoryPath = projectService.getProjectRepositoryPath();
      return FileUtils.joinPath(projectRepositoryPath, user.getLogin());
    }
    catch (RemoteException e) {
      String err = "Diretrio do usurio " + userId + " no foi encontrado.";
      throw new ServiceFailureException(err, e);
    }
  }

  /**
   * Remove o diretrio-base dos projetos de um determinado usurio.
   *
   * @param userId Id do usurio
   * @return <code>true</code> se removeu a pasta com sucesso ou se a pasta j
   *         no existia ou <code>false</code> caso a remoo tenha falhado.
   */
  public static boolean removeBasePrjDirForUser(Object userId) {
    String userProjectPath = ServerProject.getBasePrjDirForUser(userId);
    File dir = new File(userProjectPath);
    if (dir.exists()) {
      return dir.delete();
    }
    return true;
  }

  /**
   * Informa se um dado projeto existe.
   *
   * @param projectId Identificador do projeto.
   * @return Verdadeiro caso exista, falso caso contrrio.
   */
  public static boolean existsProject(Object projectId) {
    String absolutePath = getAbsolutePath(projectId);
    File projectDir = new File(absolutePath);
    boolean exists = projectDir.exists();
    boolean directory = projectDir.isDirectory();
    return exists && directory;
  }

  /**
   * Abre um projeto de um usurio. Verifica se o usuario que quer abrir o
   * projeto participa dele.
   *
   * @param projectId Identificador do projeto.
   * @param notify Indica se os clientes devem ser notificados.
   * @return Objeto que representa o projeto.
   */
  public static synchronized ServerProject openProject(Object projectId,
    boolean notify) {
    ProjectService prjService = ProjectService.getInstance();
    ServerProject sp = openProject(projectId);

    if (ProjectService.isInternalServerRequest()) {
      return sp;
    }

    // O teste de permisso precisa ser feito aps a abertura porque 
    // necessrio ter o projeto aberto para saber se um usurio tem acesso a
    // ele.
    if (!sp.userHasAccess(Service.getUser().getId())) {
      throw new PermissionException("Usuario no participa do projeto");
    }
    sp.count++;
    if (notify) {
      notifyProjectUsers(sp,
        new OpenProjectNotification(prjService.getSenderName(), sp.getName()));
    }
    return sp;
  }

  public static synchronized ServerProject openServerProject(Object projectId) {
    return openProject(projectId);
  }

  private static synchronized ServerProject openProject(Object projectId) {
    Object ownerId = getOwnerId(projectId);
    String path = getAbsolutePath(projectId);
    ServerProject sp = getProject(projectId);
    if (sp == null) {
      File dir = new File(path);
      if (!dir.exists() || !dir.isDirectory()) {
        throw new ServiceFailureException(String.format(
          "projeto '%s' no existe para o usurio '%s'",
          getProjectName(projectId), getOwnerLogin(projectId)));
      }
      final File configFile = getConfigFile(dir);
      if (!configFile.isFile()) {
        return null;
      }
      CommonProjectInfo info;
      info = ServerProject.readProjectInfoFromConfigFile(configFile, ownerId);
      if (info == null) {
        String err = "O arquivo de configurao do projeto est inconsistente.";
        throw new ServiceFailureException(err);
      }
      sp = new ServerProject(projectId, info, dir);
      sp.getTree().createdBy = ownerId;
      projects.put(projectId, sp);

      ProjectService projectService = ProjectService.getInstance();
      sp.setProjectTemplates(projectService.getProjectTemplate(), false);
      Server
        .logInfoMessage("Projeto " + projectId
          + " aberto no servidor. Numero de projetos abertos: "
          + projects.size());
    }

    return sp;
  }

  /**
   * Envia uma notificao para todos os usurios que tm acesso ao projeto (com
   * exceo do prprio dono).
   *
   * @param project - projeto
   * @param data - dados da notificao
   */
  private static void notifyProjectUsers(ServerProject project,
    Notification data) {
    User loggedUser = Service.getUser();
    if (loggedUser == null || project.getSharingType() != SharingType.PARTIAL) {
      return;
    }
    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(project.getId());
      MessageService.getInstance().send(new Message(data), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurio do projeto "
        + project.getId() + ".", e);
    }
  }

  /**
   * Recupera um projeto aberto de um usurio. Se o projeto no estiver aberto,
   * retorna null.
   *
   * @param id Identificador do projeto.
   *
   * @return Objeto que representa o projeto ou null caso o projeto no esteja
   *         aberto.
   */
  public static ServerProject getProject(Object id) {
    ServerProject sp = projects.get(id);
    return sp;
  }

  /**
   * Cria, abre e retorna um novo projeto.
   *
   * @param info Informaes do projeto.
   *
   * @return o projeto criado
   */
  public static ServerProject createProject(CommonProjectInfo info) {
    ProjectService projectService = ProjectService.getInstance();
    String projectPath;
    try {
      Object projectId = getId(info.userId, info.name);
      projectPath = getAbsolutePath(projectId);
    }
    catch (Exception e) {
      throw new ServiceFailureException("ServerProject.createProject:"
        + "\n userId: " + info.userId + "\n name: " + info.name
        + "Falha na construo da caminho para o projeto.", e);
    }
    File dir = new File(projectPath);
    if (dir.exists()) {
      String infoMsg =
        projectService.getFormattedString("ServerProject.info.project.exists",
          new Object[] { info.name });
      throw new InfoException(infoMsg);
    }
    if (!dir.mkdirs()) {
      String infoMsg =
        projectService.getFormattedString("ServerProject.info.create.dir",
          new Object[] { info.name });
      throw new InfoException(infoMsg);
    }
    /*
     * parmetros adicionais do projeto recm-criado (data de criao e tipo
     * default de compartilhamento)
     */
    final long now = System.currentTimeMillis();
    info.setAttribute(ProjectAttribute.CREATION_DATE.getAttributeKey(), now);
    ProjectPermissions.setSharingType(info, SharingType.PRIVATE);
    String id;
    try {
      id = (String) getId(info.userId, info.name);
    }
    catch (Exception e) {
      throw new ServiceFailureException("ServerProject.createProject:"
        + "\n userId: " + info.userId + "\n name: " + info.name
        + "Falha na construo do identificador do projeto.", e);
    }
    /*
     * Obter o nome do servidor e salvar na lista de propriedades do projeto
     */
    Server server = Server.getInstance();
    info.setAttribute(ProjectAttribute.SERVER_NAME.getAttributeKey(),
      server.getSystemName());

    ServerProject sp = new ServerProject(id, info, dir);
    File configFile = getConfigFile(dir);
    sp.writeProjectInfo(configFile);
    projects.put(id, sp);

    sp.setProjectTemplates(projectService.getProjectTemplate(), true);
    Server.logInfoMessage("Projeto " + id
      + " criado no servidor. Numero de projetos abertos: " + projects.size());
    sp.count++;
    return sp;
  }

  /**
   * Retorna o nome do projeto (com base no nome do arquivo de controle)
   *
   * @param configFile arquivo de controle
   * @return nome do projeto (inferido)
   */
  private static String getProjectNameFromConfigFile(File configFile) {
    String configFileName = configFile.getName();
    int idx = configFileName.lastIndexOf(".");
    String projectName = configFileName.substring(0, idx);
    return projectName;
  }

  /**
   * Retorna o nome do projeto (com base no nome do arquivo de controle)
   *
   * @param configFile arquivo de controle
   * @return nome do projeto (inferido)
   */
  private static Object getOwnerIdFromConfigFile(File configFile) {
    File parentDir = configFile.getParentFile();
    if (!parentDir.exists()) {
      return null;
    }
    try {
      String parentFileName = parentDir.getName();
      final User owner = User.getUserByLogin(parentFileName);
      if (owner == null) {
        return null;
      }
      Object ownerId = owner.getId();
      return ownerId;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Funo para gerar um arquivo de controle de projetos CSBASE vazio com as
   * informaes bsicas de nomes de usurio e projeto.
   *
   * @param configFile arquivo de controle a ser gerado.
   * @param ownerId identificador do usurio dono do projeto.
   * @param projectName nome do projeto.
   * @return indicativo de sucesso na criao.
   */
  public static boolean generateConfigFile(File configFile, Object ownerId,
    String projectName) {
    CommonProjectInfo cpi = new CommonProjectInfo();
    cpi.name = projectName;
    cpi.userId = ownerId;
    ProjectPermissions.setSharingType(cpi,
      ProjectPermissions.SharingType.PRIVATE);

    try (ObjectOutputStream out = new ObjectOutputStream(
      new FileOutputStream(configFile));) {
      out.writeObject(cpi);
      out.flush();
      out.close();
      return true;
    }
    catch (Exception e) {
      String pth = configFile.getAbsolutePath();
      String fmt = "Falha na recup. de arq. de controle de projeto: %s";
      String err = String.format(fmt, pth);
      Server.logSevereMessage(err, e);
      return false;
    }
  }

  /**
   * Monta o {@link CommonProjectInfo} com base no arquivo de controle.
   *
   * @param configFile arquivo de controle
   * @return objeto de controle ou {@code null} caso isso no seja possvel.
   */
  public static CommonProjectInfo createProjectInfo(File configFile) {
    try (ObjectInputStream in = new ObjectInputStream(new DataInputStream(
      new BufferedInputStream(new FileInputStream(configFile))))) {
      CommonProjectInfo cpi = (CommonProjectInfo) in.readObject();
      return cpi;
    }
    catch (Exception e) {
      String configFilePath = configFile.getAbsolutePath();
      String fmt = "Falha na carga do arquivo de controle de projeto: %s.";
      String err = String.format(fmt, configFilePath);
      Server.logSevereMessage(err, e);
      return null;
    }
  }

  /**
   * Indica se um arquivo de controle  consistente com sua estrutura interna.
   *
   * @param configFile arquivo de controle
   * @param cpi {@link CommonProjectInfo} de verificao.
   * @return indictivo
   */
  private static boolean isConfigFileConsistent(File configFile,
    CommonProjectInfo cpi) {
    final String configProjectName = getProjectNameFromConfigFile(configFile);
    if (!configProjectName.equals(cpi.name)) {
      String fmt = "O nome do projeto (%s)  inconsistente com o path: %s.";
      String err = String.format(fmt, cpi.name, configProjectName);
      Server.logSevereMessage(err);
      return false;
    }
    final Object configOwnerId = getOwnerIdFromConfigFile(configFile);
    if (configOwnerId == null) {
      String fmt = "O dono do projeto no existe com o path: %s.";
      String err = String.format(fmt, cpi.userId, configOwnerId);
      Server.logSevereMessage(err);
      return false;
    }
    if (!configOwnerId.equals(cpi.userId)) {
      String fmt = "O dono do projeto (%s)  inconsistente com o path: %s.";
      String err = String.format(fmt, cpi.userId, configOwnerId);
      Server.logSevereMessage(err);
      return false;
    }
    return true;
  }

  /**
   * Notificao de problema.
   *
   * @param userId usurio
   * @param notification notificao.
   * @TODO Temporrio de desenvolvimento
   */
  static private void notifyRecuperation(Object userId,
    Notification notification) {
    Vector<Object> ids = new Vector<Object>();
    try {
      ids.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    String[] usersIds = null;

    //Se o usurio no for administrador ele ser includo na lista
    if (!User.isAdmin(userId)) {
      ids.add(userId);
    }
    try {
      usersIds = ids.toArray(new String[0]);
      MessageService.getInstance().send(new Message(notification), usersIds);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao enviar notifico de problema.", e);
    }

    // Envio de email.
    MailService mailService = MailService.getInstance();
    ProjectService projectService = ProjectService.getInstance();
    String mailText = notification.toString();
    mailService.mailSomeUsersFromService(projectService, usersIds, mailText);
    // mailService.mailSupport(mailText);
  }

  /**
   * Gera o {@link CommonProjectInfo} de um projeto a partir de um {@link File}
   * para seu arquivo de configurao.
   *
   * @param configFile {@link File} para arquivo de configurao.
   * @param ownerId Dono do projeto.
   * @return {@link CommonProjectInfo} do projeto.
   */
  public static CommonProjectInfo readProjectInfoFromConfigFile(
    File configFile, Object ownerId) {
    if (configFile == null) {
      String err = "O arquivo de configurao no pode ser nulo.";
      throw new IllegalArgumentException(err);
    }

    final String configFilePath = configFile.getAbsolutePath();
    CommonProjectInfo cpi = createProjectInfo(configFile);

    // Testando se o arquivo de controle j existente est minimamente correto.
    if (cpi == null) {
      // Se o arquivo de controle estiver incorreto, o servio tenta recri-lo.
      String projectName = getProjectNameFromConfigFile(configFile);
      boolean recup = generateConfigFile(configFile, ownerId, projectName);
      if (!recup) {
        // Se a recrio do arq. de controle falha, o projeto  descartado
        // (retornando null) com alerta e logs correspondentes.
        // O projeto ser descartado.
        String fmt = "Falha de recup. de arq. de contole de projeto: %s";
        String msg = String.format(fmt, configFilePath);
        Server.logSevereMessage(msg);

        ProjectService prjService = ProjectService.getInstance();
        String senderName = prjService.getSenderName();
        notifyRecuperation(ownerId, new ProjectRecoveryFailureNotification(
          senderName, ownerId, projectName));
        return null;
      }

      // Informativo de arquivo de controle refeito com informativo
      // ao dono do mesmo.
      String fmt = "Arq. de controle de projeto recuperado: %s";
      String msg = String.format(fmt, configFilePath);
      ProjectService prjService = ProjectService.getInstance();
      String senderName = prjService.getSenderName();
      notifyRecuperation(ownerId, new ProjectRecoverySuccessNotification(
        senderName, ownerId, projectName));
      Server.logSevereMessage(msg);

      // Releitura do arquivo de controle (aps recuperao).
      cpi = createProjectInfo(configFile);
      if (cpi == null) {
        // Caso a releitura falhe, indica erro grave de regerao que
        // deve ser enviada tambm para o administrador.
        // O projeto ser descartado.
        String err = "Falha grave aps recuperao de projeto: ";
        Server.logSevereMessage(err + configFilePath);
        return null;
      }
    }

    // Teste de instrumentao se o arquivo de controle (path)  consistente
    // com as informaes contidas na estrutura CommonProjectInfo.
    // Este teste  mesmo com o arquivo original (lido corretamente) ou
    // ou o refeito pelo sistema.
    if (!isConfigFileConsistent(configFile, cpi)) {
      // No caso do arquivo ser inconsistente, notificao grave  enviada.
      // O projeto ser descartado.
      String err = "Inconsistncia grave aps recuperao de projeto: ";
      Server.logSevereMessage(err + configFilePath);
      return null;
    }
    return cpi;
  }

  /**
   * Fornece informaes do projeto.
   *
   * @return CommonProjectInfo info.
   */
  public CommonProjectInfo getInfo() {
    return info;
  }

  /**
   * Fornece o identificador do usurio ao qual o projeto pertence.
   *
   * @return Identificador do usurio.
   */
  public Object getUserId() {
    return info.userId;
  }

  /**
   * Fornece o nome do projeto.
   *
   * @return Nome do projeto.
   */
  public String getName() {
    return info.name;
  }

  /**
   * Altera o nome do projeto. Essa modificao s ser persistida quando o
   * mtodo {@link #modify} for chamado.
   *
   * @param name nome.
   */
  public void setName(String name) {
    // no permite por enquanto.
  }

  /**
   * Fornece a dscrio do projeto.
   *
   * @return A descrio do projeto.
   */
  public String getDescription() {
    return info.description;
  }

  /**
   * Altera a descrio do projeto. Essa modificao s ser persistida quando o
   * mtodo {@link #modify} for chamado.
   *
   * @param description descrio.
   */
  public void setDescription(String description) {
    info.description = description;
  }

  /**
   * Fornece os atributos especficos de um projeto. Por exemplo, o sistema de
   * coordenadas do websintesi.
   *
   * @return Atributos do projeto.
   */
  public Hashtable<String, Object> getAttributes() {
    return info.getAttributes();
  }

  /**
   * Altera os atributos especficos do projeto. Essa modificao s ser
   * persistida quando o mtodo {@link #modify} for chamado.
   *
   * @param attributes do projeto.
   */
  public void setAttributes(Hashtable<String, Object> attributes) {
    info.setAttributes(attributes);
  }

  /**
   * Retorna o arquivo associado  raiz do projeto.
   * <p>
   * IMPORTANTE: no se deve assumir que a rvore est preenchida, s 
   * garantido que o n-raiz esteja definido
   *
   * @return arquivo associado  raiz do projeto. Seus filhos podem no ter sido
   *         ainda definidos
   */
  public ServerProjectFile getTree() {
    return tree;
  }

  /**
   * Atualiza a lista de usurios com acesso apenas de leitura ao projeto.
   *
   * @param usersRO - usurios com acesso apenas de leitura ao projeto
   */
  public void setUsersRO(Set<Object> usersRO) {
    ProjectPermissions.setUsersRO(info, usersRO);
  }

  /**
   * Atualiza a lista de usurios com acesso de leitura e escrita ao projeto.
   *
   * @param usersRW - usurios com acesso apenas de leitura e escrita ao projeto
   */
  public void setUsersRW(Set<Object> usersRW) {
    ProjectPermissions.setUsersRW(info, usersRW);
  }

  /**
   * Vertifica se o projeto  pblico.
   *
   * @return true, caso o projeto seja pblico, ou false, caso contrrio.
   */
  public Boolean isPublic() {
    return ProjectPermissions.isPublic(info);
  }

  /**
   * Define o tipo de compartilhamento.
   *
   * @param type - tipo de compartilhamento
   */
  public void setSharingType(SharingType type) {
    ProjectPermissions.setSharingType(info, type);
  }

  /**
   * Define os templates do projeto, opcionalmente criando a estrutura descrita.
   *
   * @param projectTemplates os templates do projeto.
   * @param createDirs cria os diretrios do template.
   */
  private void setProjectTemplates(List<ProjectTemplate> projectTemplates, boolean createDirs) {
    this.projectTemplates.clear();
    if (projectTemplates != null && projectTemplates.size() > 0) {
      if (createDirs) {
        List<ProjectFileInfo> templateDirs = new ArrayList<>();
        for (ProjectTemplate template : projectTemplates) {
          templateDirs.add(template.getBaseDir());
        }
        tree.createFiles(templateDirs, getUserId(), false);
      }
      // Os templates s devem ser adicionados aps a criao dos diretrios para no interferir na criao
      this.projectTemplates.addAll(projectTemplates);
    }
  }

  /**
   * Obtm os templates do projeto.
   *
   * @return a lista de templates.
   */
  public List<ProjectTemplate> getProjectTemplates() {
    return projectTemplates;
  }

  /**
   * Obtm o tipo de compartilhamento.
   *
   * @return tipo do compartilhamento
   */
  public SharingType getSharingType() {
    return ProjectPermissions.getSharingType(info);
  }

  /**
   * Obtm a lista de usurios com acesso apenas de leitura ao projeto no modo
   * de compartilhamento seletivo.
   * <p>
   * <b>IMPORTANTE:</b> este mtodo no leva em considerao o tipo de
   * compartilhamento do projeto, i.e. pode retornar uma lista no-vazia mesmo
   * que o compartilhamento seja pblico ou privado.
   *
   * @return lista de usurios com acesso apenas de leitura no modo de
   *         compartilhamento seletivo. Caso no existam usurios nesta
   *         condio, retorna uma lista vazia.
   */
  public Set<Object> getUsersRO() {
    return ProjectPermissions.getUsersRO(info);
  }

  /**
   * Obtm a lista de usurios com acesso de leitura e escrita ao projeto no
   * modo de compartilhamento seletivo.
   * <p>
   * <b>IMPORTANTE:</b> este mtodo no leva em considerao o tipo de
   * compartilhamento do projeto, i.e. pode retornar uma lista no-vazia mesmo
   * que o compartilhamento seja pblico ou privado.
   *
   * @return lista de usurios com acesso de leitura e escrita no modo de
   *         compartilhamento seletivo. Caso no existam usurios nesta
   *         condio, retorna uma lista vazia.
   */
  public Set<Object> getUsersRW() {
    return ProjectPermissions.getUsersRW(info);
  }

  /**
   * Verifica se o usurio tem acesso ao projeto. Isto acontece se qualquer uma
   * das opes seguintes  satisfeita:
   * <ul>
   * <li>projeto  pblico (RO ou RW)
   * <li>usurio  o dono do projeto ou o admin
   * <li>compartilhamento  seletivo e o usurio tem acesso RO ou RW ao projeto
   * </ul>
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem acesso ao projeto (RO ou RW)
   */
  public boolean userHasAccess(Object userID) {
    return ProjectPermissions.userHasAccess(info, userID);
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RO ao projeto, ou seja,
   * se uma das seguintes condies  satisfeita:
   * <ul>
   * <li>o projeto  pblico apenas para leitura
   * <li>o compartilhamento  seletivo e o usurio possui acesso RO
   * </ul>
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem qualquer tipo de acesso RO ao
   *         projeto
   */
  public boolean userHasAccessRO(Object userID) {
    return ProjectPermissions.userHasAccessRO(info, userID);
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RW ao projeto, ou seja,
   * se uma das seguintes condies for satisfeita:
   * <ul>
   * <li>usurio  o admin ou o dono do projeto
   * <li>este  pblico para leitura e escrita
   * <li>o compartilhamento  seletivo e o usurio possui acesso RW
   * </ul>
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem acesso RW ao projeto
   */
  public boolean userHasAccessRW(Object userID) {
    return ProjectPermissions.userHasAccessRW(info, userID);
  }

  /**
   * Verifica se o compartilhamento  seletivo e usurio tem acesso RO ao
   * projeto.
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o compartilhamento  seletivo e o usurio tem
   *         acesso RO ao projeto
   */
  public boolean userHasSelectiveAccessRO(Object userID) {
    return ProjectPermissions.userHasSelectiveAccessRO(info, userID);
  }

  /**
   * Verifica se o compartilhamento  seletivo e usurio tem acesso RW ao
   * projeto.
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o compartilhamento  seletivo e o usurio tem
   *         acesso RW ao projeto
   */
  public boolean userHasSelectiveAccessRW(Object userID) {
    return ProjectPermissions.userHasSelectiveAccessRW(info, userID);
  }

  /**
   * Verifica se o usurio  o dono do projeto.
   *
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio  o dono do projeto
   */
  public boolean userIsOwner(Object userID) {
    return info.userId.equals(userID);
  }

  /**
   * Persiste as modificaes feitas nas informaes do projeto. Caso alguma
   * informao no possa ser alterada, seu valor original ser reestabelecido.
   * Notifica os observadores que ocorreu uma mudana nas informaes do
   * projeto.
   */
  public void modify() {
    File configFile = getConfigFile(dir);
    if (!configFile.isFile()) {
      return;
    }
    writeProjectInfo(configFile);
    Object[] arg = { info };
    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(info.projectId);
      MessageService.getInstance().send(
        new Message(ProjectEvent.makeEvent(info.projectId,
          ProjectEvent.INFO_MODIFIED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + info.projectId + ".", e);
    }
  }

  /**
   * Persiste as informaes do projeto.
   *
   * @param configFile .
   */
  private void writeProjectInfo(File configFile) {
    ObjectOutputStream out = null;
    try {
      out =
        new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
          new FileOutputStream(configFile))));
      out.writeObject(info);
    }
    catch (IOException e) {
      String errMsg = "Error ao gerar " + configFile.getAbsolutePath();
      throw (new ServiceFailureException(errMsg, e));
    }
    finally {
      try {
        if (out != null) {
          out.close();
        }
      }
      catch (IOException e) {
        throw (new ServiceFailureException("Erro fechando stream.", e));
      }
    }
  }

  /**
   * Sinaliza que a rvore do projeto deve ser atualizada. O evento  enviado
   * apenas com a raiz da rvore, sem os filhos (mas com a informao de se este
   * possui filhos).
   *
   * @see #rebuildTree()
   */
  public void refreshTree() {
    this.tree.makeSureExists();
    ProjectService projectService = ProjectService.getInstance();
    Object[] arg = { projectService.buildSingleClientProjectFile(tree) };
    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(info.projectId);
      MessageService.getInstance().send(
        new Message(ProjectEvent.makeEvent(info.projectId,
          ProjectEvent.TREE_CHANGED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + info.projectId + ".", e);
    }
  }

  /**
   * Reconstri a rvore do projeto a partir do filesystem. <b>Todos</b> os ns
   * so lidos recursivamente.
   * <p>
   * <b>IMPORTANTE:</b> esta operao  custosa para projetos com muitos
   * arquivos. Apenas os arquivos de controle do CSBase so descartados, todos
   * os demais arquivos e diretrios sero percorridos. O evento enviado em
   * resposta contm a raiz da rvore, que est completamente preenchida.
   *
   * @see #refreshTree()
   */
  public void rebuildTree() {
    this.tree.makeSureExists();
    ProjectService projectService = ProjectService.getInstance();
    Object[] arg = { projectService.buildClientProjectSubtree(tree) };
    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(info.projectId);
      MessageService.getInstance().send(
        new Message(ProjectEvent.makeEvent(info.projectId,
          ProjectEvent.TREE_CHANGED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + info.projectId + ".", e);
    }
  }

  /**
   * Remove o projeto. Caso este mtodo retorne com sucesso, o objeto que
   * representa o projeto no poder mais ser utilizado.
   */
  public void remove() {
    tree.remove(false);
    File configFile = getConfigFile(dir);
    if (!configFile.delete()) {
      throw new ServiceFailureException("ServerProject:remove:"
        + " erro na remoo do arquivo " + configFile.getAbsolutePath());
    }
    tree = null;
    Set<Object> users = new HashSet<Object>();
    users.add(getUserId());
    users.addAll(getUsersRO());
    users.addAll(getUsersRW());
    if (projects.remove(info.projectId) == this) {
      Object[] arg = { info.projectId };
      try {
        MessageService.getInstance().send(
          new Message(ProjectEvent.makeEvent(info.projectId,
            ProjectEvent.PROJECT_DELETED, arg)), users.toArray(new String[0]));
      }
      catch (RemoteException e) {
        Server.logSevereMessage("Erro ao notificar os usurios do projeto "
          + info.projectId + ".", e);
      }
    }
    else {
      String errMsg =
        "ServerProject:remove: erro na remoo do projeto " + info.projectId
          + " da hashtable";
      throw new ServiceFailureException(errMsg);
    }
  }

  /**
   * Notifica os usurios sobre o fechamento do projeto, mas no deve tirar ele
   * de memria. Olhar pendncia 377. Vai ser corrigida com a reimplentao do
   * servio de projetos.
   *
   * @param notify indicativo de notificao aos clientes.
   */
  public void close(final boolean notify) {
    if (notify) {
      ProjectService projectService = ProjectService.getInstance();

      notifyProjectUsers(
        this,
        new CloseProjectNotification(projectService.getSenderName(), this
          .getName()));
    }
    count--;
  }

  /**
   * Retorna um texto descritivo do projeto.
   *
   * @return .
   */
  @Override
  public String toString() {
    String text = "";
    text = text + "Id: " + info.projectId + "\n";
    text = text + "Userid: " + info.userId + "\n";
    text = text + "Name: " + info.name + "\n";
    text = text + "Descricao: " + info.description + "\n";
    text = text + "Arvore: " + tree + "\n";
    text = text + "Diretorio: " + dir + "\n";
    text = text + "No clientes usando: " + count + "\n";
    return text;
  }

  /**
   * Constri um <code>ServerProject</code>
   *
   * @param id Identificador do projeto.
   * @param info {@link CommonProjectInfo} referente ao arquivo de configurao
   *        do projeto.
   * @param dir Armazena o diretrio raiz do projeto.
   */
  private ServerProject(Object id, CommonProjectInfo info, File dir) {
    super();
    info.projectId = id;
    this.info = info;
    this.dir = dir;
    this.tree = ServerProjectFile.createRoot(dir, id);
    this.count = 0;
    this.projectTemplates = new ArrayList<>();
  }
}
