package csbase.server.services.administrationservice;

import java.util.List;
import java.util.Set;

import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.logic.UserOutline;
import csbase.logic.UserUpdate;

/**
 * Interface para acesso a dados de usurios. A criao de objetos que
 * implementam essa interface  responsabilidade da fbrica de DAOs do servio
 * de administrao ({@link AdministrationDAOFactory} ).
 * 
 * @author Jorge Alexandre
 */
public interface UserDAO {

  /**
   * Recupera um usurio da fonte de dados pelo seu identificador.
   * 
   * @param id identificador do usurio.
   * @return referncia para o usurio ou nulo se o usurio no existir na fonte
   *         de dados.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  User readUser(Object id) throws DAOException;

  /**
   * Recupera uma lista de informaes sumarizadas necessrias para fazer a
   * atualizao de uma cpia do repositrio de usurios. Essas informaes so
   * representadas por objetos da classe UserUpdate.
   * 
   * @return uma referncia para a lista com os resumos de atualizao de todos
   *         os usurios.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  List<UserUpdate> readAllUserUpdates() throws DAOException;

  /**
   * Recupera o conjunto de identificadores de todos os usurios existentes na
   * fonte de dados.
   * 
   * @return uma referncia para o conjunto de identificadores.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  Set<Object> readAllUserIds() throws DAOException;

  /**
   * Recupera da fonte de dados, uma lista de informaes sumarizadas de todos
   * os usurios do sistema.
   * 
   * @return uma referncia para a lista com as informaes sumarizadas dos
   *         usurios.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  List<UserOutline> readAllUserOutlines() throws DAOException;

  /**
   * Recupera todos os usurios que existem na fonte de dados.
   * 
   * @return uma referncia para a lista com os usurios.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  List<User> readAllUsers() throws DAOException;

  /**
   * Persiste o usurio na fonte de dados.
   * 
   * @param info informaes de usurio que sero persistida.
   * @return uma referncia para o usurio persistido contendo o seu
   *         identificador.
   * @throws DAOException se ocorrer algum problema na incluso do usurio na
   *         fonte de dados.
   */
  User createUser(UserInfo info) throws DAOException;

  /**
   * Modifica um usurio na fonte de dados.
   * 
   * @param id identificador do usurio.
   * @param info informaes que sero modificadas.
   * @return uma referncia para o usurio modificado.
   * @throws DAOException se ocorrer algum problema na modificao do usurio na
   *         fonte de dados.
   */
  User modifyUser(Object id, UserInfo info) throws DAOException;

  /**
   * Exclui um usurio na fonte de dados.
   * 
   * @param id identificador do usurio.
   * @throws DAOException se ocorrer algum problema na excluso do usurio na
   *         fonte de dados.
   */
  void deleteUser(Object id) throws DAOException;

  /**
   * Tranforma um super-usurio em um usurio comum. IMPORTANTE: Se o usurio
   * passado por parmetro for um super-usurio, cria uma instncia de
   * <code>User</code> para o usurio delegado por ele, seno deve ser lanada
   * uma <code>PermissionException</code>.
   * 
   * @param superuser referncia para um super-usurio.
   * @param delegatedLogin identificador de login para um usurio comum
   *        (delegado).
   * @return uma referncia para o usurio delegado.
   * @throws DAOException se ocorrer algum problema na consulta a fonte de
   *         dados.
   */
  User changeUser(User superuser, String delegatedLogin) throws DAOException;
}
