/**
 * $Id: DiagnosticService.java 155643 2014-08-25 18:18:00Z karla $
 */
package csbase.server.services.diagnosticservice;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import scs.core.ComponentId;
import csbase.logic.CommandFinalizationType;
import csbase.logic.UserOutline;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.diagnosticservice.CommandExecutionStatisticsInfo;
import csbase.logic.diagnosticservice.DeploymentInfo;
import csbase.logic.diagnosticservice.LoginStatisticsInfo;
import csbase.logic.diagnosticservice.PropertyInfo;
import csbase.logic.diagnosticservice.ResourceException;
import csbase.logic.diagnosticservice.ServerBasicInfo;
import csbase.logic.diagnosticservice.ServerDiagnosticInfo;
import csbase.logic.diagnosticservice.ServerStatisticsInfo;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.UsersStatisticsInfo;
import csbase.remote.DiagnosticServiceInterface;
import csbase.remote.ServerEntryPoint;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.algorithmservice.AlgorithmService;
import csbase.server.services.diagnosticservice.engine.Engine;
import csbase.server.services.diagnosticservice.monitors.csfs.CSFSMonitor;
import csbase.server.services.diagnosticservice.monitors.disk.DiskMonitor;
import csbase.server.services.diagnosticservice.monitors.openbus.OpenbusMonitor;
import csbase.server.services.diagnosticservice.monitors.openbus.ServiceOfferMonitor;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.schedulerservice.SchedulerService;
import csbase.server.services.serverservice.ServerService;
import csbase.server.services.sgaservice.SGAService;

/**
 * Representa o servio de Diagnstico do servidor
 * 
 * @author Tecgraf
 */
public class DiagnosticService extends Service implements
  DiagnosticServiceInterface {

  /**
   * Engine da DiagLib
   */
  private Engine diagEngine;

  /**
   * Construtor padro.
   * 
   * @throws ServerException se ocorrer falha na construo desse servio
   */
  protected DiagnosticService() throws ServerException {
    super(SERVICE_NAME);
    Engine.setLocale(getDefaultLocale());
    diagEngine = new Engine();
  }

  /**
   * Obtm uma referncia para a instncia nica (singleton) do servio de
   * diagnstico.
   * 
   * @return referncia para a instncia nica do servio de diagnstico.
   */
  public static DiagnosticService getInstance() {
    return (DiagnosticService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initService() throws ServerException {
    Server.logInfoMessage("Iniciando o servio de Diagnstico!");

    //Disponibilidade do CSFS
    if (getBooleanProperty("csfs")) {
      try {
        diagEngine.addMonitor(new CSFSMonitor());
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao criar monitor do CSFS", e);
      }
    }

    //Disponibilidade do barramento
    if (getBooleanProperty("openbus")) {
      try {
        diagEngine.addMonitor(new OpenbusMonitor());
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao criar monitor do barramento", e);
      }
    }

    //Disponibilidade dos servios publicados
    if (getBooleanProperty("published.services")) {
      Set<ComponentId> components =
        OpenBusService.getInstance().getRegisteredComponents();
      for (final ComponentId component : components) {
        try {
          diagEngine.addMonitor(new ServiceOfferMonitor(component));
        }
        catch (Exception e) {
          Server.logSevereMessage(
            "Erro ao criar o monitor de components publicados", e);
        }
      }
    }

    try {
      diagEngine.addMonitor(new DiskMonitor());
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao criar os monitores de disco", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void shutdownService() throws ServerException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Cria o servio DiagnosticService.
   * 
   * @throws ServerException se ocorrer falha na criao do servio
   */
  public static void createService() throws ServerException {
    extracted();
  }

  /**
   * Mtodo auxiliar para criar as instncias do servio.
   * 
   * @return um objeto {@link DiagnosticService}
   * @throws ServerException se ocorrer falha na criao do servio
   */
  private static DiagnosticService extracted() throws ServerException {
    return new DiagnosticService();
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public Status getStatus(String resourceName) {
    try {
      return diagEngine.getStatus(resourceName, this.getDefaultLocale());
    }
    catch (ResourceException e) {
      Server.logWarningMessage(MessageFormat.format(
        "Solicitado o estado de recurso no monitorado: {0}.", resourceName));
      return new Status(resourceName, StatusCode.NOT_MONITORED);
    }
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public ServerBasicInfo getServerBasicInfo() {
    ServerEntryPoint entryPoint = Server.getInstance().getEntryPoint();
    String systemName = null;
    String systemVersion = null;
    try {
      systemName = entryPoint.getSystemName();
    }
    catch (Exception e) {
      systemName = "";
    }
    try {
      systemVersion = entryPoint.getVersionName();
    }
    catch (Exception e) {
      systemVersion = "";
    }
    long startupTime = ServerService.getInstance().getStartupTime();
    String operationalSystem =
      String.format("%s %s %s", DeploymentInfo.OS_NAME, DeploymentInfo.OS_ARCH,
        DeploymentInfo.OS_VERSION);
    String javaVersion =
      String.format("%s %s (%s)", DeploymentInfo.JAVA_NAME,
        DeploymentInfo.JAVA_VERSION, DeploymentInfo.JAVA_VENDOR);
    List<String> arguments =
      ManagementFactory.getRuntimeMXBean().getInputArguments();
    StringBuffer JVMArgs = new StringBuffer();
    for (String arg : arguments) {
      JVMArgs.append(arg);
      JVMArgs.append(" ");
    }
    String projectDir =
      getPath(new File(ProjectService.getInstance().getProjectRepositoryPath()));
    String algoDir =
      getPath(new File(AlgorithmService.getInstance()
        .getAlgorithmRepositoryPath()));
    String persistencyDir =
      getPath(new File(Server.getInstance().getPersistencyRootDirectoryName()));
    String serverRunningDir =
      getPath(new File(Server.getInstance().getRunningDirectoryName()));
    return new ServerBasicInfo(systemName, systemVersion, startupTime,
      operationalSystem, Server.getInstance().getHostName(), Server
        .getInstance().getHostAddr(), Server.getInstance()
        .getServerHostCharsetName(), javaVersion, JVMArgs.toString(), Server
        .getInstance().getRegistryPort(), Server.getInstance().getServerLibs(),
      projectDir, algoDir, persistencyDir, serverRunningDir);
  }

  /**
   * Obtm o caminho para um diretrio que representa um repositrio no sistema
   * de arquivos.
   * 
   * @param file o objeto que faz referncia ao repositrio no sistema de
   *        arquivos.
   * @return o caminho
   */
  private String getPath(File file) {
    try {
      return file.getCanonicalPath();
    }
    catch (Exception e) {
      return file.getAbsolutePath();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UsersStatisticsInfo getUsersStatisticsInfo() throws RemoteException {
    int numUsers = ServerService.getInstance().getNumRegisteredUsers();
    ServerEntryPoint entryPoint = Server.getInstance().getEntryPoint();
    UserOutline[] loggedUsers = entryPoint.getLoggedUsers();
    return new UsersStatisticsInfo(numUsers, loggedUsers);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LoginStatisticsInfo getLoginStatisticsInfo() throws RemoteException {
    Map<String, Integer> succeededLogins =
      ServerService.getInstance().getLoginStats(true);
    Map<String, Integer> failedLogins =
      ServerService.getInstance().getLoginStats(false);
    return new LoginStatisticsInfo(succeededLogins, failedLogins);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandExecutionStatisticsInfo getCommandExecutionStatisticsInfo()
    throws RemoteException {
    Map<ExecutionType, Integer> exeStats =
      SchedulerService.getInstance().getExeTypeStats();
    int flowStats = SchedulerService.getInstance().getFlowExecutionStats();
    Map<String, Integer> algoStats =
      SchedulerService.getInstance().getAlgoStats(false);
    Map<String, Integer> flowAlgoStats =
      SchedulerService.getInstance().getAlgoStats(true);
    Map<String, Integer> userStats =
      SchedulerService.getInstance().getUserStats();
    Map<String, Integer> sgasStats = SGAService.getInstance().getSGAsStats();
    Map<CommandFinalizationType, Integer> resultsStats =
      SGAService.getInstance().getExeResultsStats(false);
    Map<CommandFinalizationType, Integer> flowResultsStats =
      SGAService.getInstance().getExeResultsStats(true);
    return new CommandExecutionStatisticsInfo(exeStats, flowStats, algoStats,
      flowAlgoStats, userStats, sgasStats, resultsStats, flowResultsStats);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DeploymentInfo getDeploymentInfo() throws RemoteException {
    return ServerService.getInstance().getDeploymentInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, PropertyInfo> getPropertiesInfo() {
    return Server.getInstance().getPropertiesInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ServerDiagnosticInfo getServerDiagnosticInfo() throws RemoteException {
    /* Informaes bsicas do servidor */
    ServerBasicInfo basicInfo = getServerBasicInfo();
    /* Propriedades de configurao do servidor */
    Map<String, PropertyInfo> propertiesInfo = getPropertiesInfo();
    /* Status do Openbus */
    Status openbusStatus = null;
    if (getBooleanProperty("openbus")) {
      openbusStatus = getStatus(OPENBUS_RESOURCE);
    }
    /* Status do CSFS */
    Status csfsStatus = null;
    if (getBooleanProperty("csfs")) {
      csfsStatus = getStatus(CSFS_RESOURCE);
    }
    /* Status dos componentes publicados do Openbus */
    List<Status> publishedComponentsStatus = null;
    if (getBooleanProperty("published.services")) {
      Set<ComponentId> components =
        OpenBusService.getInstance().getRegisteredComponents();
      publishedComponentsStatus = new ArrayList<Status>();
      for (ComponentId componentId : components) {
        Status status = getStatus(componentId.name);
        publishedComponentsStatus.add(status);
      }
    }
    /* Status de ocupao do disco */
    Status diskStatus = getStatus(SERVER_DISK_RESOURCE);

    /* Informaes estatsticas */
    UsersStatisticsInfo usersStatisticsInfo = getUsersStatisticsInfo();
    LoginStatisticsInfo loginStatisticsInfo = getLoginStatisticsInfo();
    CommandExecutionStatisticsInfo commandExecutionStatisticsInfo =
      getCommandExecutionStatisticsInfo();
    ServerStatisticsInfo statisticsInfo =
      new ServerStatisticsInfo(usersStatisticsInfo, loginStatisticsInfo,
        commandExecutionStatisticsInfo, new Date(basicInfo.startUpTime));

    /* Contri um objeto consolidado com todas as informaes */
    return new ServerDiagnosticInfo(basicInfo, propertiesInfo.values().toArray(
      new PropertyInfo[0]), statisticsInfo, openbusStatus,
      publishedComponentsStatus.toArray(new Status[0]), csfsStatus, diskStatus);
  }
}
