package csbase.server.services.repositoryservice;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.ServiceFailureException;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Servio que gerencia os repositrios de arquivos.
 *
 * NOTA: Todas as implementaes de {@link IRepository} gerenciadas por este
 * servio devem ter a visibilidade de pacote para garantir que apenas este
 * servio seja capaz de cria-los. Isso garante que todos os outros servios que
 * usam o {@link RepositoryService}, respeitem o contrato de abstrair o tipo do
 * repositrio, isto , trabalhem apenas com {@link IRepository} e
 * {@link IRepositoryFile}.
 *
 * @see IRepository
 * @see IRepositoryFile
 *
 * @author Tecgraf/PUC-Rio
 */
public class RepositoryService extends Service {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "RepositoryService";

  /** Mapa que armazena os repositrios. */
  private Map<String, IRepository> repositories;

  /**
   * Construtor.
   *
   * @throws ServerException se ocorrer erro na inicializao do servio.
   */
  protected RepositoryService() throws ServerException {
    super(SERVICE_NAME);
    repositories = new HashMap<String, IRepository>();
  }

  /**
   * Obtm a instncia do servio.
   *
   * @return instncia do servio.
   */
  public static RepositoryService getInstance() {
    return (RepositoryService) getInstance(SERVICE_NAME);
  }

  /**
   * Registra um novo repositrio no servidor.
   *
   * @param uri identificador nico do repositrio.
   */
  public synchronized void registerLocalRepository(String uri) {
    registerLocalRepository(uri, false);
  }

  /**
   * Registra um novo repositrio no servidor.
   *
   * @param uri identificador nico do repositrio.
   * @param dirMustExist indica que o diretrio do repositrio deve existir
   *        localmente: No  preciso criar o diretrio.
   */
  public synchronized void registerLocalRepository(String uri,
    boolean dirMustExist) {
    if (containRepository(uri)) {
      String f = "Repositrio j cadastrado: %s";
      throw new IllegalArgumentException(String.format(f, uri));
    }
    File file = new File(uri);
    if (!file.exists()) {
      if (dirMustExist) {
        String f =
          "No foi possvel registrar o repositrio: diretrio %s no existe.";
        throw new IllegalArgumentException(String.format(f, uri));
      }
      if (!file.mkdir()) {
        String f = "No foi possvel criar o repositrio: %s";
        throw new IllegalArgumentException(String.format(f, uri));
      }
    }
    else {
      if (!file.isDirectory()) {
        String f =
          "No foi possvel registrar o repositrio: %s no  diretrio.";
        throw new IllegalArgumentException(String.format(f, uri));
      }
    }

    repositories.put(uri, new LocalRepository(uri));
  }

  /**
   * Desregistra um repositrio do servidor.
   *
   * @param uri identificador nico do repositrio.
   * @param removeRepository flag que indica se o diretrio do repositrio
   *        dever ser removido.
   */
  public synchronized void unregisterLocalRepository(String uri,
    boolean removeRepository) {
    if (containRepository(uri)) {
      if (removeRepository) {
        removeLocalRepository(uri);
      }
      repositories.remove(uri);
    }
  }

  /**
   * Desregistra e apaga um repositrio do servidor.
   * 
   * @param uri identificador nico do repositrio.
   */
  public synchronized void deleteLocalRepository(String uri) {
    unregisterLocalRepository(uri, true);
  }

  /**
   * Retorna true se existe um repositrio cadastrado com a dada URI, false caso
   * contrrio.
   *
   * @param uri identificador nico do repositrio.
   * @return true se existe um repositrio cadastrado com a dada URI, false caso
   *         contrrio.
   */
  public boolean containRepository(String uri) {
    return repositories.get(uri) != null;
  }

  /**
   * Retorna o repositrio do servidor dado sua URI.
   *
   * @param uri identificador nico do repositrio.
   * @return repositrio.
   */
  public IRepository getRepository(String uri) {
    return repositories.get(uri);
  }

  /**
   * Mtodo auxiliar e temporrio para a criao de uma viso de arquivo local a
   * partir de um File de java.
   *
   * @param file File local
   * @return arquivo local temporrio na mquina do servidor.
   */
  public IRepositoryFile createLocalTempFile(File file) {
    return new LocalTempFile(file);
  }

  /**
   * Constri o servio.
   *
   * @throws ServerException se ocorrer erro na criao do servio.
   */
  public static void createService() throws ServerException {
    new RepositoryService();
  }

  /** {@inheritDoc} */
  @Override
  protected void initService() throws ServerException {
  }

  /** {@inheritDoc} */
  @Override
  protected void shutdownService() throws ServerException {

  }

  /** {@inheritDoc} */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Remove o repositrio apontado pela uri.
   * 
   * @param uri identificador do repositrio.
   */
  private void removeLocalRepository(String uri) {
    File file = new File(uri);
    if (file.exists()) {
      if (!FileUtils.delete(file)) {
        String errMsg = "No foi possvel apagar o repositrio: " + file
          .getAbsolutePath();
        throw new ServiceFailureException(errMsg);
      }
    }
    else {
      String errMsg = "RepositoryService:unregisterLocalRepository: " + uri
        + " no foi possvel apagar o repositrio " + file.getAbsolutePath();
      Server.logWarningMessage(errMsg);
    }
  }

}
