/**
 * $Id: AdministrationService.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.server.services.administrationservice;

import java.io.File;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import javax.swing.ImageIcon;

import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.AdministrationEvent;
import csbase.logic.IdInterface;
import csbase.logic.Permission;
import csbase.logic.Platform;
import csbase.logic.PlatformInfo;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.logic.User;
import csbase.logic.UserAdministrationPermission;
import csbase.logic.UserGroup;
import csbase.logic.UserGroupInfo;
import csbase.logic.UserInfo;
import csbase.logic.UserOutline;
import csbase.logic.UserPasswordPermission;
import csbase.logic.UserProjectInfo;
import csbase.logic.UserUpdate;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.TransactionCallbackInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.TransactionManager;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.ProjectService;
import csbase.util.messages.Message;

/**
 * A classe <code>AdministrationService</code> implementa o servio central de
 * administrao, responsvel pela gerncia de usurios, grupos de usurios,
 * perfis e plataformas. Servidores locais usam instncias da sub-classe
 * {@link LocalAdministrationService}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class AdministrationService extends Service implements
  AdministrationServiceInterface, TransactionCallbackInterface {
  /**
   * diretrio de fotos de usurios
   */
  private final String PHOTOS_SUBDIRECTORY = "photos";

  /**
   * Gerente de transaes
   */
  protected TransactionManager transaction;

  /**
   * Objeto de acesso a dados de usurios
   */
  protected UserDAO userDAO;

  /**
   * Objeto de acesso a dados de UserGroup's
   */
  protected UserGroupDAO userGroupsDAO;

  /**
   * Objeto de acesso a dados de perfis
   */
  protected RoleDAO roleDAO;

  /**
   * Objeto de acesso a dados de plataformas
   */
  protected PlatformDAO platformDAO;

  /**
   * Objeto de acesso a dados de Permisses
   */
  protected PermissionDAO permissionDAO;

  /**
   * Constri uma instncia do servio.
   * 
   * @throws ServerException Caso ocorra algum erro na criao do servio.
   */
  public static void createService() throws ServerException {
    if (Server.getInstance().isCentralServer()) {
      new AdministrationService();
    }
    else {
      new LocalAdministrationService();
    }
  }

  /**
   * Cria uma instncia do servio de administrao.
   * 
   * @throws ServerException Caso ocorra algum erro na criao do servio.
   */
  protected AdministrationService() throws ServerException {
    super(SERVICE_NAME);
    AdministrationDAOFactory daoFactory = null;
    final String DAO_FACTORY_CLASS_NAME_PROPERTY = "dao.factory.class.name";
    if (isPropertyNull(DAO_FACTORY_CLASS_NAME_PROPERTY)) {
      Server.logInfoMessage(
        "Fbrica DAO no ajustada. Usando fbrica default...");
      daoFactory = new DefaultAdministrationDAOFactory();
    }
    else {
      String daoFactoryClassName = getStringProperty(
        DAO_FACTORY_CLASS_NAME_PROPERTY);
      Class<?> daoFactoryClass;
      try {
        daoFactoryClass = Class.forName(daoFactoryClassName);
        daoFactory = (AdministrationDAOFactory) daoFactoryClass.newInstance();
      }
      catch (ClassNotFoundException e) {
        Server.logInfoMessage(
          "A classe da fbrica de DAOs do AdminstrationService ("
            + daoFactoryClassName
            + ") no foi encontrada no classpath do servidor");
        daoFactory = new DefaultAdministrationDAOFactory();
      }
      catch (IllegalAccessException e) {
        Server.logInfoMessage(
          "A classe da fbrica de DAOs do AdminstrationService ("
            + daoFactoryClassName + ") ou o seu construtor no est acessvel");
        daoFactory = new DefaultAdministrationDAOFactory();
      }
      catch (InstantiationException e) {
        Server.logInfoMessage(
          "A classe da fbrica de DAOs do AdminstrationService ("
            + daoFactoryClassName + ") no pode ser instanciada");
        daoFactory = new DefaultAdministrationDAOFactory();
      }
    }
    Server.logInfoMessage("Usando a implementao '"
      + daoFactory.getClass().getName() + "' da fbrica de "
      + "DAOs do AdminstrationService");

    try {
      this.transaction = new TransactionManager();
      this.userDAO = daoFactory.createUserDAO();
      this.userGroupsDAO = daoFactory.createUserGroupDAO();
      this.roleDAO = daoFactory.createRoleDAO();
      this.platformDAO = daoFactory.createPlatformDAO();
      this.permissionDAO = daoFactory.createPermissionDAO();
    }
    catch (Exception e) {
      throw new ServerException(e);
    }
    ClientRemoteLocator.administrationService = this;
  }

  /**
   * Obtm uma instncia do servico.
   * 
   * @return Instncia do servico.
   */
  public static AdministrationService getInstance() {
    return (AdministrationService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() {
    Server.logInfoMessage("Terminando servio de administrao...");
  }

  /**
   * Obtm o diretrio de fotos.
   * 
   * @return O diretrio de fotos.
   */
  public String getPhotoDirectory() {
    final Server server = Server.getInstance();
    return server.getPersistencyRootDirectoryName() + File.separator
      + PHOTOS_SUBDIRECTORY;
  }

  /**
   * Criao do diretorio padro de auditagem de SGAs.
   * 
   * @throws ServerException se ocorrer um erro de segurana ou I/O.
   */
  private void createPhotoDirectory() throws ServerException {
    String dname = getPhotoDirectory();
    Server.checkDirectory(dname);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    createPhotoDirectory();
  }

  /**
   * Notifica alterao aos observadores.
   * 
   * @param type O tipo do evento.
   * @param item O argumento que ser enviado aos observadores.
   */
  protected void notifyObservers(int type, Object item) {
    AdministrationEvent event = new AdministrationEvent(type, item);
    sendEvent(event);
    // Notifica localmente para que as classes da lgica fiquem corretas:
    try {
      if (item instanceof User) {
        User.update(event);
      }
      else if (item instanceof UserGroup) {
        UserGroup.update(event);
      }
      else if (item instanceof Role) {
        Role.update(event);
      }
      else if (item instanceof Platform) {
        Platform.update(event);
      }
      else if (item instanceof Permission) {
        Permission.update(event);
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na notificao local", e);
    }
  }

  /**
   * Indica se, para o argumento fornecido, o evento especificado precisa ser
   * notificado ao observador.
   * 
   * @param arg argumento do observador.
   * @param event evento a ser enviado.
   * 
   * @return se o evento deve ser enviado.
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    if (arg == null) {
      return true;
    }
    Object item = ((AdministrationEvent) event).item;
    if (item == null) {
      return true;
    }
    Object id = null;
    if (item instanceof IdInterface) {
      if (item instanceof Permission) {
        return true;
      }
      id = ((IdInterface) item).getId();
    }
    else if (item instanceof User) {
      id = ((User) item).getId();
    }
    else if (item instanceof Platform) {
      id = ((Platform) item).getId();
    }
    if (arg.equals(id)) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean lock(TransactionCallbackInterface cb) {
    return transaction.lock(cb);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLocked() {
    return transaction.isLocked();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlock(TransactionCallbackInterface cb) {
    transaction.unlock(cb);
  }

  /**
   * Verifica se a transao est travada.
   * 
   * @throws ServiceFailureException Caso a transao esteja travada.
   */
  protected void checkTransaction() throws ServiceFailureException {
    if (transaction.isLocked()) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.operation.lock"));
    }
  }

  //
  // Controle de User
  //
  /**
   * {@inheritDoc}
   */
  @Override
  public User getUser(Object id) {
    try {
      return userDAO.readUser(id);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.user"), id), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<UserUpdate> getAllUserUpdates() {
    try {
      return userDAO.readAllUserUpdates();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.users"), e.getMessage(), e);
    }
  }

  /**
   * Obtm os identificadores de todos os usurios do sistema.
   * 
   * @return Um conjunto com os identificadores de todos os usurios do sistema.
   */
  public Set<Object> getAllUserIds() {
    try {
      return userDAO.readAllUserIds();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.users"), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<UserOutline> getAllUserOutlines() {
    try {
      return userDAO.readAllUserOutlines();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.users"), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<User> getAllUsers() {
    try {
      return userDAO.readAllUsers();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.users"), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode criar usurios.
   * 
   * @throws PermissionException Caso o usurio no possa criar usurios.
   */
  protected void checkCreateUserPermission() throws PermissionException {
    User loggedUser = Service.getUser();
    Permission permission =
      loggedUser.getPermission(UserAdministrationPermission.class);
    if (!loggedUser.isAdmin() && (permission == null)) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public User createUser(UserInfo info) {
    checkCreateUserPermission();
    if (!Service.getUser().isAdmin()) {
      Server.logInfoMessage("Criando Usurio: " + info.getLogin());
    }
    try {
      User user = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          user = userDAO.createUser(info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.CREATE, user);
      return user;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.create.user"), info.getLogin()), e
          .getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio corrente pode alterar um usurio.
   * 
   * @param id O identificador do usurio que ser alterado.
   * @param info Informaes sobre o usurio que ser alterado.
   * 
   * @throws RemoteException falha de rmi
   * @throws PermissionException Caso o usurio corrente no possa alterar um
   *         usurio.
   */
  protected void checkModifyUserPermission(Object id, UserInfo info)
    throws RemoteException, PermissionException {
    User loggedUser = Service.getUser();
    Permission permission =
      loggedUser.getPermission(UserAdministrationPermission.class);
    if (!loggedUser.isAdmin() && (permission == null)) {
      if (!loggedUser.getId().equals(id)) {
        throw new PermissionException();
      }
      else if (loggedUser.getId().equals(id)) {
        // O usurio no pode alterar seu UserGroup:
        User newUser = new User(id, info);
        if (!UserGroup.getUserGroupFromUser(newUser).equals(UserGroup
          .getUserGroupFromUser(loggedUser))) {
          throw new PermissionException();
        }

        // O usurio no pode alterar seus perfis:
        Object[] userRoleIds = loggedUser.getRoleIds();
        Object[] newUserRoleIds = newUser.getRoleIds();
        if (userRoleIds.length != newUserRoleIds.length) {
          throw new PermissionException();
        }
        for (int i = 0; i < userRoleIds.length; i++) {
          if (!userRoleIds[i].equals(newUserRoleIds[i])) {
            throw new PermissionException();
          }
        }

        // O usurio no pode alterar suas permisses:
        Object[] userPermissionIds = loggedUser.getPermissionIds();
        Object[] newUserPermissionIds = newUser.getPermissionIds();
        if (userPermissionIds.length != newUserPermissionIds.length) {
          throw new PermissionException();
        }
        for (int i = 0; i < userPermissionIds.length; i++) {
          if (!userPermissionIds[i].equals(newUserPermissionIds[i])) {
            throw new PermissionException();
          }
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getPhoto(Object id) {
    if (id == null) {
      return null;
    }
    final String fname = id.toString() + ".gif";
    final String PATH = getPhotoDirectory() + File.separator + fname;
    final File file = new File(PATH);
    if (!file.exists()) {
      return null;
    }
    return new ImageIcon(PATH);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public File getPhotoFile(Object id) {
    if (id == null) {
      return null;
    }
    final String fname = id.toString() + ".gif";
    final String PATH = getPhotoDirectory() + File.separator + fname;
    final File file = new File(PATH);
    if (!file.exists()) {
      return null;
    }
    return file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public User modifyUser(Object id, UserInfo info) throws RemoteException {
    checkModifyUserPermission(id, info);
    if (!Service.getUser().isAdmin()) {
      Server.logInfoMessage("Criando Usurio: " + id);
    }
    try {
      User user = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          user = userDAO.modifyUser(id, info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.MODIFY, user);
      return user;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.modify.user"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode remover usurios.
   * 
   * @throws PermissionException Caso o usurio no possa remover usurios.
   */
  protected void checkDeleteUserPermission() throws PermissionException {
    User loggedUser = Service.getUser();
    Permission permission =
      loggedUser.getPermission(UserAdministrationPermission.class);
    if (!loggedUser.isAdmin() && (permission == null)) {
      throw new PermissionException();
    }
  }

  /**
   * Verifica se um usurio possui projetos.
   * 
   * @param userId O identificador do usurio.
   * 
   * @throws AdministrationDeleteException Caso o usurio possua projetos.
   */
  protected void checkNoProjects(Object userId)
    throws AdministrationDeleteException {
    ProjectService projService = ProjectService.getInstance();
    if (projService != null) {
      List<UserProjectInfo> projects = projService.getProjectsFromUser(userId);
      if ((projects != null) && (projects.size() > 0)) {
        throw new AdministrationDeleteException(String.format(getString(
          "AdministrationService.error.delete.user.having.project"), userId));
      }
    }
  }

  /**
   * Remove o usurio de todos os projetos a que ele tem acesso, porm no 
   * dono.
   * 
   * @param userId id do usurio.
   */
  protected void removeUserFromSharedProjects(Object userId) {
    ProjectService projService = ProjectService.getInstance();
    if (projService != null) {
      List<UserProjectInfo> sharedProjectsInfo = projService
        .getProjectsSharedWithUser(userId);
      for (UserProjectInfo anUserProjectInfo : sharedProjectsInfo) {
        try {
          projService.removeUser(anUserProjectInfo.getProjectId(), userId);
        }
        catch (Exception e) {
          Server.logSevereMessage(String.format(
            "Falha ao remover o usurio '%s' do projeto '%s'.", userId,
            anUserProjectInfo.getProjectId()), e);
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteUser(Object id) throws RemoteException,
    AdministrationDeleteException {
    checkDeleteUserPermission();
    checkNoProjects(id);
    if (!Service.getUser().isAdmin()) {
      Server.logInfoMessage("Excluindo Usurio: " + id);
    }
    try {
      User user = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          user = User.getUser(id);
          if (user == null) {
            throw new ServiceFailureException(String.format(getString(
              "AdministrationService.error.invalid.user"), id));
          }
          removeUserFromSharedProjects(id);
          userDAO.deleteUser(id);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.DELETE, user);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.delete.user"), id), e.getMessage(), e);
    }
  }

  /**
   * Tranforma um super-usurio em um usurio comum. IMPORTANTE: Se o usurio
   * passado por parmetro for um super-usurio, cria uma instncia de
   * <code>User</code> para o usurio delegado por ele, seno deve ser lanada
   * uma <code>PermissionException</code>.
   * 
   * @param superuser referncia para um super-usurio.
   * @param delegatedLogin identificador de login para um usurio comum
   *        (delegado).
   * @return uma referncia para o usurio delegado.
   * @throws ServiceFailureException se ocorrer algum problema no acesso aos
   *         dados.
   * @throws PermissionException se o usurio passado no parmetro
   *         <code>superuser</code> NO for um super-usurio.
   */
  public User changeUser(User superuser, String delegatedLogin) {
    try {
      return userDAO.changeUser(superuser, delegatedLogin);
    }
    catch (DAOException e) {
      String msg =
        String.format(getString("AdministrationService.error.change.user"),
          delegatedLogin);
      Server.logSevereMessage("changeUser: " + msg, e);
      throw new ServiceFailureException(msg);
    }
  }

  //
  // Controle de UserGroup
  //
  /**
   * {@inheritDoc}
   */
  @Override
  public UserGroup getUserGroup(Object id) {
    try {
      return userGroupsDAO.readUserGroup(id);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.usergroup"), id), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<UserGroup> getAllUserGroups() {
    try {
      return userGroupsDAO.readAllUserGroups();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.usergroups"), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode criar grupos de usurios.
   * 
   * @throws PermissionException Caso o usurio no possa criar grupos d
   *         usurios.
   */
  protected void checkCreateUserGroupPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserGroup createUserGroup(UserGroupInfo info) {
    checkCreateUserGroupPermission();
    try {
      UserGroup userGroup = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          userGroup = userGroupsDAO.createUserGroup(info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.CREATE, userGroup);
      return userGroup;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.create.usergroup"), info.name), e
          .getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode alterar grupos de usurios.
   * 
   * @throws PermissionException Caso o usurio no possa alterar grupos de
   *         usurios.
   */
  protected void checkModifyUserGroupPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserGroup modifyUserGroup(Object id, UserGroupInfo info) {
    checkModifyUserGroupPermission();
    try {
      UserGroup userGroup = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          userGroup = userGroupsDAO.modifyUserGroup(id, info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.MODIFY, userGroup);
      return userGroup;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.modify.usergroup"), id), e.getMessage(),
        e);
    }
  }

  /**
   * Verifica se o usurio pode remover grupos de usurios.
   * 
   * @throws PermissionException Caso o usurio no possa remover grupos de
   *         usurios.
   */
  protected void checkDeleteUserGroupPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteUserGroup(Object id) throws RemoteException,
    AdministrationDeleteException {
    checkDeleteUserGroupPermission();
    try {
      UserGroup userGroup = null;
      synchronized (transaction) {
        boolean userInGroup = false;
        checkTransaction();
        try {
          lock(this);
          for (User user : User.getAllUsers()) {
            /*
             * Verifica se o user eh o admin, pois o admin nao possui UserGroup.
             */
            if (!user.getId().equals(User.getAdminId())) {
              if (UserGroup.getUserGroupFromUser(user).getId().equals(id)) {
                userInGroup = true;
                break;
              }
            }
          }
          if (userInGroup) {
            throw new AdministrationDeleteException(String.format(getString(
              "AdministrationService.error.delete.usergroup.having.user"), id));
          }
          userGroup = UserGroup.getUserGroup(id);
          if (userGroup == null) {
            throw new ServiceFailureException(String.format(getString(
              "AdministrationService.error.invalid.usergroup"), id));
          }
          userGroupsDAO.deleteUserGroup(id);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.DELETE, userGroup);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.delete.usergroup"), id), e.getMessage(),
        e);
    }
  }

  //
  // Controle de Role
  //
  /**
   * {@inheritDoc}
   */
  @Override
  public List<Role> getAllRoles() {
    try {
      return roleDAO.readAllRoles();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.roles"), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Role getRole(Object id) {
    try {
      return roleDAO.readRole(id);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.role"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode criar perfis.
   * 
   * @throws PermissionException Caso o usurio no possa criar perfis.
   */
  protected void checkCreateRolePermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Role createRole(RoleInfo info) {
    checkCreateRolePermission();
    try {
      Role role = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          role = roleDAO.createRole(info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.CREATE, role);
      return role;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.create.role"), info.name), e.getMessage(),
        e);
    }
  }

  /**
   * Verifica se o usurio pode alterar perfis.
   * 
   * @throws PermissionException Caso o usurio no possa alterar perfis.
   */
  protected void checkModifyRolePermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Role modifyRole(Object id, RoleInfo info) {
    checkModifyRolePermission();
    try {
      Role role = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          role = roleDAO.modifyRole(id, info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.MODIFY, role);
      return role;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.modify.role"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode remover perfis.
   * 
   * @throws PermissionException Caso o usurio no possa remover perfis.
   */
  protected void checkDeleteRolePermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteRole(Object id) throws AdministrationDeleteException,
    RemoteException {
    checkDeleteRolePermission();
    try {
      Role role = null;
      synchronized (transaction) {
        boolean userInRole = false;
        checkTransaction();
        try {
          lock(this);
          userInRole = User.hasAnyUserWithRole(id);
          if (userInRole) {
            throw new AdministrationDeleteException(String.format(getString(
              "AdministrationService.error.delete.role.having.user"), id));
          }
          role = Role.getRole(id);
          if (role == null) {
            throw new ServiceFailureException(String.format(getString(
              "AdministrationService.error.invalid.role"), id));
          }
          roleDAO.deleteRole(id);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.DELETE, role);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.delete.role"), id), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<String> getPermissionClasses() {
    final String propName = "permission.class";
    return getStringListProperty(propName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Permission> getAllPermissions() {
    try {
      return permissionDAO.readAllPermissions();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.permissions"), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode obter permisses com senhas.
   * 
   * @throws PermissionException Caso o usurio no possa obter permisses com
   *         senhas.
   */
  protected void checkGetPermissionWithPassword() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Permission> getAllPermissionsWithPasswords() {
    checkGetPermissionWithPassword();
    try {
      List<Permission> permissions = permissionDAO.readAllPermissions();
      for (int i = 0; i < permissions.size(); i++) {
        if (permissions.get(i) instanceof UserPasswordPermission) {
          UserPasswordPermission upp =
            (UserPasswordPermission) permissions.get(i);
          upp.setPassword(upp.getLocalPassword());
        }
      }
      return permissions;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.permissions"), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission getPermission(Object id) {
    try {
      return permissionDAO.readPermission(id);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.permission"), id), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission getPermissionWithPassword(Object id) {
    checkGetPermissionWithPassword();
    try {
      Permission permission = permissionDAO.readPermission(id);
      if (permission instanceof UserPasswordPermission) {
        UserPasswordPermission upp = (UserPasswordPermission) permission;
        upp.setPassword(upp.getLocalPassword());
      }
      return permission;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.permission"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode criar permisses.
   * 
   * @throws PermissionException Caso o usurio no possa criar permisses.
   */
  protected void checkCreatePermissionPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission createPermission(Permission permission) {
    checkCreatePermissionPermission();
    try {
      Permission result = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          result = permissionDAO.createPermission(permission);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.CREATE, result);
      return result;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.create.permission"), permission.getName()),
        e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode alterar permisses.
   * 
   * @throws PermissionException Caso o usurio no possa alterar permisses.
   */
  protected void checkModifyPermissionPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission modifyPermission(Object id, Permission permission) {
    checkModifyPermissionPermission();
    try {
      Permission result = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          result = permissionDAO.modifyPermission(id, permission);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.MODIFY, result);
      return result;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.modify.permission"), id), e.getMessage(),
        e);
    }
  }

  /**
   * Verifica se o usurio pode remover permisses.
   * 
   * @throws PermissionException Caso o usurio no possa remover permisses.
   */
  protected void checkDeletePermissionPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deletePermission(Object id) throws AdministrationDeleteException,
    RemoteException {
    checkDeletePermissionPermission();
    try {
      Permission permission = null;
      synchronized (transaction) {
        checkTransaction();
        boolean userInPermission = false;
        boolean roleInPermission = false;
        try {
          lock(this);
          userInPermission = User.hasAnyUserWithPermission(id);
          roleInPermission = Role.hasAnyRoleWithPermission(id);
          if (userInPermission || roleInPermission) {
            throw new AdministrationDeleteException(String.format(getString(
              "AdministrationService.error.delete.permission.having.user"),
                  id));
          }
          permission = Permission.getPermission(id);
          if (permission == null) {
            throw new ServiceFailureException(String.format(getString(
              "AdministrationService.error.invalid.permission"), id));
          }
          permissionDAO.deletePermission(id);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.DELETE, permission);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.delete.permission"), id), e.getMessage(),
        e);
    }
  }

  //
  // Controle de Platform
  /**
   * {@inheritDoc}
   */
  @Override
  public List<Platform> getAllPlatforms() {
    try {
      return platformDAO.readAllPlatforms();
    }
    catch (DAOException e) {
      throw new ServiceFailureException(getString(
        "AdministrationService.error.get.platforms"), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Platform getPlatform(Object id) {
    try {
      return platformDAO.readPlatform(id);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.get.platform"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode criar plataformas.
   * 
   * @throws PermissionException Caso o usurio no possa criar plataformas.
   */
  protected void checkCreatePlatformPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Platform createPlatform(PlatformInfo info) {
    checkCreatePlatformPermission();
    try {
      Platform platform = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          platform = platformDAO.createPlatform(info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.CREATE, platform);
      return platform;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.create.platform"), info.name), e
        .getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode alterar plataformas.
   * 
   * @throws PermissionException Caso o usurio no possa alterar plataformas.
   */
  protected void checkModifyPlatformPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Platform modifyPlatform(Object id, PlatformInfo info) {
    checkModifyPlatformPermission();
    try {
      Platform platform = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          platform = platformDAO.modifyPlatform(id, info);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.MODIFY, platform);
      return platform;
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.modify.platform"), id), e.getMessage(), e);
    }
  }

  /**
   * Verifica se o usurio pode remover plataformas.
   * 
   * @throws PermissionException Caso o usurio no possa remover plataformas.
   */
  protected void checkDeletePlatformPermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deletePlatform(Object id) throws RemoteException {
    checkDeletePlatformPermission();
    try {
      Platform platform = null;
      synchronized (transaction) {
        checkTransaction();
        try {
          lock(this);
          platform = Platform.getPlatform(id);
          if (platform == null) {
            throw new ServiceFailureException(String.format(getString(
              "AdministrationService.error.invalid.platform"), id));
          }
          platformDAO.deletePlatform(id);
        }
        finally {
          unlock(this);
        }
      }
      notifyObservers(AdministrationEvent.DELETE, platform);
    }
    catch (DAOException e) {
      throw new ServiceFailureException(String.format(getString(
        "AdministrationService.error.delete.platform"), id), e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isAlive() throws RemoteException {
    return isActive();
  }

  /**
   * Envia um evento administrativo para todos os clientes ativos.
   * 
   * @param event o evento administrativo a ser enviado
   */
  private void sendEvent(AdministrationEvent event) {
    MessageService.getInstance().sendToAll(new Message(event));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLocale(Locale locale) {
    Service.setUserLocale(locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getCurrentLocale() {
    Locale locale = getThreadLocale();
    if (locale == null) {
      locale = getDefaultLocale();
    }
    return locale;
  }

}
