/*
 * $Id: EventLogService.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.server.services.eventlogservice;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import csbase.logic.User;
import csbase.logic.eventlogservice.LogsInfo;
import csbase.remote.EventLogServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.loginservice.LoginService;
import tecgraf.javautils.core.io.FileUtils;

/**
 * A classe <code>EventLogService</code> implementa o servio de log de eventos,
 * que precisem ser auditados no servidor. O servio apresenta o conceito de
 * fila ou grupo de log que  definido hierarquicamente com um array de strings,
 * podendo atingir qualquer nvel de detalhe.
 *
 * @author Tecgraf/PUC-Rio
 */
public class EventLogService extends Service
  implements EventLogServiceInterface {
  /**
   * Nome do diretrio raiz onde so gravadas as filas de eventos.
   */
  private String eventDirName = null;

  /**
   * Nome da fila padro, caso seja enviado algum log-cliente sem a correta
   * definio de uma fila.
   */
  final private static String NO_NAME = "no_name";

  /**
   * Campo que armazena o sufixo dos arquivos de log-cliente.
   */
  final private static String FILE_SUFIX = "log";

  /**
   * Criao de um identificador de arquivo para auditagem com a data fornecida
   * como parmetro.
   *
   *
   * @param date data para identificar o arquivo
   *
   * @return uma string que representa um nmero nico.
   */
  private String createId(Date date) {
    final String format = "{0,date,yyyy-MM-dd}";
    final MessageFormat formatter = new MessageFormat(format);
    Object[] args = new Object[1];
    args[0] = date;
    StringBuffer text = new StringBuffer();
    formatter.format(args, text, null);
    return text.toString();
  }

  /**
   * Mtodo de montagem do nome do arquivo de log com base na fila. Cria um novo
   * Id nico para o arquivo com data atual.
   *
   * @param queue fila de armazenamento
   * @return um nome de arquivo que define o log do grupo ou fila concatenado
   *         com <code>.xxx</code> (sufixo de arquivo definido); ou {@code null}
   *         em caso de falha.
   */
  final private String getQueueFileName(final String[] queue) {
    final String id = createId(new Date());
    return getQueueFileName(queue, id);
  }

  /**
   * Mtodo de montagem do nome do arquivo de log com base na fila e no id
   * fornecido.
   *
   * @param queue fila de armazenamento
   * @param id identificador do arquivo
   * @return um nome de arquivo que define o log do grupo ou fila concatenado
   *         com <code>.xxx</code> (sufixo de arquivo definido); ou {@code null}
   *         em caso de falha.
   */
  final private String getQueueFileName(final String[] queue, String id) {
    final String sep = "/";
    final String pt = ".";
    if (queue == null || queue.length < 1) {
      return eventDirName + sep + NO_NAME + pt + id + pt + FILE_SUFIX;
    }
    final int n = queue.length;
    if (n == 1) {
      final String fName = (queue[0] == null ? NO_NAME : queue[0]);
      return fName + pt + id + pt + FILE_SUFIX;
    }

    int min = 4;
    if (n < 5) {
      min = n - 1;
    }
    StringBuffer baseDirName = new StringBuffer(eventDirName);
    int i;
    for (i = 0; i < min; i++) {
      String dName = (queue[i] == null ? NO_NAME : queue[i]);
      baseDirName.append(sep + dName);
      final Server server = Server.getInstance();
      try {
        server.checkDirectory(baseDirName.toString());
      }
      catch (ServerException e) {
        return null;
      }
    }
    if (n < 5) {
      String fName = (queue[i] == null ? NO_NAME : queue[i]);
      return baseDirName + sep + fName + pt + id + pt + FILE_SUFIX;
    }
    String fName = baseDirName + sep;
    final String xName = (queue[4] == null ? NO_NAME : queue[4]);
    fName = fName + xName;
    for (i = 5; i < n; i++) {
      final String qName = (queue[i] == null ? NO_NAME : queue[i]);
      fName = fName + pt + qName;
    }
    return fName + pt + id + pt + FILE_SUFIX;
  }

  /**
   * Mtodo de montagem do nome de uma fila ou grupo de log-cliente, com base no
   * array de strings que o define. Caso o array seja invlido ou inconsistente,
   * ser retornado o nome padro.
   *
   * @param q array de strings da fila.
   * @return um nome da fila
   */
  private String getQueueName(final String[] q) {
    final String bar = "|";
    if (q == null || q.length <= 0) {
      return NO_NAME;
    }
    String qName = q[0];
    final int n = q.length;
    for (int i = 1; i < n; i++) {
      qName = qName + bar + q[i];
    }
    return "[" + qName + "]";
  }

  /**
   * Definio padro do mtodo abstrato de <code>Service</code>.
   */
  @Override
  final public boolean has2Update(Object o1, Object o2) {
    return true;
  }

  /**
   * Retorna a instncia do servio no lado servidor.
   *
   * @return o objeto <code>EventLogService</code>
   */
  public static EventLogService getInstance() {
    return (EventLogService) getInstance(SERVICE_NAME);
  }

  /**
   * Adio de informao de eventlog relativa ao cliente.
   *
   * @param queue a fila de dados a ser gravada (que ter o prefixo
   *        <code>"client"</code>.
   * @param info o array de informaes.
   * @return um indicativo do correto registro.
   */
  @Override
  public boolean addClientInformation(final String[] queue,
    final String[] info) {
    String[] reQueue = null;
    if (queue == null || queue.length <= 0) {
      return false;
    }

    final int N = queue.length;
    reQueue = new String[N + 1];
    reQueue[0] = "client";
    for (int i = 0; i < N; i++) {
      reQueue[i + 1] = queue[i];
    }
    final boolean added = addInformation(reQueue, info);
    return added;
  }

  /**
   * Adio de informao de eventlog relativa ao servidor.
   *
   * @param queue a fila de dados a ser gravada (que ter o prefixo
   *        <code>"server"</code>.
   * @param info o array de informaes.
   * @return um indicativo do correto registro.
   */
  public boolean addServerInformation(final String[] queue,
    final String[] info) {
    String[] reQueue = null;
    if (queue == null || queue.length <= 0) {
      return false;
    }

    final int N = queue.length;
    reQueue = new String[N + 1];
    reQueue[0] = "server";
    for (int i = 0; i < N; i++) {
      reQueue[i + 1] = queue[i];
    }
    final boolean added = addInformation(reQueue, info);
    return added;
  }

  /**
   * Mtodo efetivo de gravao (cliente e servidor).
   *
   * @param queue fila
   * @param info informao
   * @return indicativo de sucesso.
   */
  private boolean addInformation(final String[] queue, final String[] info) {
    final String queueName = getQueueName(queue);
    final User user = Service.getUser();
    final String userName = (user == null ? " - "
      : user.getId().toString()
        + LoginService.getInstance().getRealUserForLog(Service.getKey()));
    final String fileName = getQueueFileName(queue);
    if (fileName == null) {
      final String err = "<<" + queueName + ">>: Falha de verificao!";
      Server.logSevereMessage(err);
      return false;
    }

    final File file = new File(fileName);
    if (!file.exists()) {
      final boolean created =
        createAndWriteHeaderForNewFile(queueName, fileName, file);
      if (!created) {
        return false;
      }
    }

    if (file.isDirectory()) {
      final String err = "<<" + queueName + ">>: Escrita em diretrio!";
      Server.logSevereMessage(err);
      return false;
    }

    if (!file.canWrite()) {
      final String err = "<<" + queueName + ">>: Sem permisso de escrita";
      Server.logSevereMessage(err);
      return false;
    }

    final boolean written = writeInfoToFile(info, userName, file);
    return written;
  }

  /**
   * Gravao no arquivo da informao.
   *
   * @param info informativo
   * @param userName nome do usurio
   * @param file arquivo a ser gravado.
   * @return indicativo
   */
  private boolean writeInfoToFile(final String[] info, final String userName,
    final File file) {
    final Server server = Server.getInstance();
    final Charset charset = server.getSystemDefaultCharset();
    try (FileOutputStream stream = new FileOutputStream(file, true);
         OutputStreamWriter writer = new OutputStreamWriter(stream, charset)) {
      final Date dt = new Date();
      final long tm = dt.getTime();
      writer.write(tm + " ; " + dt + " ; " + userName);

      String[] data = info;
      if (info == null || info.length == 0) {
        data = new String[] { "*" };
      }

      for (int i = 0; i < data.length; i++) {
        writer.write(" ; " + data[i]);
      }
      writer.write("\n");
      writer.flush();
    }
    catch (IOException ioe) {
      final String path = file.getAbsolutePath();
      final String err = "Falha na gravao de evento em " + path;
      Server.logSevereMessage(err, ioe);
      return false;
    }
    return true;
  }

  /**
   * @param queueName
   * @param fileName
   * @param file
   * @return indicativo de arquivo criado e de header gravado.
   */
  private boolean createAndWriteHeaderForNewFile(final String queueName,
    final String fileName, final File file) {
    final String path = file.getAbsolutePath();
    Server.logInfoMessage("Pedido de criao de: " + path);
    try {
      file.createNewFile();
      Server.logInfoMessage("Arquivo criado: " + path);
    }
    catch (IOException ioe) {
      Server.logSevereMessage("Falha na criao: " + path + " - ", ioe);
      return false;
    }

    final String dt = (new Date()).toString();
    final Server server = Server.getInstance();
    final Charset charset = server.getSystemDefaultCharset();
    try(FileOutputStream stream = new FileOutputStream(file);
        OutputStreamWriter writer = new OutputStreamWriter(stream, charset)) {
      writer.write("# Arquivo de auditagem de eventos CSBASE.\n");
      writer.write("# Arquivo: [" + queueName + "]\n");
      writer.write("# Abertura em: " + dt + "\n");
      writer.write("# \n");
      Server.logInfoMessage("Header gravado em: " + path);
    }
    catch (IOException ioe) {
      final String err = "Falha na gravao de header: " + path;
      Server.logSevereMessage(err, ioe);
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LogsInfo find(final String[][] queues, final Date initialDate,
    final Date finalDate) {

    final Server server = Server.getInstance();
    final Charset charset = server.getSystemDefaultCharset();

    EventLogParser logParser = new EventLogParser(initialDate, finalDate);
    List<String> ids = createIdList(initialDate, finalDate);

    for (int i = 0; i < queues.length; i++) {

      String[] fileQueue = queues[i];
      for (String fileId : ids) {

        final String fileName = getQueueFileName(fileQueue, fileId);
        final File file = new File(fileName);
        if (!file.exists()) {
          // no existe arquivo de log para fila e id indicados
          continue;
        }

        try {
          FileInputStream fileStream = new FileInputStream(fileName);
          logParser.parse(fileStream, charset);
          fileStream.close();
        }
        catch (Exception e) {
          final String queueName = getQueueName(fileQueue);
          final String err =
            "<<" + queueName + ">>: Ignorando arquivo de log " + fileName + ".";
          Server.logInfoMessage(err);
        }
      }
    }
    return logParser.getInfo();
  }

  /**
   *
   * Criao de uma lista de identificadores de arquivo para auditagem usando as
   * data inicial e final fornecida como parmetro.
   *
   * @param initialDate data inicial para identificar os arquivos
   * @param finalDate data final para identificar os arquivos
   *
   * @return uma lista de strings que representam os identificadores.
   */
  private List<String> createIdList(final Date initialDate,
    final Date finalDate) {
    List<String> list = new ArrayList<String>();
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(initialDate);
    while (!calendar.getTime().after(finalDate)) {
      String id = createId(calendar.getTime());
      list.add(id);
      calendar.add(Calendar.DAY_OF_MONTH, 1);
    }
    return list;
  }

  /**
   * Inicializao do servio.
   *
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  final public void initService() throws ServerException {
    eventDirName = getStringProperty("directory");
    Server.getInstance().checkDirectory(eventDirName);
  }

  /**
   * Trmino do servio.
   */
  @Override
  final public void shutdownService() {
  }

  /**
   * Constri a instncia do servio, atravs de seu construtor
   * <code>protected</code>.
   *
   * @throws ServerException se houver exceo na instanciao.
   */
  public static void createService() throws ServerException {
    new EventLogService();
  }

  /**
   * Construtor padro que somente define o nome do servio com base em
   * <code>EventLogServiceInterface.SERVICE_NAME</code>.
   *
   * @throws ServerException se houver falha na construo do servio.
   */
  protected EventLogService() throws ServerException {
    super(SERVICE_NAME);
  }
}
