/*
 * FileConfig.java
 * 
 * $Author: cviana $ $Date: 2017-03-13 16:58:16 -0300 (Mon, 13 Mar 2017) $
 * $Revision: 179359 $
 */
package csbase.server.services.projectservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;

import csbase.exception.ServiceFailureException;
import csbase.logic.ProjectFileType;
import csbase.server.Server;

/**
 * Modela a configurao de um arquivo. A configurao descreve um arquivo na
 * rea de projetos.
 */
class FileConfig {
  /**
   * Extenso dos arquivos de configurao.
   */
  private static final String CONFIG_EXTENSION = ".csbase";

  /**
   * Extenso dos arquivos de descrio.
   */
  static final String DESCRIPTION_EXTENSION = ".csbase_description";

  /**
   * Constante de configurao indicativa de tipo do arquivo.
   */
  protected static final String TYPE = "TYPE";

  /**
   * Constante de configurao indicativa de arquivo em construo.
   */
  protected static final String UNDERCONSTRUCTION = "UNDERCONSTRUCTION";

  /**
   * Constante de configurao indicativa do criador do arquivo.
   */
  protected static final String CREATEDBY = "CREATEDBY";

  /**
   * Constante de configurao indicativa da data de criao do arquivo.
   */
  protected static final String CREATIONDATE = "CREATIONDATE";

  /**
   * Indica as informaes sobre um arquivo atualizvel.
   */
  private static final String UPDATABLEFILEINFO = "UPDATABLEFILEINFO";

  /**
   * Prefixo de arquivos de configurao
   */
  private static final String PREFIX = ".";

  /**
   * Prefixo para os arquivos de controle temporrios criados pelo sistema de
   * arquivos remoto do UNIX. Estes so removidos automaticamente pelo NFS e
   * precisam ser ignorados, caso existam.
   */
  private static final String NFS_PREFIX = ".nfs";

  /**
   * Representa o arquivo original no sistema de arquivos.
   */
  protected File file;

  /**
   * Representa o arquivo de configurao no sistema de arquivos.
   */
  protected File configFile;

  /**
   * Representa o diretrio no qual est este arquivo. Caso seja o diretrio
   * raiz, seu valor  <code>null</code>.
   */
  protected ServerProjectFile parent;

  /**
   * Mantm o estado do arquivo: se est em construo ou no.
   */
  protected boolean isUnderConstruction;

  /**
   * Indica o tipo do arquivo.
   */
  protected String type;

  /**
   * Usurio que criou ou solicitou a criao do arquivo.
   */
  protected Object createdBy;

  /**
   * Data de criao do arquivo.
   */
  protected long creationDate;

  /**
   * Informaes referentes a arquivos atualizveis.
   */
  protected UpdatableFileInfo updatableFileInfo;

  /**
   * Carrega as informaes do arquivo de configurao.
   * 
   * @return true, caso o arquivo tenha sido carregado, ou false, caso
   *         contrrio.
   */
  public boolean readConfiguration() {
    if (!configFile.isFile()) {
      Server.logFineMessage("FileConfig:readConfigurationFile: "
        + configFile.getAbsolutePath() + " no  arquivo ou no existe");
      return false;
    }
    try (ObjectInputStream in = new ObjectInputStream(new DataInputStream(
      new BufferedInputStream(new FileInputStream(configFile))))) {
      boolean flag = true;
      while (flag) {
        Object name;
        Object value;
        try {
          name = in.readObject();
          if (name == null) {
            flag = false;
            continue;
          }
          value = in.readObject();
        }
        catch (Exception e) {
          Server.logSevereMessage(
            "Erro na leitura de uma propriedade: " + " do arquivo de " +
              "configuracao: " + configFile
              .getAbsolutePath(), e);
          break;
        }
        if (name.equals(TYPE)) {
          type = (String) value;
        }
        else if (name.equals(UNDERCONSTRUCTION)) {
          isUnderConstruction = ((Boolean) value).booleanValue();
        }
        else if (name.equals(CREATEDBY)) {
          createdBy = value;
        }
        else if (name.equals(CREATIONDATE)) {
          creationDate = ((Long) value).longValue();
        }
        else if (name.equals(UPDATABLEFILEINFO)) {
          this.updatableFileInfo = (UpdatableFileInfo) value;
        }
        else {
          Server.logInfoMessage(
            "FileConfig:readConfigurationFile: " + name + " ignorado em " +
              configFile
              .getAbsolutePath());
        }
      }
      if (file.isDirectory()) {
        if (ProjectFileType.UNKNOWN.equals(type)) {
          type = ProjectFileType.DIRECTORY_TYPE;
          writeConfiguration();
        }
      }
      return true;
    }
    catch (EOFException e) {
      // Arquivo foi corrompido. Atribui valores default e escreve arquivo
      type = file
        .isDirectory() ? ProjectFileType.DIRECTORY_TYPE : ProjectFileType
        .UNKNOWN;
      isUnderConstruction = false;
      createdBy = "admin";
      creationDate = (new Date()).getTime();
      writeConfiguration();
      return true;
    }
    catch (Exception e) {
      Server.logSevereMessage("FileConfig:readConfigurationFile: " + configFile
        .getAbsolutePath() + " corrompido.", e);
      return false;
    }
  }

  /**
   * Grava as informaes do arquivo de configuraes.
   */
  public void writeConfiguration() {
    try (ObjectOutputStream out = new ObjectOutputStream(new DataOutputStream(
      new BufferedOutputStream(new FileOutputStream(configFile))))) {
      out.writeObject(TYPE);
      out.writeObject(new String(type));
      out.writeObject(UNDERCONSTRUCTION);
      out.writeObject(new Boolean(isUnderConstruction));
      out.writeObject(CREATEDBY);
      out.writeObject(createdBy);
      out.writeObject(CREATIONDATE);
      out.writeObject(new Long(creationDate));
      out.writeObject(UPDATABLEFILEINFO);
      out.writeObject(this.updatableFileInfo);
      out.writeObject(null); // EOF
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "FileConfig:writeConfigurationFile: " + "erro ao gerar " + configFile
          .getAbsolutePath(), e);
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Obtm a data de criao do arquivo.
   * 
   * @return data de criao em milesegundos.
   */
  public long getCreationDate() {
    return creationDate;
  }

  /**
   * Obtm o arquivo correspondente a essa configurao.
   * 
   * @return O arquivo descrito pela configurao.
   */
  public File getFile() {
    return file;
  }

  /**
   * Obtm o arquivo que armazena a configurao em si.
   * 
   * @return O arquivo de configurao.
   */
  public File getConfigFile() {
    return configFile;
  }

  /**
   * Informa se o arquivo indicado  um arquivo de configurao.
   * 
   * @param file O arquivo.
   * @return Verdadeiro se  um arquivo de configurao, falso caso contrrio.
   */
  public static boolean isConfigFile(File file) {
    return isConfigFileName(file.getName());
  }

  /**
   * Informa se o arquivo indicado  um arquivo de descrio.
   * 
   * @param file O arquivo.
   * @return Verdadeiro se  um arquivo de descrio, falso caso contrrio
   */
  public static boolean isDescriptionFile(File file) {
    return isDescriptionFileName(file.getName());
  }

  /**
   * Informa se o nome  de um arquivo de configurao.
   * 
   * @param fileName O nome do arquivo.
   * @return Verdadeiro se  de um arquivo de configurao, falso caso
   *         contrrio.
   */
  public static boolean isConfigFileName(String fileName) {
    return fileName.startsWith(FileConfig.PREFIX)
      && fileName.endsWith(FileConfig.CONFIG_EXTENSION);
  }

  /**
   * Informa se o nome  de um arquivo de descrio.
   * 
   * @param fileName O nome do arquivo.
   * @return Verdadeiro se  de um arquivo de descrio, falso caso contrrio.
   */
  public static boolean isDescriptionFileName(String fileName) {
    return fileName.startsWith(FileConfig.PREFIX)
      && fileName.endsWith(FileConfig.DESCRIPTION_EXTENSION);
  }

  /**
   * Informa se o nome  de um arquivo de controle temporrio do UNIX.
   * 
   * @param fileName O nome do arquivo.
   * @return Verdadeiro se  de um arquivo de controle temporrio, falso caso
   *         contrrio.
   */
  public static boolean isUnixFileName(String fileName) {
    return fileName.startsWith(FileConfig.NFS_PREFIX);
  }

  /**
   * Indica se o nome  de um arquivo de controle (configurao ou descrio) ou
   * um arquivo de controle temporrio do NFS UNIX.
   * 
   * @param fileName O nome do arquivo.
   * @return Verdadeiro se  de um arquivo de controle, falso caso contrrio.
   */
  public static boolean isControlledFileName(String fileName) {
    return isConfigFileName(fileName) || isDescriptionFileName(fileName)
      || isUnixFileName(fileName);
  }

  /**
   * Obtm um objeto<code>File</code> renomeando um outro <code>File</code>. O
   * arquivo no  renomeado fisicamente. Apenas  criada um objeto
   * <code>File</code> que representa esse arquivo com outro nome.
   * 
   * @param file O objeto <code>File</code> original.
   * @param name O nome do arquivo representado pelo novo objeto
   *        <code>File</code>.
   * 
   * @return Um <code>File</code> que representa o novo nome.
   */
  protected static File renamedFile(File file, String name) {
    String fileName = file.getName();
    String filePath = file.getAbsolutePath();
    String dir = filePath.substring(0, filePath.length() - fileName.length());
    return new File(dir + name);
  }

  /**
   * Cria um novo <code>File</code> com nome de configurao.
   * 
   * @see #createConfigFileName(String)
   * @param file o arquivo origem
   * @return um novo <code>File</code>
   */
  protected static File renamedConfigFile(File file) {
    String fileName = file.getName();
    return renamedFile(file, createConfigFileName(fileName));
  }

  /**
   * Cria um novo <code>File</code> com nome de arquivo de descrio.
   * 
   * @see #createDescriptionFileName(String)
   * @param file o arquivo origem
   * @return um novo <code>File</code> com nome de arquivo de descrio
   */
  protected static File renamedDescriptionFile(File file) {
    String fileName = file.getName();
    return renamedFile(file, createDescriptionFileName(fileName));
  }

  /**
   * Cria o nome de arquivo de configurao.
   * 
   * @param fileName nome do arquivo de origem
   * @return o nome de arquivo de configurao
   */
  protected static String createConfigFileName(String fileName) {
    String prefix = FileConfig.PREFIX;
    String ext = FileConfig.CONFIG_EXTENSION;
    return prefix + fileName + ext;
  }

  /**
   * Cria o nome de arquivo de descrio.
   * 
   * @param fileName nome do arquivo de origem
   * @return o nome de arquivo de descrio
   */
  protected static String createDescriptionFileName(String fileName) {
    String prefix = FileConfig.PREFIX;
    String ext = FileConfig.DESCRIPTION_EXTENSION;
    return prefix + fileName + ext;
  }

  /**
   * Cria a representao de configurao de um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   */
  protected FileConfig(File file) {
    this.file = file;
    this.configFile = renamedConfigFile(file);
    this.isUnderConstruction = false;
    if (file.isDirectory()) {
      this.type = ProjectFileType.DIRECTORY_TYPE;
    }
    else {
      this.type = ProjectFileType.UNKNOWN;
    }
    this.createdBy = null;
    this.creationDate = 0;
  }
}
