/*
 * $Id: ProjectDataService.java 179359 2017-03-13 19:58:16Z cviana $
 */
package csbase.server.services.projectservice.v1_00;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.omg.CORBA.Any;
import org.omg.CORBA.ORB;

import csbase.exception.ServiceFailureException;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.projectservice.ServerProject;
import csbase.server.services.projectservice.UpdatableFileInfo;
import scs.core.ComponentContext;
import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.MaxClientsReachedException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.ftc.utils.Utils;
import tecgraf.openbus.data_service.DataAccessDenied;
import tecgraf.openbus.data_service.DataDescription;
import tecgraf.openbus.data_service.DataKey;
import tecgraf.openbus.data_service.DataNotFound;
import tecgraf.openbus.data_service.DataView;
import tecgraf.openbus.data_service.IHierarchicalDataServiceHelper;
import tecgraf.openbus.data_service.IHierarchicalDataServiceOperations;
import tecgraf.openbus.data_service.IHierarchicalNavigationDataService;
import tecgraf.openbus.data_service.InvalidDataKey;
import tecgraf.openbus.data_service.Metadata;
import tecgraf.openbus.data_service.ServiceFailure;
import tecgraf.openbus.data_service.UnknownViewInterface;
import tecgraf.openbus.data_service.UnknownViews;
import tecgraf.openbus.data_service.UnstructuredData;
import tecgraf.openbus.data_service.UnstructuredDataHelper;
import tecgraf.openbus.data_service.UnstructuredDataImpl;
import tecgraf.openbus.data_service.project.ProjectItemDescriptionImpl;
import tecgraf.openbus.project.PATH_SEPARATOR;
import tecgraf.openbus.project.ProjectItemDescription;

/**
 * Representa um Servio de Dados de Projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectDataService implements IHierarchicalDataServiceOperations {
  /**
   * Prefixo utilizado nos nomes das chaves de traduo.
   */
  private static final String LNG_PREFIX = "ProjectDataService.";
  /**
   * Array vazio de descritores de dados. Utilizado principalmente para retorno.
   */
  private static final DataDescription[] EMPTY_DATA_DESCRIPTION_ARRAY =
    new DataDescription[0];
  /**
   * O nome do metadado que representa o caminho absoluto de um dado.
   */
  private static final String ABSOLUTE_PATH_METADATUM_NAME = "ABSOLUTE_PATH";
  /**
   * A instncia nica do Servio de Dados de Projeto.
   */
  private static ProjectDataService instance;

  /**
   * Cria um Servico de Dados de Projeto.
   */
  private ProjectDataService() {
  }

  /**
   * Obtm a instncia nica do Servio de Dados de Projeto.
   * 
   * @return A instncia nica do Servio de Dados de Projeto.
   */
  public static ProjectDataService getInstance() {
    if (instance == null) {
      instance = new ProjectDataService();
    }
    return instance;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription[] getRoots() throws ServiceFailure {
    User user = Service.getUser();
    if (user == null) {
      Server.logSevereMessage(MessageFormat.format("Usurio {0} no existe.",
        user));
      throw new ServiceFailure();
    }

    Object userId = user.getId();

    ProjectService projectService = ProjectService.getInstance();
    List<UserProjectInfo> projects = new ArrayList<UserProjectInfo>();
    try {
      projects.addAll(projectService.getProjectsFromUser(userId));
      projects.addAll(projectService.getProjectsSharedWithUser(userId));
    }
    catch (ServiceFailureException e) {
      Server.logSevereMessage(MessageFormat.format(
        "Ocorreu um erro ao obter os nomes dos projetos do usurio {0}",
        new Object[] { userId }), e);
      throw new ServiceFailure();
    }
    if (projects.size() == 0) {
      return EMPTY_DATA_DESCRIPTION_ARRAY;
    }
    List<DataDescription> roots =
      new ArrayList<DataDescription>(projects.size());
    for (UserProjectInfo projectInfo : projects) {
      String projectName = projectInfo.getProjectName();
      try {
        CommonClientProject project =
          projectService.openProject(projectInfo.getProjectId(), false);
        ClientProjectFile rootFile = project.getRoot();
        try {
          roots.add(this.createDataDescription(rootFile));
        }
        catch (InvalidDataKey e) {
          Server.logSevereMessage(MessageFormat.format(
            "Erro ao criar descrio para o arquivo {0}.",
            new Object[] { rootFile.getStringPath() }), e);
          throw new ServiceFailure();
        }
      }
      catch (ServiceFailureException e) {
        Server.logSevereMessage(MessageFormat.format(
          "Ocorreu um erro ao abrir o projeto {0} do usurio {1}.",
          new Object[] { projectName, userId }), e);
        throw new ServiceFailure();
      }
    }
    return roots.toArray(EMPTY_DATA_DESCRIPTION_ARRAY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription[] getChildren(byte[] fKey) throws ServiceFailure,
    InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile parent = getProjectFile(dataKey);
    if (!parent.isDirectory()) {
      throw new ServiceFailure();
    }

    ClientProjectFile[] children = null;
    try {
      children = parent.getChildren();
    }
    catch (RemoteException e) {
      throw new ServiceFailure();
    }

    if (children == null) {
      return EMPTY_DATA_DESCRIPTION_ARRAY;
    }

    List<DataDescription> nodes =
      new ArrayList<DataDescription>(children.length);
    for (int i = 0; i < children.length; i++) {
      nodes.add(this.createDataDescription(children[i]));
    }
    return nodes.toArray(EMPTY_DATA_DESCRIPTION_ARRAY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription getParent(byte[] fKey) throws ServiceFailure,
    InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile file = getProjectFile(dataKey);
    if (file == null) {
      throw new InvalidDataKey(fKey);
    }
    ClientProjectFile parent = file.getParent();
    if (parent == null) {
      return null;
    }
    return this.createDataDescription(parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataDescription getDataDescription(byte[] fKey) throws ServiceFailure,
    InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile file = getProjectFile(dataKey);
    if (file == null) {
      throw new InvalidDataKey(dataKey.getKey());
    }
    return this.createDataDescription(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataView getDataView(byte[] fKey, String fViewInterface)
    throws ServiceFailure, UnknownViewInterface, InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile file = getProjectFile(dataKey);
    if (file == null) {
      throw new InvalidDataKey(dataKey.getKey());
    }
    return this.createDataView(file, fViewInterface);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataView[] getDataViewSeq(byte[][] fKeys, String fViewInterface)
    throws ServiceFailure, UnknownViewInterface, InvalidDataKey, DataNotFound {
    DataView[] viewSeq = new DataView[fKeys.length];
    for (int i = 0; i < fKeys.length; i++) {
      viewSeq[i] = this.getDataView(fKeys[i], fViewInterface);
    }
    return viewSeq;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] createData(byte[] fParentKey, DataDescription fPrototype)
    throws ServiceFailure, InvalidDataKey, DataNotFound {
    DataKey parentKey = new DataKey(fParentKey);
    ClientProjectFile parent = getProjectFile(parentKey);
    if (parent == null) {
      throw new InvalidDataKey(parentKey.getKey());
    }
    if (!parent.isDirectory()) {
      throw new ServiceFailure();
    }

    ProjectItemDescription prototype = (ProjectItemDescription) fPrototype;
    try {
      if (prototype.fIsContainer) {
        parent.createFile(prototype.fName, ProjectFileType.DIRECTORY_TYPE);
      }
      else {
        parent.createFile(prototype.fName, prototype.fType);
      }
    }
    catch (RemoteException e) {
      throw new ServiceFailure();
    }
    catch (ServiceFailureException e) {
      String userLogin = Service.getUser().getLogin();
      String projectName = ServerProject.getProjectName(parent.getProjectId());
      Server
        .logSevereMessage(
          MessageFormat
            .format(
              "Ocorreu um erro ao criar o dado {0} no projeto {1} pertencente ao usurio {2}.",
              new Object[] { prototype.fName, projectName, userLogin }), e);
      throw new ServiceFailure();
    }

    String dataId = generateDataId(parent, prototype.fName);
    DataKey key1 = this.generateDataKey(dataId);
    DataKey key = key1;
    return key.getKey();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] copyData(byte[] fParentKey, byte[] fSourceKey)
    throws ServiceFailure, UnknownViews, DataNotFound, InvalidDataKey,
    DataAccessDenied {
    DataKey parentKey = new DataKey(fParentKey);
    ClientProjectFile parent = getProjectFile(parentKey);
    DataKey sourceKey = new DataKey(fSourceKey);
    ClientProjectFile source = getProjectFile(sourceKey);

    try {
      source.copy(parent);
    }
    catch (RemoteException e) {
      throw new ServiceFailure();
    }

    String dataId = generateDataId(parent, source.getName());
    DataKey dataKey = this.generateDataKey(dataId);
    return dataKey.getKey();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void moveData(byte[] fKey, byte[] fNewParentKey)
    throws ServiceFailure, UnknownViews, InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    DataKey newParentKey = new DataKey(fNewParentKey);
    ClientProjectFile file = getProjectFile(dataKey);
    ClientProjectFile newParent = getProjectFile(newParentKey);
    try {
      file.move(newParent);
    }
    catch (RemoteException e) {
      throw new ServiceFailure();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateData(byte[] fKey, byte[] fSourceKey) throws ServiceFailure,
    UnknownViews, InvalidDataKey, DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    DataKey sourceKey = new DataKey(fSourceKey);
    ClientProjectFile destFile = getProjectFile(dataKey);
    ClientProjectFile sourceFile = getProjectFile(sourceKey);
    if (sourceFile.equals(destFile)) {
      Server.logSevereMessage("Ocorreu uma tentativa de copiar o arquivo "
        + sourceFile.getStringPath() + " sobre ele mesmo.");
      throw new ServiceFailure();
    }
    Object projectId = destFile.getProjectId();
    String[] path = destFile.getPath();
    try (OutputStream out = new BufferedOutputStream(
      ProjectService.getInstance().getOutputStream(projectId, path))) {
      sourceFile.download(out, 8 * 1024, null);
    }
    catch (FileLockedException e) {
      Server.logSevereMessage(MessageFormat.format(
        "No foi possvel atualizar o arquivo {0}, pois o mesmo est bloqueado.",
        new Object[] { destFile.getStringPath() }), e);
      throw new ServiceFailure();
    }
    catch (IOException e) {
      Server.logSevereMessage(MessageFormat
        .format("Erro ao atualizar o arquivo {0}.",
          new Object[] { destFile.getStringPath() }), e);
      throw new ServiceFailure();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteData(byte[] fKey) throws ServiceFailure, InvalidDataKey,
    DataNotFound {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile file = getProjectFile(dataKey);
    if (file == null) {
      throw new InvalidDataKey(fKey);
    }
    if (file.getParent() == null) {
      ProjectService.getInstance().removeProject(file.getProjectId());
    }
    else {
      try {
        file.remove();
      }
      catch (RemoteException e) {
        String userLogin = Service.getUser().getLogin();
        String projectName = ServerProject.getProjectName(file.getProjectId());
        Server
          .logSevereMessage(
            MessageFormat
              .format(
                "Ocorreu um erro ao remover o arquivo {0} no projeto {1} pertencente ao usurio {2}.",
                new Object[] { file.getName(), projectName, userLogin }), e);
        throw new ServiceFailure();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] copyDataFrom(byte[] fParentKey, byte[] fSourceKey)
    throws ServiceFailure, InvalidDataKey, UnknownViews, DataAccessDenied,
    DataNotFound {

    OpenBusService.getInstance().joinChain();

    try {
      ProjectItemDescription parentDescription =
        (ProjectItemDescription) this.getDataDescription(fParentKey);
      String parentAbsolutePath = null;
      for (int i = 0; i < parentDescription.fMetadata.length; i++) {
        if (parentDescription.fMetadata[i].fName
          .equals(ABSOLUTE_PATH_METADATUM_NAME)) {
          Any any = parentDescription.fMetadata[i].fValue;
          parentAbsolutePath = any.extract_string();
        }
      }

      DataKey sourceDataKey = new DataKey(fSourceKey);
      IHierarchicalNavigationDataService sourceDataService =
        FindService.find(sourceDataKey);
      if (sourceDataService == null) {
        Server.logSevereMessage("Fonte de dados no foi encontrada.");
        throw new ServiceFailure();
      }

      ProjectItemDescription sourceDescription =
        (ProjectItemDescription) sourceDataService
          .getDataDescription(fSourceKey);

      String path =
        parentDescription.fPath + PATH_SEPARATOR.value
          + sourceDescription.fName;
      String absolutePath =
        parentAbsolutePath + PATH_SEPARATOR.value + sourceDescription.fName;
      List<Metadata> metadata = createMetadataList(absolutePath);
      long date = System.currentTimeMillis();

      String userLogin = Service.getUser().getLogin();
      ProjectItemDescription prototype =
        new ProjectItemDescriptionImpl(sourceDescription.fName,
          createViewSet(sourceDescription.fIsContainer), metadata, userLogin,
          sourceDescription.fDescription, path, sourceDescription.fType, 0,
          sourceDescription.fIsContainer, sourceDescription.fCanRead,
          sourceDescription.fCanWrite, date, date);

      byte[] key = this.createData(fParentKey, prototype);

      this.updateDataFrom(key, fSourceKey);

      return key;
    }
    finally {
      OpenBusService.getInstance().exitChain();
    }
  }

  /**
   * Cria um conjunto com as vises suportadas pelos dados oferecidos por este
   * servio.
   * 
   * @param container Indica se o dado  um continer.
   * 
   * @return O conjunto de vises.
   */
  private static Set<String> createViewSet(boolean container) {
    Set<String> views = new HashSet<String>();
    if (!container) {
      views.add(UnstructuredDataHelper.id());
    }
    return views;
  }

  /**
   * Cria uma lista de metadados para os dados oferecidos por este servio.
   * 
   * @param absolutePath O caminho absoluto do dado.
   * 
   * @return A lista de metadados.
   */
  private static List<Metadata> createMetadataList(String absolutePath) {
    List<Metadata> metadataList = new ArrayList<Metadata>();

    ORB orb = OpenBusService.getInstance().getORB();
    Any any;

    if (absolutePath != null) {
      any = orb.create_any();
      any.insert_string(absolutePath);
      metadataList.add(new Metadata(ABSOLUTE_PATH_METADATUM_NAME, any));
    }

    return metadataList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateDataFrom(byte[] fKey, byte[] fSourceKey)
    throws ServiceFailure, InvalidDataKey, UnknownViews, DataNotFound,
    DataAccessDenied {
    DataKey dataKey = new DataKey(fKey);
    ClientProjectFile file = getProjectFile(dataKey);

    DataKey sourceDataKey = new DataKey(fSourceKey);
    IHierarchicalNavigationDataService sourceDataService =
      FindService.find(sourceDataKey);
    if (sourceDataService == null) {
      Server.logSevereMessage("Fonte de dados no foi encontrada.");
      throw new ServiceFailure();
    }

    ProjectItemDescription sourceDescription =
      (ProjectItemDescription) sourceDataService
        .getDataDescription(sourceDataKey.getKey());

    UnstructuredData sourceView;
    try {
      sourceView =
        (UnstructuredData) sourceDataService.getDataView(
          sourceDataKey.getKey(), UnstructuredDataHelper.id());
    }
    catch (UnknownViewInterface e1) {
      throw new UnknownViews();
    }

    UnstructuredData view;
    try {
      view =
        (UnstructuredData) this.createDataView(file, UnstructuredDataHelper
          .id());
    }
    catch (UnknownViewInterface e) {
      throw new UnknownViews();
    }

    SyncRemoteFileChannel channel;
    try {
      channel =
        new SyncRemoteFileChannel(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      Server
        .logSevereMessage(
          MessageFormat
            .format(
              "Erro ao converter o identificador do dado de destino ({0}) para bytes.",
              new Object[] { dataKey.getDataId() }), e);
      throw new ServiceFailure();
    }

    SyncRemoteFileChannel sourceChannel;
    try {
      sourceChannel =
        new SyncRemoteFileChannel(sourceDataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), sourceView.fWritable, sourceView.fHost,
          sourceView.fPort, sourceView.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      Server
        .logSevereMessage(
          MessageFormat
            .format(
              "Erro ao converter o identificador do dado de origem ({0}) para bytes.",
              new Object[] { sourceDataKey.getDataId() }), e);
      throw new ServiceFailure();
    }

    try {
      channel.open(false);
      sourceChannel.open(true);
    }
    catch (PermissionException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new DataAccessDenied();
    }
    catch (FileNotFoundException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new ServiceFailure();
    }
    catch (FailureException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new ServiceFailure();
    }
    catch (MaxClientsReachedException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new ServiceFailure();
    }

    try {
      channel.syncTransferFrom(sourceChannel, 0, sourceDescription.fSize);
    }
    catch (PermissionException e) {
      Server.logSevereMessage("Sem permisso para escrever no arquivo.", e);
      throw new ServiceFailure();
    }
    catch (FailureException e) {
      Server.logSevereMessage("Falha na transferncia do arquivo.", e);
      throw new ServiceFailure();
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      Server.logSevereMessage("Tentativa de escrita em arquivo.", e);
      throw new ServiceFailure();
    }
    finally {
      try {
        channel.close();
      }
      catch (FailureException e) {
        e.printStackTrace();
      }
      finally {
        try {
          sourceChannel.close();
        }
        catch (FailureException e) {
        }
      }
    }

    if (!ProjectService.getInstance().setUpdatableFileInfo(
      file,
      new UpdatableFileInfo(OpenBusProjectFileUpdater.class, sourceDataKey
        .getKey()))) {
      Server.logWarningMessage(MessageFormat.format(
        "A informao sobre atualizao do arquivo {0} no foi definida",
        new Object[] { file.getStringPath() }));
    }
  }

  /**
   * Atualiza um arquivo a partir de um dado localizado em um outro servidor.
   * 
   * @param projectId O identificador do projeto.
   * @param pathArray O caminho para o arquivo.
   * @param fSourceKey A chave unvoca do dado de origem, ou seja, o dado
   *        armazenado no outro servidor.
   * 
   * @throws ServiceFailure Caso ocorra uma falha no servio.
   * @throws InvalidDataKey Caso a chave do dado de origem no seja vlida.
   * @throws UnknownViews Caso as vises oferecidas pelo dado de origem no
   *         sejam conhecidas deste servidor.
   * @throws DataNotFound Caso o dado de origem no exista.
   * @throws DataAccessDenied Caso o usurio solicitante no tenha acesso ao
   *         dado de origem
   */
  public void updateDataFrom(Object projectId, String[] pathArray,
    byte[] fSourceKey) throws ServiceFailure, InvalidDataKey, UnknownViews,
    DataNotFound, DataAccessDenied {
    String dataId = generateDataId(projectId, pathArray);
    DataKey dataKey = this.generateDataKey(dataId);
    this.updateDataFrom(dataKey.getKey(), fSourceKey);
  }

  /**
   * Obtm informaes sobre o arquivo de projeto representado por uma chave.
   * 
   * @param dataKey A chave.
   * 
   * @return As informaes sobre o arquivo de projeto.
   * 
   * @throws InvalidDataKey Caso a chave no represente um arquivo.
   * @throws ServiceFailure Caso ocorra uma falha no servio.
   * @throws DataNotFound Caso o arquivo de projeto no exista.
   */
  private static ClientProjectFile getProjectFile(DataKey dataKey)
    throws InvalidDataKey, ServiceFailure, DataNotFound {
    String dataId = dataKey.getDataId();

    String[] splittedKey = dataId.split(ProjectService.FILE_ID_SEPARATOR);
    if (splittedKey.length < ProjectService.MINIMUM_FILE_ID_SIZE) {
      throw new InvalidDataKey(dataKey.getKey());
    }

    Object projectId = splittedKey[0];
    ServerProject serverProject = ServerProject.getProject(projectId);
    Object userId = ServerProject.getOwnerId(projectId);
    CommonClientProject project;
    try {
      project = ProjectService.getInstance().openProject(projectId, false);
    }
    catch (ServiceFailureException e) {
      Server.logSevereMessage(MessageFormat.format(ProjectService.getInstance()
        .getString(LNG_PREFIX + "error_openning_project"), serverProject
        .getName(), userId), e);
      throw new ServiceFailure();
    }

    if (splittedKey.length == ProjectService.MINIMUM_FILE_ID_SIZE) {
      ClientProjectFile file = project.getRoot();
      if (file == null) {
        throw new DataNotFound(new byte[][] { dataKey.getKey() });
      }
      return file;
    }
    else {
      String[] filePath =
        new String[splittedKey.length - ProjectService.MINIMUM_FILE_ID_SIZE];
      for (int i = 0; i < filePath.length; i++) {
        filePath[i] = splittedKey[i + ProjectService.MINIMUM_FILE_ID_SIZE];
      }
      try {
        ClientProjectFile file = project.getFile(filePath);
        if (file == null) {
          throw new DataNotFound(new byte[][] { dataKey.getKey() });
        }
        return file;
      }
      catch (RemoteException e) {
        throw new ServiceFailure();
      }
    }
  }

  /**
   * Gera um identificador nico para um arquivo.
   * 
   * @param file O arquivo.
   * 
   * @return O identificador nico.
   */
  private static String generateDataId(ClientProjectFile file) {
    return generateDataId(file.getProjectId(), file.getPath());
  }

  /**
   * Gera um identificador para um arquivo de projeto.
   * 
   * @param projectId O identificador do arquivo.
   * @param pathArray O caminho para o arquivo de projeto.
   * @return O identificador.
   */
  private static String generateDataId(Object projectId, String[] pathArray) {
    String dataId = (String) projectId;
    for (String path : pathArray) {
      dataId += ProjectService.FILE_ID_SEPARATOR + path;
    }
    return dataId;
  }

  /**
   * Gera um identificador para um arquivo de projeto a partir do seu diretrio
   * de origem.
   * 
   * @param parent O diretrio de origem do arquivo.
   * @param childName O nome do arquivo.
   * 
   * @return O identificador.
   */
  private static String generateDataId(ClientProjectFile parent,
    String childName) {
    String dataId = generateDataId(parent);
    dataId += ProjectService.FILE_ID_SEPARATOR + childName;
    return dataId;
  }

  /**
   * Gera um identificador unvoco a partir de um identificador de um arquivo de
   * projeto.
   * 
   * @param dataId O identificador do arquivo de projeto.
   * 
   * @return O identificador unvoco.
   * 
   * @throws InvalidDataKey Caso ocorra algum erro ao criar o identificador
   *         unvoco.
   */
  private DataKey generateDataKey(String dataId) throws InvalidDataKey {
    ORB orb = OpenBusService.getInstance().getORB();

    ComponentContext context =
      OpenBusService.getInstance().getComponentContext(
        IHierarchicalDataServiceHelper.id());
    String iComponentIOR = orb.object_to_string(context.getIComponent());

    return new DataKey(dataId, IHierarchicalDataServiceHelper.id(), context
      .getComponentId(), null, iComponentIOR);
  }

  /**
   * Cria uma descrio de dado para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * 
   * @return A descrio do dado.
   * 
   * @throws InvalidDataKey Caso no seja possvel criar a chave unvoca do
   *         dado.
   */
  private ProjectItemDescriptionImpl createDataDescription(
    ClientProjectFile file) throws InvalidDataKey {
    String dataId = generateDataId(file);
    DataKey dataKey = this.generateDataKey(dataId);

    String description = null;
    try {
      description = file.getDescription();
    }
    catch (RemoteException e) {
      Server.logWarningMessage(MessageFormat.format(
        "Erro ao obter a descrio do arquivo {0}.", new Object[] { file
          .getStringPath() }));
    }
    catch (ServiceFailureException e) {
      Server.logWarningMessage(MessageFormat.format(
        "Erro ao obter a descrio do arquivo {0}.", new Object[] { file
          .getStringPath() }));
    }

    String[] pathArray = file.getPath();
    String path = "";
    for (int i = 0; i < pathArray.length; i++) {
      path += PATH_SEPARATOR.value + pathArray[i];
    }

    return new ProjectItemDescriptionImpl(dataKey.getKey(), file.getName(),
      createViewSet(file.isDirectory()), createMetadataList(file
        .getStringPath()), (String) file.whoCreated(), description, path, file
        .getType(), file.size(), file.isDirectory(), true, !file.isDirectory(),
      file.getCreationDate(), file.getModificationDate());
  }

  /**
   * Cria uma viso para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * @param viewInterface O nome da interface da viso.
   * 
   * @return A viso.
   * 
   * @throws UnknownViewInterface Caso a viso solicitada no seja oferecida
   *         pelo dado.
   * @throws InvalidDataKey Caso ocorra erro ao criar a chave unvoca do arquivo
   *         de projeto.
   * @throws ServiceFailure Caso ocorra alguma falha inesperada durante a
   *         criao da viso.
   */
  private DataView createDataView(ClientProjectFile file, String viewInterface)
    throws UnknownViewInterface, ServiceFailure, InvalidDataKey {
    if (viewInterface.equals(UnstructuredDataHelper.id())) {
      return this.createUnstructuredData(file);
    }
    throw new UnknownViewInterface();
  }

  /**
   * Cria uma viso no-estruturada para um arquivo de projeto.
   * 
   * @param file O arquivo de projeto.
   * 
   * @return A viso no-estruturada.
   * 
   * @throws UnknownViewInterface Caso a viso solicitada no seja oferecida
   *         pelo dado.
   * @throws InvalidDataKey Caso ocorra erro ao criar a chave unvoca do arquivo
   *         de projeto.
   * @throws ServiceFailure Caso ocorra alguma falha inesperada durante a
   *         criao da viso.
   */
  private UnstructuredData createUnstructuredData(ClientProjectFile file)
    throws UnknownViewInterface, ServiceFailure, InvalidDataKey {
    if (file.isDirectory()) {
      throw new UnknownViewInterface();
    }
    String dataIdString = generateDataId(file);
    DataKey dataKey = this.generateDataKey(dataIdString);
    ProjectService prjSrv = ProjectService.getInstance();
    try {
      RemoteFileChannelInfo accessInfo = prjSrv.createFileChannelInfo(file);
      return new UnstructuredDataImpl(dataKey.getKey(), accessInfo.getHost(),
        accessInfo.getPort(), accessInfo.getKey(), true);
    }
    catch (Exception e) {
      String errorMessage =
        MessageFormat
          .format(
            "Ocorreu um erro durante a criao da viso no estruturada do arquivo {0}",
            new Object[] { file.getStringPath() });
      Server.logSevereMessage(errorMessage, e);
      throw new ServiceFailure();
    }
  }
}
