package csbase.server.services.repositoryservice;

import java.io.IOException;
import java.util.List;

/**
 * Interface que define um repositrio.
 *
 * A idia bsica do repositrio  encapsular todas as operaes feitas sobre os
 * arquivos de repositrio. Alm disso, essa interface permite abstrair qual
 * repositrio est sendo manipulado e tambm a sua localizao. Por fim, todo
 * repositrio  gerenciado pelo servio {@link RepositoryService}.
 *
 * @see IRepositoryFile
 * @see RepositoryService
 *
 * @author Tecgraf/PUC-Rio
 */
public interface IRepository {

  /**
   * Identificador nico do repositrio.
   *
   * @return identificador.
   */
  String getURI();

  /**
   * Retorna o arquivo dado seu caminho no repositrio.
   *
   * @param path caminho.
   * @return arquivo do repositrio ou null caso o arquivo no exista.
   */
  IRepositoryFile getFile(String... path);

  /**
   * Retorna o diretrio dado seu caminho no repositrio.
   *
   * @param path caminho.
   * @return diretrio do repositrio.
   */
  IRepositoryFile getDir(String... path);

  /**
   * Retorna lista de arquivos do repositrio.
   *
   * @return lista de arquivos do repositrio.
   */
  List<IRepositoryFile> listFiles();

  /**
   * Cria um novo arquivo no diretrio especificado. O caminho do diretrio deve
   * ser informado em forma de array para evitar problemas de separadores
   * dependentes de plataforma.
   *
   * Exemplos:
   * 
   * (Diretrio corrente) "./" -> <code>null</code> ou array vazio
   * 
   * (Caminho relativo) "./dir/abc/" -> {"dir", "abc"}
   * 
   * (Caminho absoluto) "/tmp/dir/abc/" -> { "", "tmp", "dir", "abc"}.
   *
   * @param path caminho do diretrio pai.
   * @param name o nome do arquivo a ser criado.
   * @return o arquivo criado.
   * @throws RepositoryFileException em caso de erro ao criar o arquivo.
   */
  IRepositoryFile createFile(String[] path, String name)
    throws RepositoryFileException;

  /**
   * Cria novo diretrio.
   *
   * @param path caminho do novo diretrio.
   * @return diretrio.
   * @throws IOException em caso de erro de IO.
   */
  IRepositoryFile createDir(String... path) throws IOException;

  /**
   * Renomeia o arquivo.
   *
   * Obs: Mesmo comportamento do comando move.
   *
   * @param srcFilePath o caminho do arquivo, relativo ao repositrio, a ser
   *        renomeado.
   * @param newFileName o novo nome para o arquivo.
   * @return true se o arquivo foi renomeado, false caso contrrio.
   * @throws IOException em caso de erro de IO.
   */
  boolean renameFile(String srcFilePath, String newFileName) throws IOException;

  /**
   * Remove arquivo ou diretrio no caminho especificado. O diretrio no
   * precisa estar vazio para ser removido.
   *
   * @param path caminho.
   * @return <code>true</code> se o arquivo ou diretrio foi removido,
   *         <code>false</code> caso contrrio.
   *
   * @throws IOException em caso de erro ao remover o arquivo ou diretrio.
   */
  boolean removeFile(String... path) throws IOException;

  /**
   * Verifica se um arquivo existe no repositrio.
   *
   * @param path caminho para o arquivo.
   * @return true se o arquivo existe, false caso contrrio.
   */
  boolean fileExists(String... path);

  /**
   * Bloqueia o arquivo.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @param file arquivo.
   * @return true se o arquivo foi bloqueado, false caso contrrio.
   */
  boolean lock(IRepositoryFile file);

  /**
   * Libera o bloqueio do arquivo.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @param file arquivo.
   * @return true se o arquivo foi liberado, false caso contrrio.
   */
  boolean release(IRepositoryFile file);

  /**
   * Verifica se o arquivo est bloqueado.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @param file arquivo.
   * @return true se o arquivo estiver bloqueado, false caso contrrio.
   */
  boolean isLocked(IRepositoryFile file);

  /**
   * Retorna uma cpia do arquivo indicado por srcPath dentro do caminho
   * indicado por targetDirPath.
   *
   * @param srcPath o caminho do arquivo de origem.
   * @param destPath o destino da nova cpia.
   * @return a nova cpia.
   * @throws IOException em caso de falha na cpia do arquivo.
   */
  IRepositoryFile copyFile(String srcPath, String... destPath)
    throws IOException;

  /**
   * Retorna o padro utilizado para separar as hierarquias de path deste
   * repositrio.
   *
   * @return o separador.
   */
  char getFileSeparator();
}
