package csbase.server.services.restservice;

import csbase.logic.Role;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.MailServiceInterface;
import csbase.remote.RestServiceInterface;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.mailservice.MailService;
import ibase.common.NotFoundException;
import ibase.exception.InternalServiceException;
import ibase.rest.api.authentication.v1.adapter.AuthenticationService;
import ibase.rest.api.authentication.v1.adapter.ParseException;
import ibase.rest.api.authentication.v1.adapter.UnauthorizedException;
import java.rmi.RemoteException;
import java.time.Clock;
import java.time.Instant;
import java.util.*;

/**
 * Implementao do adaptador CSBase para um servio de autenticao.
 *
 * @author Tecgraf/PUC-Rio
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseAuthenticationServiceImpl implements AuthenticationService {

    private static final String ADMIN_USER = "admin";
    /**
     * Chave para o EMAIL do usurio
     */
    public static final String ATTRIBUTE_USER_EMAIL = "userEmail";
    /**
     * Chave para o NOME do usurio
     */
    public static final String ATTRIBUTE_USER_NAME = "userName";
    /**
     * Chave para a SENHA do usurio
     */
    public static final String ATTRIBUTE_USER_PASSWORD = "userPassword";


    /**
     * {@inheritDoc}
     */
    @Override
    public void setLocale(Locale locale) {
        ClientRemoteLocator.administrationService.setLocale(locale);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String parserToken(String token, Map<String, Object> attributes)
            throws ParseException {
        RestServiceInterface restService = RestService.getInstance();
        try {
            String userId = restService.parserToken(token, attributes);
            if (userId != null) {
                Service.setUserId(userId);
            }
            return userId;
        } catch (csbase.exception.ParseException e) {
            throw new ParseException(e.getMessage());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ibase.common.User login(String login, String password)
            throws UnauthorizedException {
        csbase.logic.User user = LoginService.getInstance().checkLogin(login,
                password);
        if (user == null) {
            throw new UnauthorizedException(RestService.getInstance().getString(
                    "RestService.unauthorized.login.error"));
        }
        return new ibase.common.User(user.getLogin(), user.getName(),
                Arrays.asList(user.getEmails()), Arrays.asList(user.getRoleIds()), null);
    }

    /**
     * Cria um usurio guest usando um token de validao com o login e senha,
     * e retorna um token de acesso.
     *
     * @param userToken o token de validao de acesso do usurio
     * @return o token de acesso criado
     * @throws UnauthorizedException Caso o login ou senha sejam invlidos
     */
    @Override
    public ibase.common.User acceptNewUser(String userToken) throws UnauthorizedException {
        RestServiceInterface restService = RestService.getInstance();
        try {
            Map<String, Object> attributes = new Hashtable();
            String userId = restService.parserToken(userToken, attributes);
            String login = (String) attributes.get(ATTRIBUTE_USER_EMAIL);
            String email = (String) attributes.get(ATTRIBUTE_USER_EMAIL);
            String name = (String) attributes.get(ATTRIBUTE_USER_NAME);
            String password = (String) attributes.get(ATTRIBUTE_USER_PASSWORD);
            AdministrationService adminService = AdministrationService.getInstance();
            User user = adminService.getUser(login);
            if (user == null) {
                createGuestUser(login, name, email, password);
                Object[] roleIds = new Object[]{getVisitorRole()};
                return new ibase.common.User(login, name,
                        Arrays.asList(email), Arrays.asList(roleIds), null);
            } else {
                throw new UnauthorizedException(RestService.getInstance().getFormattedString(
                        "RestService.unauthorized.newuser.error", new Object[]{login}));
            }
        } catch (csbase.exception.ParseException e) {
            throw new UnauthorizedException(e.getMessage());
        }
    }

    /**
     * Modifica os dados de um usurio.
     *
     * @return o usurio
     * @throws UnauthorizedException Caso o login ou senha sejam invlidos
     */
    @Override
    public ibase.common.User updateUser(ibase.common.User user) throws UnauthorizedException, NotFoundException {
        try {
            Service.setUserId(ADMIN_USER);
            AdministrationService administrationService = AdministrationService.getInstance();

            User csbaseUser = administrationService.getUser(user.login);
            if (csbaseUser == null) {
                return null;
            }
            String[] currentEmails = csbaseUser.getEmails();
            user.emails.addAll(Arrays.asList(currentEmails));
            UserInfo userInfo = new UserInfo(csbaseUser.getUserInfo().getAttributes());
            userInfo.setAttribute(User.NAME, user.name);
            userInfo.setAttribute(User.EMAILS, user.emails.toArray(new String[0]));
            userInfo.setAttribute(User.PHOTO_LINK, user.avatarURL);
            Clock clock = Clock.systemDefaultZone();
            long lastUpdate = clock.millis();
            userInfo.setAttribute(User.LAST_UPDATE, new Long(lastUpdate));
            administrationService.modifyUser(user.login, userInfo);
            return user;
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        } finally {
            Service.setUserId(null);
        }
    }

    /**
     * Altera a senha de um usurio e retorna o usurio.
     *
     * @param userToken o token para validar a requisio de mudana de senha
     * @param password a nova senha
     * @return os dados do usurio
     * @throws UnauthorizedException Caso o token do usurio seja invlido
     */
    @Override
    public ibase.common.User acceptNewPassword(String userToken, String password) throws UnauthorizedException {
        RestServiceInterface restService = RestService.getInstance();
        try {
            Map<String, Object> attributes = new Hashtable();
            String userId = restService.parserToken(userToken, attributes);
            String login = (String) attributes.get(ATTRIBUTE_USER_EMAIL);
            AdministrationService adminService = AdministrationService.getInstance();
            User user = adminService.getUser(login);
            if (user == null) {
                throw new UnauthorizedException(RestService.getInstance().getFormattedString(
                        "RestService.unauthorized.newuser.error", new Object[]{login}));
            }
            changeUserPassword(login, password);
            return new ibase.common.User(login, user.getName(),
                    Arrays.asList(user.getEmails()), Arrays.asList(user.getRoleIds()), null);
        } catch (csbase.exception.ParseException e) {
            throw new UnauthorizedException(e.getMessage());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String createToken(String userId, Map<String, Object> attributes,
                              java.util.Date expirationDate, java.util.Date issuedDate) throws UnauthorizedException {
        try {
            AdministrationService administrationService = AdministrationService.getInstance();
            RestServiceInterface restService = RestService.getInstance();
            return restService.createToken(userId, attributes, expirationDate, issuedDate);
        } catch (Throwable e) {
            throw new UnauthorizedException(RestService.getInstance().getString(
                    "RestService.unauthorized.login.error"));
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String createToken(String userId, Map<String, Object> attributes,
                              java.util.Date expirationDate) throws UnauthorizedException {
        Instant now = Instant.now();
        return createToken(userId, attributes, expirationDate, Date.from(now));
    }


    /**
     * Valida se o novo usurio pode ser autenticado, enviando um email para confirmar o link com o token gerado.
     *
     * @param name nome do usurio
     * @param email email do usurio
     * @param link  link para validar se o novo usurio pode ser criado
     */
    @Override
    public void validateNewUserByEmail(String name, String email, String link) {
        try {
            User user = User.getUser(email);
            if (user!=null) {
                // throw UserAlreadyExists
            }
            String systemName = Server.getInstance().getSystemName();
            String msg = RestService.getInstance().getFormattedString(
                    "RestService.newuser.validation.message", new Object[] {name, systemName, link});
            MailServiceInterface service = MailService.getInstance();
            String[] recipient = new String[]{email};
            boolean sendEmail = service.sendMail(null, recipient, msg);
            if (!sendEmail) {
                throw new UnauthorizedException(RestService.getInstance().getString(
                        "RestService.unauthorized.email.send.error"));
            }
        } catch (Throwable e) {
            e.printStackTrace();
            throw new InternalServiceException(e);
        }
    }

    /**
     * Valida se a senha pode ser alterada, enviando um email para confirmar o link com o token gerado.
     *
     * @param email email do usurio
     * @param link link para validar se a senha pode ser alterada
     */
    public void validateNewPasswordByEmail(String email, String link) {
        try {
            User user = User.getUser(email);
            if (user==null) {
                // throw UserNotFoundException
            }
            String systemName = Server.getInstance().getSystemName();
            String msg = RestService.getInstance().getFormattedString(
                    "RestService.newpassword.validation.message", new Object[] {user.getName(), systemName, link});
            MailServiceInterface service = MailService.getInstance();
            String[] recipient = new String[]{email};
            boolean sendEmail = service.sendMail(null, recipient, msg);
            if (!sendEmail) {
                throw new UnauthorizedException(RestService.getInstance().getString(
                        "RestService.unauthorized.email.send.error"));
            }
        } catch (Throwable e) {
            e.printStackTrace();
            throw new InternalServiceException(e);
        }
    }

    /**
     * Envia um email para o suporte com uma notificao.
     * @param content contedo da notificao
     */
    public void notifySupport(String content) {
        // Envia um email para o suporte do sistema avisando de um novo usurio GUEST
        MailServiceInterface service = MailService.getInstance();
        try {
            service.mailSupport(content);
        } catch (RemoteException e) {
            e.printStackTrace();
        }
    }

    /**
     * Verifica se um usurio existe.
     *
     * @param userId login do usurio
     * @return {@code true} ou {@code false}
     */
    @Override
    public boolean userExists(String userId) {
        AdministrationService adminService = AdministrationService.getInstance();
        User user = adminService.getUser(userId);
        return user != null;
    }

    private void createGuestUser(String login, String name, String email, String password) throws UnauthorizedException {
        try {
            Service.setUserId(ADMIN_USER);
            AdministrationService administrationService = AdministrationService.getInstance();
            Object roleId = getVisitorRole();
            if (roleId == null) {
                throw new UnauthorizedException(RestService.getInstance().getString(
                        "RestService.no.visitor.role.login.error"));
            }

            Instant now = Instant.now();
            UserInfo newUserInfo = new UserInfo(login, name, new String[]{email},
                    new Object[]{roleId}, null, now.toEpochMilli());
            newUserInfo.setAttribute(User.PASSWORD, password);
            administrationService.createUser(newUserInfo);
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        } finally {
            Service.setUserId(null);
        }
    }

    private void changeUserPassword(String login, String password) throws UnauthorizedException {
        try {
            Service.setUserId(ADMIN_USER);
            AdministrationService administrationService = AdministrationService.getInstance();
            User user = administrationService.getUser(login);
            //Calendar c = Calendar.getInstance();
            //long time = c.getTime().getTime();
            Instant now = Instant.now();
            UserInfo newUserInfo = new UserInfo(login, user.getName(), user.getEmails(),
                    user.getRoleIds(), null, now.toEpochMilli());
            newUserInfo.setAttribute(User.PASSWORD, password);
            administrationService.modifyUser(user.getId(), newUserInfo);
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        } finally {
            Service.setUserId(null);
        }
    }

    private Object getVisitorRole() {
        Service.setUserId(ADMIN_USER);
        AdministrationService administrationService = AdministrationService.getInstance();
        List<Role> allRoles = administrationService.getAllRoles();
        for (Role r : allRoles) {
            if (r.getName().equals(GUEST_ROLE_ID)) {
                return r.getId();
            }
        }
        return null;
    }
}
