package csbase.server.services.restservice.websocket.messenger;

import com.google.common.collect.EvictingQueue;
import csbase.logic.UserNotification;
import csbase.logic.UserOutline;
import csbase.server.Server;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.restservice.websocket.*;
import csbase.server.services.restservice.websocket.utils.PersistentObject;
import csbase.server.services.restservice.websocket.utils.WebSocketUtils;
import org.glassfish.grizzly.websockets.*;

import java.util.ArrayList;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONObject;

public class CSBaseMessenger extends CSBaseWebSocketApplication {

	// Messenger history max size
	private static final int MESSENGER_HISTORY_MAX_SIZE = 100;

	// Messenger persistence file name
	private static final String MESSENGER_FILE = "wsmessenger.dat";

	// Messages persistence
	private PersistentObject<EvictingQueue<String>> messages = new PersistentObject<>(
			WebSocketUtils.generatePath(MESSENGER_FILE), EvictingQueue.create(MESSENGER_HISTORY_MAX_SIZE));

	public CSBaseMessenger() {
		super();

	}

	@Override
	public void onConnect(WebSocket ws) {
		super.onConnect(ws);

		if (ws instanceof CSBaseWebSocket) {
			// Upon connection, broadcast ulupd message and send message history
			// to the current user
			broadcastUserListUpdateMessage();
			ws.send(createMessengerHistoryMessage().toString());
		}

	}

	@Override
	public void onMessage(WebSocket websocket, String json) {

		try {
			// Cast to CSBaseWebSocket
			CSBaseWebSocket ws = (CSBaseWebSocket) websocket;

			// Parse message JSON
			Message incomingMessage = new Message(json);

			incomingMessage.setUser(ws.getUser());

			// Project change message
			if (incomingMessage.isProjectChange()) {

				// Instantiate project if message content is not null
				Object content = incomingMessage.getContent();
				Project project = content != null ? new Project((JSONObject) content) : null;

				// Update connection
				ws.setProject(project);
				connections.compute(ws.getUser().getLogin(), (user, userWebSockets) -> {
					userWebSockets.removeIf(userWebSocket -> userWebSocket.getId().equals(ws.getId()));
					userWebSockets.add(ws);
					return userWebSockets;
				});

				// Send prchg message only to users who have access to
				// the given project
				Message projectChange = createProjectChangeMessage(ws.getUser(), ws.getProject());
				this.connections.forEach((userLogin, csBaseWebSockets) -> {

					if (ws.getProject() == null) {
						broadcaster.broadcast(csBaseWebSockets, projectChange.toString());
					} else if (ws.getProject().isVisible(userLogin)) {
						broadcaster.broadcast(csBaseWebSockets, projectChange.toString());
					}

				});

				// Chat message
			} else if (incomingMessage.isChat()) {

				// Add message to history
				messages.get().add(incomingMessage.toString());

				// Broadcast to online users
				broadcast(incomingMessage);

				// Broadcast to CSBase desktop client
				UserNotification userNotification = new UserNotification(ws.getUser().getLogin(),
						incomingMessage.getContent().toString(), false, false);
				csbase.util.messages.Message csbaseMessage = new csbase.util.messages.Message(userNotification);
				UserOutline[] loggedUsers = LoginService.getInstance().getLoggedUsers();
				for (UserOutline loggedUser : loggedUsers) {
					MessageService.getInstance().send(csbaseMessage, loggedUser.getLogin());
				}

			}
		} catch (Exception e) {
			e.printStackTrace();
			Server.logSevereMessage(
					"Error processing message " + json + " sent by " + ((CSBaseWebSocket) websocket).getUser(), e);
		}
	}

	@Override
	public void onClose(WebSocket socket, DataFrame frame) {
		super.onClose(socket, frame);

		if (socket instanceof CSBaseWebSocket) {
			broadcastUserListUpdateMessage();
			messages.save();
		}

	}
	/**
	 * Broadcasts a message
	 *
	 * @param message
	 *            the message
	 */
	private void broadcast(Message message) {

		final List<CSBaseWebSocket> targets = new ArrayList<>();
		this.connections.values().forEach(targets::addAll);

		broadcaster.broadcast(targets, message.toString());
	}

	/**
	 * Broadcasts a ulupd message
	 *
	 */
	public void broadcastUserListUpdateMessage() {
		final String NO_PROJECT = "";

		// Each user has a customized ulupd message due to project visibility
		// settings
		this.connections.forEach((targetUser, targetUserWebSockets) -> {

			// Instantiate new message
			Message usersOnline = new Message();
			usersOnline.setType(Message.TYPE_USERLIST_UPDATE);

			// Instantiate projects JSON, which will be the content of the ulupd
			// message
			JSONObject projects = new JSONObject();

			this.connections.forEach((user, userWebSockets) -> {

				// Get current project from user, i.e., the last one he opened
				long openedAt = 0;
				Project currentProject = null;
				User currentUser = null;

				for (CSBaseWebSocket ws : userWebSockets) {
					// User will always be the same
					currentUser = ws.getUser();

					// If project of current connection was opened last, update
					// current project
					if (ws.getProject() != null && ws.getModifiedAt() > openedAt) {
						currentProject = ws.getProject();
						openedAt = ws.getModifiedAt();
					}
				}

				// Compute project id, i.e., empty string if currentProject is
				// null, id otherwise
				String currentProjectId = currentProject != null ? currentProject.getId() : "";

				// If project already has a user connected (which means it is
				// visible to the target user), add current user to the specific
				// project
				if (projects.has(currentProjectId)) {
					JSONObject currentProjectUsers = projects.getJSONObject(currentProjectId);
					currentProjectUsers.put("users", currentProjectUsers.getJSONArray("users").put(currentUser));
					projects.put(currentProjectId, currentProjectUsers);
				}
				// If project is null or not visible to the target user, add
				// current user to NO_PROJECT list
				else {
					if (currentProject == null || !currentProject.isVisible(targetUser)) {
						JSONObject usersProject = new JSONObject();
						usersProject.put("users", new JSONArray().put(currentUser));
						usersProject.put("project", NO_PROJECT);
						projects.put(currentProjectId, usersProject);
					} else {
						JSONObject usersProject = new JSONObject();
						usersProject.put("users", new JSONArray().put(currentUser));
						usersProject.put("project", currentProject);
						projects.put(currentProjectId, usersProject);
					}
				}

			});

			usersOnline.setContent(projects);

			// Broadcast ulupd message to target user connections
			broadcaster.broadcast(targetUserWebSockets, usersOnline.toString());

		});
	}

	public Message createProjectChangeMessage(User user, Project project) {
		Message projectChange = new Message();
		projectChange.setType(Message.TYPE_PROJECT_CHANGE);
		projectChange.setUser(user);
		projectChange.setContent(project);
		return projectChange;
	}

	private Message createMessengerHistoryMessage() {
		JSONArray jsonArray = new JSONArray();
		messages.get().forEach(jsonArray::put);
		Message history = new Message();
		history.setType(Message.TYPE_CHAT_HISTORY);
		history.setContent(jsonArray);
		return history;
	}

}
