/**
 * $Id: AlgorithmService.java 163880 2015-04-20 20:21:35Z mjulia $
 */

package csbase.server.services.opendreamsservice.algorithm.v1_0;

import java.io.File;
import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.openbus.algorithmservice.v1_0.AlgorithmBasicInfo;
import tecgraf.openbus.algorithmservice.v1_0.AlgorithmDetailedInfo;
import tecgraf.openbus.algorithmservice.v1_0.AlgorithmDoesNotExistsException;
import tecgraf.openbus.algorithmservice.v1_0.AlgorithmType;
import tecgraf.openbus.algorithmservice.v1_0.AuthorizationException;
import tecgraf.openbus.algorithmservice.v1_0.Configurator;
import tecgraf.openbus.algorithmservice.v1_0.ConfiguratorErrorException;
import tecgraf.openbus.algorithmservice.v1_0.ExecutionType;
import tecgraf.openbus.algorithmservice.v1_0.FTCFile;
import tecgraf.openbus.algorithmservice.v1_0.FlowConfiguratorImpl;
import tecgraf.openbus.algorithmservice.v1_0.FlowNode;
import tecgraf.openbus.algorithmservice.v1_0.NoPermissionException;
import tecgraf.openbus.algorithmservice.v1_0.Parameter;
import tecgraf.openbus.algorithmservice.v1_0.Program;
import tecgraf.openbus.algorithmservice.v1_0.Property;
import tecgraf.openbus.algorithmservice.v1_0.SimpleConfigurator;
import tecgraf.openbus.algorithmservice.v1_0.SimpleConfiguratorImpl;
import tecgraf.openbus.algorithmservice.v1_0.UserDoesNotExistsException;
import tecgraf.openbus.algorithmservice.v1_0.Version;
import tecgraf.openbus.algorithmservice.v1_0.VersionDoesNotExistsException;
import tecgraf.openbus.algorithmservice.v1_0.parameters.EnumerationListParameterImpl;
import tecgraf.openbus.algorithmservice.v1_0.parameters.EnumerationParameterImpl;
import tecgraf.openbus.algorithmservice.v1_0.parameters.RawValueParameterHelper;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmOutline;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.Node;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.HiddenParameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.ftcservice.FTCService;
import csbase.server.services.openbusservice.OpenBusService;

/**
 * O servio <code>AlgorithmService</code>  uma faceta do componente OpenDreams
 * responsvel pela obteno de informaes sobre algoritmos solicitadas por
 * sistemas externos ao CSBase.
 *
 * @author Tecgraf
 */
public class AlgorithmService extends
  tecgraf.openbus.algorithmservice.v1_0.AlgorithmServicePOA {

  /**
   * Mensagem de erro para verso no encontrada
   * */
  private static final String ERROR_VERSION_NOT_FOUND =
    "A verso {0} do algoritmo {1} no est cadastrada no sistema.";

  /**
   * Mensagem de erro para algoritmo no encontrado
   * */
  private static final String ERROR_ALGO_NOT_FOUND =
    "O algoritmo {0} no est cadastrado no sistema.";

  /**
   * Mensagem de erro ao criar configurador de algoritmo
   */
  private static final String ERROR_CONF_VERSION_NOT_FOUND =
    "Ocorreu um erro ao criar o configurador do algoritmo {0}.";

  /**
   * Mensagem de erro quando usurio no tem permisso para executar algoritmo
   */
  private static final String ERROR_NO_PERMISSION =
    "O usurio {0} no tem permisso par executar o algoritmo {1}.";

  /**
   * Mensagem de erro quando usurio no existe no CSBase
   */
  private static final String ERROR_USER_NOT_EXISTS =
    "Usurio {0} no existe no CSBase.";

  /**
   * Referncia para o servio AlgorithmService.
   */
  private final csbase.server.services.algorithmservice.AlgorithmService algorithmService;

  /**
   * Referncia para o servio FTC.
   */
  private final FTCService ftcService = FTCService.getInstance();

  /**
   * Constri o objeto que implementa a interface <code>AlgorithmService</code>.
   */
  public AlgorithmService() {
    algorithmService =
      csbase.server.services.algorithmservice.AlgorithmService.getInstance();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmBasicInfo[] getAlgorithms() throws AuthorizationException {
    AlgorithmOutline[] algoOutlineList = algorithmService.getAllOutlines();

    Hashtable<String, AlgorithmBasicInfo> permittedAlgorithms =
      new Hashtable<String, AlgorithmBasicInfo>();

    CategorySet categorySet = algorithmService.getAllCategories();
    List<String> categoriesFullNames;

    String sessionUser = OpenBusService.getInstance().initCSBaseAccess();

    try {
      User user = User.getUserByLogin(sessionUser);
      if (user == null) {
        throw new AuthorizationException(getUserNotExistsMessage(sessionUser));
      }

      for (AlgorithmOutline algo : algoOutlineList) {
        if (AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
          user, Service.getSystemId(), algo.getName())) {
          if (!permittedAlgorithms.containsKey(algo.getName())) {
            permittedAlgorithms.put(algo.getName(), createAlgorithmInfo(algo));
          }
        }

        categoriesFullNames =
          categorySet.getAlgorithmCategoriesFullNames(algo.getName());
        if (CategoryAlgorithmsExecutionPermission
          .checkSystemAndCategoriesExecPermission(user, Service.getSystemId(),
            categoriesFullNames)) {
          if (!permittedAlgorithms.containsKey(algo.getName())) {
            permittedAlgorithms.put(algo.getName(), createAlgorithmInfo(algo));
          }
        }
      }
    }
    catch (AuthorizationException e) {
      throw e;
    }
    catch (Exception e) {
      Server
        .logInfoMessage("Erro na recuperao de permisso durante a busca dos algoritmos para o usurio "
          + sessionUser + ".");
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }

    return permittedAlgorithms.values().toArray(
      new AlgorithmBasicInfo[permittedAlgorithms.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmDetailedInfo getAlgorithmDetail(String algorithmName)
    throws AlgorithmDoesNotExistsException, NoPermissionException,
    ConfiguratorErrorException, AuthorizationException {
    AlgorithmDetailedInfo algorithmInfoDetail = new AlgorithmDetailedInfo();

    String sessionUser = OpenBusService.getInstance().initCSBaseAccess();

    try {
      if (!checkUserPermission(algorithmName, sessionUser)) {
        throw new NoPermissionException(getNoPermissionErrorMessage(
          algorithmName, sessionUser));
        // throw new AuthorizationException(getUserNotExistsMessage(sessionUser));

      }

      algorithmInfoDetail = getDelailInfo(algorithmName);

    }
    catch (UserDoesNotExistsException e) {
      // throw new NoPermissionException(getNoPermissionErrorMessage(
      //   algorithmName, sessionUser));
      throw new AuthorizationException(getUserNotExistsMessage(sessionUser));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }

    return algorithmInfoDetail;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmDetailedInfo[] getAlgorithmsDetails(String[] algorithmNameList)
    throws AlgorithmDoesNotExistsException, NoPermissionException,
    ConfiguratorErrorException, AuthorizationException {
    Vector<AlgorithmDetailedInfo> algorithmDetailedInfoList =
      new Vector<AlgorithmDetailedInfo>();
    for (String algorithmName : algorithmNameList) {
      AlgorithmDetailedInfo algoDetail = getAlgorithmDetail(algorithmName);

      if (algoDetail != null) {
        algorithmDetailedInfoList.add(algoDetail);
      }
    }

    return algorithmDetailedInfoList
      .toArray(new AlgorithmDetailedInfo[algorithmDetailedInfoList.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean checkUserPermission(String algorithmName, String userId)
    throws UserDoesNotExistsException, AlgorithmDoesNotExistsException {
    boolean hasPermission = false;

    CategorySet categorySet = algorithmService.getAllCategories();
    List<String> categoriesFullNames;
    User user;

    try {
      OpenBusService.getInstance().initCSBaseAccess();

      user = User.getUserByLogin(userId);
      if (user == null) {
        throw new UserDoesNotExistsException("Usurio " + userId
          + " no existe.");
      }
      csbase.logic.algorithms.AlgorithmInfo logicAlgorithmInfo =
        csbase.logic.algorithms.AlgorithmInfo.getAlgorithmInfo(algorithmName);
      if (logicAlgorithmInfo == null) {
        throw new AlgorithmDoesNotExistsException(
          getAlgoNotFoundMessage(algorithmName));
      }

      hasPermission =
        AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
          user, Service.getSystemId(), algorithmName);

      categoriesFullNames =
        categorySet.getAlgorithmCategoriesFullNames(algorithmName);
      hasPermission =
        hasPermission
          || CategoryAlgorithmsExecutionPermission
            .checkSystemAndCategoriesExecPermission(user,
              Service.getSystemId(), categoriesFullNames);
    }
    catch (Exception e) {
      Server.logInfoMessage("Erro ao recuperar permisso do usurio " + userId
        + " para o algoritmo " + algorithmName + ".");
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }

    return hasPermission;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Configurator getConfigurator(String algorithmName,
    String algorithmVersion) throws ConfiguratorErrorException,
    VersionDoesNotExistsException, AlgorithmDoesNotExistsException,
    AuthorizationException {

    String sessionUser = OpenBusService.getInstance().initCSBaseAccess();
    try {
      User user = User.getUserByLogin(sessionUser);
      if (user == null) {
        throw new AuthorizationException(getUserNotExistsMessage(sessionUser));
      }
    }
    catch (AuthorizationException e) {
      throw e;
    }
    catch (Exception e) {
      throw new ConfiguratorErrorException();
    }

    try {
      AlgorithmInfo logicAlgorithmInfo =
        csbase.logic.algorithms.AlgorithmInfo.getAlgorithmInfo(algorithmName);

      if (logicAlgorithmInfo == null) {
        throw new AlgorithmDoesNotExistsException(
          getAlgoNotFoundMessage(algorithmName));
      }

      AlgorithmVersionId versionId =
        AlgorithmVersionId.create(algorithmVersion);
      if (versionId == null) {
        throw new VersionDoesNotExistsException(getVersionNotFoundMessage(
          algorithmName, algorithmVersion));
      }

      AlgorithmConfigurator algorithmConfigurator =
        algorithmService.createAlgorithmConfigurator(algorithmName, versionId);

      return createConfigurator(algorithmConfigurator);
    }
    catch (AlgorithmDoesNotExistsException e) {
      throw e;
    }
    catch (VersionDoesNotExistsException e) {
      throw e;
    }
    catch (AlgorithmNotFoundException e) {
      throw new VersionDoesNotExistsException(getVersionNotFoundMessage(
        algorithmName, algorithmVersion));
    }
    catch (Exception e) {
      throw new ConfiguratorErrorException(
        getConfiguratorErrorMessage(algorithmName));
    }
  }

  /*****************************************************************************
   * Mtodos utilitrios
   ****************************************************************************/

  /**
   * Cria um descritor de arquivo FTC a partir de um arquivo.
   *
   * @param file arquivo para criar descritor de arquivo FTC
   *
   * @return descritor de arquivo FTC
   * @throws Exception
   */
  private FTCFile createFTCFile(File file) throws Exception {
    if (file.isDirectory()) {
      throw new Exception("Caminho informado  um diretrio");
    }

    RemoteFileChannelInfo channel;

    channel = ftcService.createFileChannelInfo(file, true);
    FTCFile ftcFile = new FTCFile();
    ftcFile.id = channel.getIdentifier();
    ftcFile.host = channel.getHost();
    ftcFile.port = channel.getPort();
    ftcFile.key = channel.getKey();

    return ftcFile;
  }

  /**
   * Verifica se uma string  {@code null} e em caso afirmativo retirna uma
   * string vazia
   *
   * @param str String para
   *
   * @return Uma String vazia caso a entrada seja {@code null} e a prpria
   *         entrada caso contrrio
   */
  private String fillNullStringParam(String str) {
    if (str == null) {
      return "";
    }
    else {
      return str;
    }
  }

  /**
   * Obtm mensagem de erro para o caso do algoritmo especificado no ter sido
   * encontrado.
   *
   * @param algoId identificador do algoritmo no encontrado.
   *
   * @return mensagem de erro.
   */
  private String getAlgoNotFoundMessage(Object algoId) {
    String msg = ERROR_ALGO_NOT_FOUND;
    Object[] args = new Object[] { algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso da verso especificada no ter sido
   * encontrada.
   *
   * @param algoId identificador do algoritmo ao qual deveria pertencer a verso
   *        no encontrada.
   * @param versionId identificador da verso no encontrada.
   *
   * @return mensagem de erro.
   */
  private String getVersionNotFoundMessage(Object algoId, Object versionId) {
    String msg = ERROR_VERSION_NOT_FOUND;
    Object[] args = new Object[] { versionId, algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro quando no for possvel obter o configurador de
   * algoritmo.
   *
   * @param algoId identificador do algoritmo
   *
   * @return mensagem de erro.
   */
  private String getConfiguratorErrorMessage(Object algoId) {
    String msg = ERROR_CONF_VERSION_NOT_FOUND;
    Object[] args = new Object[] { algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no possuir permisso de
   * execuo do algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param userId identificador do usurio
   *
   * @return mensagem de erro.
   */
  private String getNoPermissionErrorMessage(Object algoId, Object userId) {
    String msg = ERROR_NO_PERMISSION;
    Object[] args = new Object[] { userId, algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no existir no CSBase.
   *
   * @param userId identificador do usurio
   *
   * @return mensagem de erro.
   */
  private String getUserNotExistsMessage(Object userId) {
    String msg = ERROR_USER_NOT_EXISTS;
    Object[] args = new Object[] { userId };
    return MessageFormat.format(msg, args);
  }

  /*****************************************************************************
   * Mtodos para converter as classes de lgica para as classes usadas pelo
   * servio OpenDreams
   ****************************************************************************/

  /**
   * Obtm as informaes detalhadas de um algoritmo.
   *
   * @param algorithmName nome do algoritmo
   *
   * @return os detalhes do algoritmo
   *
   * @throws AlgorithmDoesNotExistsException
   * @throws ConfiguratorErrorException
   */
  private AlgorithmDetailedInfo getDelailInfo(String algorithmName)
    throws AlgorithmDoesNotExistsException, ConfiguratorErrorException {
    AlgorithmDetailedInfo algorithmInfoDetail = new AlgorithmDetailedInfo();
    csbase.logic.algorithms.AlgorithmInfo logicAlgorithmInfo = null;

    try {
      logicAlgorithmInfo =
        csbase.logic.algorithms.AlgorithmInfo.getAlgorithmInfo(algorithmName);
    }
    catch (Exception e) {
      throw new ConfiguratorErrorException(
        getConfiguratorErrorMessage(algorithmName));
    }

    if (logicAlgorithmInfo == null) {
      throw new AlgorithmDoesNotExistsException(
        getAlgoNotFoundMessage(algorithmName));
    }

    algorithmInfoDetail.name = logicAlgorithmInfo.getName();
    algorithmInfoDetail.description =
      fillNullStringParam(logicAlgorithmInfo.getDescription());

    Vector<AlgorithmVersionInfo> versionsInfo =
      logicAlgorithmInfo.getVersions();
    Vector<Version> versions = new Vector<Version>();
    for (AlgorithmVersionInfo vInfo : versionsInfo) {
      try {
        versions.add(createAlgorithmVersion(vInfo));
      }
      catch (Exception e) {
        throw new ConfiguratorErrorException(e.getMessage());
      }

      algorithmInfoDetail.versions = new Version[versions.size()];
      versions.toArray(algorithmInfoDetail.versions);

      Hashtable<String, String> properties =
        logicAlgorithmInfo.getPropertyValues();

      Vector<Property> algorithmProperties = new Vector<Property>();

      for (Enumeration<String> keys = properties.keys(); keys.hasMoreElements();) {
        String key = keys.nextElement();
        Property p = new Property(key, properties.get(key));
        algorithmProperties.add(p);
      }

      algorithmInfoDetail.properties =
        algorithmProperties.toArray(new Property[algorithmProperties.size()]);

      CategorySet categorySet = algorithmService.getAllCategories();
      List<String> categoriesFullNames;
      categoriesFullNames =
        categorySet.getAlgorithmCategoriesFullNames(algorithmInfoDetail.name);

      algorithmInfoDetail.categories =
        categoriesFullNames.toArray(new String[categoriesFullNames.size()]);
    }

    return algorithmInfoDetail;
  }

  /**
   * Cria um objeto da classe {@link AlgorithmBasicInfo} a partir de um objeto
   * da classe {@link AlgorithmOutline}
   *
   * @param algorithmOutline
   *
   * @return informaes do algoritmo
   * @throws Exception
   */
  private AlgorithmBasicInfo createAlgorithmInfo(
    AlgorithmOutline algorithmOutline) throws Exception {
    String algorithmName = algorithmOutline.getName();
    csbase.logic.algorithms.AlgorithmInfo logicAlgorithmInfo =
      csbase.logic.algorithms.AlgorithmInfo.getAlgorithmInfo(algorithmName);

    String description =
      fillNullStringParam(logicAlgorithmInfo.getDescription());

    return new AlgorithmBasicInfo(algorithmName, description);
  }

  /**
   * Cria um objeto da classe {@link Version} a partir de um objeto da classe
   * {@link AlgorithmVersionInfo}.
   *
   * @param vInfo informao da verso
   *
   * @return verso criada
   * @throws AlgorithmNotFoundException
   * @throws ConfiguratorErrorException
   */
  private Version createAlgorithmVersion(AlgorithmVersionInfo vInfo)
    throws AlgorithmNotFoundException, ConfiguratorErrorException {
    Version version = new Version();

    AlgorithmVersionId versionId = vInfo.getId();
    version.major = versionId.getMajor();
    version.minor = versionId.getMinor();
    version.patch = versionId.getPatch();
    version.stringValue = versionId.toString();

    // TODO Implementao provisria. Verificar como ser a soluo final para requerimentos.
    Set<String> platforms = vInfo.getPlatformsNames();
    Vector<Program> programs = new Vector<Program>(platforms.size());
    for (String platform : platforms) {
      Program prog = new Program();
      prog.name = platform;
      prog.requirements = new String[] { platform };

      programs.add(prog);
    }
    version.programs = programs.toArray(new Program[programs.size()]);

    try {
      version.configFile =
        createFTCFile(new File(vInfo.getConfiguratorDirPath() + File.separator
          + AlgorithmVersionInfo.CONFIG_FILE));
    }
    catch (Exception e) {
      Server
        .logInfoMessage("Erro ao acessar aquivo de configurao do algoritmo "
          + vInfo.getInfo().getName() + " .");
    }

    return version;
  }

  /**
   * Cria um objeto da classe {@link Configurator} a partir de um objeto da
   * classe {@link AlgorithmConfigurator} e de um caminho para o arquivo de
   * configurao de um algoritmo.
   *
   * @param algorithmConfigurator Configurador de algoritmo
   *
   * @return configurador
   * @throws ConfiguratorErrorException
   */
  private Configurator createConfigurator(
    AlgorithmConfigurator algorithmConfigurator)
    throws ConfiguratorErrorException {
    Configurator configurator = null;

    switch (algorithmConfigurator.getConfiguratorType()) {
      case SIMPLE:
        SimpleAlgorithmConfigurator simpleConfigurator =
          (SimpleAlgorithmConfigurator) algorithmConfigurator;

        configurator = new SimpleConfiguratorImpl();
        configurator.type = AlgorithmType.ALGORITHM;

        Vector<Parameter> paramVector = new Vector<Parameter>();

        for (SimpleParameter<?> simpleParam : simpleConfigurator
          .getSimpleParameters()) {
          if ((simpleParam.isEnabled() || !simpleParam.ignoreIfDisabled())
            || (simpleParam.isVisible() || !simpleParam.ignoreIfInvisible())) {
            String paramType =
              ParameterMapping.getInstance().getIDLType(simpleParam.getType());
            if (paramType == null) {
              paramType = RawValueParameterHelper.id();
            }
            Parameter param = this.makeParameter(simpleParam, paramType);
            paramVector.add(param);
          }
        }

        for (HiddenParameter hiddenParam : simpleConfigurator
          .getHiddenParameters()) {
          String paramType = RawValueParameterHelper.id();
          Parameter param = this.makeParameter(hiddenParam, paramType);
          paramVector.add(param);
        }

        Parameter[] parameters = new Parameter[paramVector.size()];
        SimpleConfigurator.class.cast(configurator).parameters =
          paramVector.toArray(parameters);
        break;
      case FLOW:
        FlowAlgorithmConfigurator flowConfigurator =
        (FlowAlgorithmConfigurator) algorithmConfigurator;

        configurator = new FlowConfiguratorImpl();
        configurator.type = AlgorithmType.FLOW;

        Vector<FlowNode> algos = new Vector<FlowNode>();
        for (Node node : flowConfigurator.getNodes()) {
          FlowNode fNode = new FlowNode();
          AlgorithmConfigurator nodeConfig = node.getConfigurator();
          fNode.name = nodeConfig.getAlgorithmName();
          fNode.id = node.getId();
          fNode.config = createConfigurator(nodeConfig);
          algos.add(fNode);
        }

        FlowConfiguratorImpl.class.cast(configurator).algorithms =
          algos.toArray(new FlowNode[algos.size()]);
        break;
    }

    switch (algorithmConfigurator.getExecutionType()) {
      case SIMPLE:
        configurator.execType = ExecutionType.SIMPLE;
        break;
      case MULTIPLE:
        configurator.execType = ExecutionType.MULTIPLE;
        break;
    }

    configurator.algorithmName = algorithmConfigurator.getAlgorithmName();
    configurator.algorithmVersion =
      algorithmConfigurator.getAlgorithmVersionId().toString();

    return configurator;
  }

  /**
   * Constri o descritor do parmetro para ser retornado pelo servio.
   *
   * @param configuratorParam o configurador do parmetro
   * @param paramType o tipo do parmetro
   * @return a descrio do parmetro.
   */
  private Parameter makeParameter(
    csbase.logic.algorithms.parameters.Parameter<?> configuratorParam,
    String paramType) {
    Parameter param = new Parameter();
    param.name = configuratorParam.getName();
    param.type = paramType;
    // Coloca as opes disponveis no parmetro do tipo enumerao simples
    if (EnumerationParameter.class.isInstance(configuratorParam)) {
      param.value = new EnumerationParameterImpl();
      for (EnumerationItem item : EnumerationParameter.class.cast(
        configuratorParam).getItems()) {
        EnumerationParameterImpl.class.cast(param.value).addItem(
          new tecgraf.openbus.algorithmservice.v1_0.parameters.EnumerationItem(
            fillNullStringParam(item.getId()), fillNullStringParam(item
              .getLabel()), fillNullStringParam(item.getValue()),
            fillNullStringParam(item.getDescription()), true));
      }
    }

    // Coloca as opes disponveis no parmetro do tipo enumerao multipla
    if (EnumerationListParameter.class.isInstance(configuratorParam)) {
      param.value = new EnumerationListParameterImpl();
      for (EnumerationItem item : EnumerationListParameter.class.cast(
        configuratorParam).getItems()) {
        EnumerationListParameterImpl.class.cast(param.value).addItem(
          new tecgraf.openbus.algorithmservice.v1_0.parameters.EnumerationItem(
            fillNullStringParam(item.getId()), fillNullStringParam(item
              .getLabel()), fillNullStringParam(item.getValue()),
            fillNullStringParam(item.getDescription()), true));
      }
    }
    return param;
  }
}
