/*
 * SGA.java
 *
 * $Author: fpina $ $Revision: 177931 $ - $Date: 2007-10-25 13:54:22 -0300 (Thu,
 * 25 Oct 2007) $
 */
package csbase.server.services.sgaservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.rmi.RemoteException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import csbase.exception.OperationFailureException;
import csbase.logic.CapacityType;
import csbase.logic.ClientSGAFile;
import csbase.logic.CommandInfo;
import csbase.logic.CommandInfoCache;
import csbase.logic.CommandStatus;
import csbase.logic.FailureFinalizationType;
import csbase.logic.SGAInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.remote.EventLogServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.eventlogservice.EventLogService;
import sgaidl.Pair;
import sgaidl.SGACommand;
import sgaidl.SGAControlAction;
import sgaidl.SGADaemonOperations;
import sgaidl.SGAPath;
import sgaidl.SGAProperties;
import sgaidl.SGA_ALGORITHM_ROOT_DIR;
import sgaidl.SGA_BOOLEAN_TRUE;
import sgaidl.SGA_CSFS_HOST;
import sgaidl.SGA_CSFS_PORT;
import sgaidl.SGA_CSFS_ROOT_DIR;
import sgaidl.SGA_FILE_SEPARATOR;
import sgaidl.SGA_HAS_DISK_ACCESS;
import sgaidl.SGA_JOBS_INFO;
import sgaidl.SGA_NODE_CLOCK_SPEED_MHZ;
import sgaidl.SGA_NODE_LOAD_AVG_15MIN_PERC;
import sgaidl.SGA_NODE_LOAD_AVG_1MIN_PERC;
import sgaidl.SGA_NODE_LOAD_AVG_5MIN_PERC;
import sgaidl.SGA_NODE_MEMORY_RAM_FREE_PERC;
import sgaidl.SGA_NODE_MEMORY_RAM_INFO_MB;
import sgaidl.SGA_NODE_MEMORY_SWAP_FREE_PERC;
import sgaidl.SGA_NODE_MEMORY_SWAP_INFO_MB;
import sgaidl.SGA_NODE_NAME;
import sgaidl.SGA_NODE_NUMBER_OF_JOBS;
import sgaidl.SGA_NODE_NUM_PROCESSORS;
import sgaidl.SGA_NODE_PLATFORM_ID;
import sgaidl.SGA_NODE_RESOURCES_SEQ;
import sgaidl.SGA_PROJECT_ROOT_DIR;
import sgaidl.SGA_SANDBOX_ROOT_DIR;
import tecgraf.javautils.core.io.FileUtils;

/**
 * A classe <code>SGA</code> representa o gerenciador de uma mquina hospedeira
 * (ou servidor), capaz de atender  solicitao de execuo remota de
 * algoritmos. Alm de comandar a execuo remota de um algoritmo, um objeto SGA
 * permite a consulta  configurao da mquina hospedeira (<i>hostname</i>,
 * plataforma de execuo, nmero de processadores, ...) e a seu estado corrente
 * (carga no processador, memria livre disponvel).
 *
 * @author Ana Moura, Andr Clinio e Cristina Ururahy
 */
class SGA {

  /** Nome do arquivo de persistncia para comandos em execuo. */
  private final static String EXECUTION_COMMANDS_FILENAME_SUFFIX =
    "-execution-commands.dat";

  /** Constante para converso para milisegundos */
  private static final int MILLIS = 1000;

  /** Constante que indica a ausncia de determinado dado do histrico */
  private static final int NO_DATA = -1;

  /** Tabela de comandos em execuo no SGA. */
  private Hashtable<Object, Command> commandTable = null;

  /** Servio que gerencia este SGA. */
  private SGAService sgaService = null;

  /**
   * Informaes de cada um dos ns do SGA. Podem ser serializadas para o
   * cliente.
   */
  private SGAInfo[] info = null;

  /** Nome que representa o SGA */
  private String name = null;

  /** Mquina que deve ser contactada para transferncias via CSFS. */
  private String csfsHost;

  /** Porta da mquina que deve ser contactada para transferncias via CSFS. */
  private int csfsPort;

  /** Diretrio que deve ser usado para transferncias via CSFS. */
  private String[] csfsRootDir;

  /** Indica se o SGA tem acesso ao disco. */
  private boolean hasDiskAccess;

  /** Representa as informaes dos jobs em execuo no SGA. */
  private String jobsInfo;

  /** Indica se o SGA  um cluster. */
  private boolean isCluster;

  /** Indica se o SGA est apto a executar comandos. */
  private boolean enabled;

  /** Um watchdog verifica se o SGA est atualizando seus dados. */
  /*
   * Quando o SGA atualiza seus dados ele faz uma marca (patWatchDog) e de
   * tempos em tempos o SGAService retira essa marca (resetWatchDog). Quando o
   * SGAService vai retirar a marca e ela no est l, significa que o SGA no
   * atualizou seus dados. Neste caso, o SGAService considera que o SGA est
   * inacessvel.
   */
  private boolean watchdog;

  /**
   * Propriedades do SGA.
   */
  //TODO Por enquanto vamos deixar String, String. Avaliar melhor.
  private Map<String, String> sgaProperties;

  /** Referncia remota para o gerenciador da mquina */
  private SGADaemonOperations remoteReference;

  /** Timestamp do registro */
  private long registerTimestamp;

  /**
   * Flag indicativo de sada da thread de aquisio de dados dos SGAs.
   * */
  private boolean exitAcquisitionDataThread = false;

  /**
   * Tabela com o mapeamento entre as capacidades declaradas na idl corba e as
   * capacidades da enumerao Java
   */
  private static HashMap<Integer, CapacityType> capacityMap = null;

  /** Executor que usa um pool de threads para submisso de comandos. */
  private final ExecutorService commandExecutor;

  /** Timestamp relativo ao ltimo update de informaes */
  private long updateTimestamp;

  /**
   * Mtodo de registro de um evento para o EventLog (servidor).
   *
   * @param queue um identificador da fila no EventLog.
   * @param info a informao (<b>simples</b>) a ser registrada
   * @see EventLogServiceInterface
   */
  private void logEvent(final String[] queue, final String info) {
    final EventLogService eventService = EventLogService.getInstance();
    final int N = queue.length;
    final String[] reQueue = new String[N + 1];
    reQueue[0] = "sga";
    for (int i = 0; i < N; i++) {
      reQueue[i + 1] = queue[i];
    }
    try {
      eventService.addServerInformation(reQueue, new String[] { info });
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de log de eventos " + e);
    }
  }

  /**
   * Atualiza a configurao esttica da mquina hospedeira.
   *
   * @param sgaProperties informaes estticas dos ns
   */
  private void setSGAProperties(SGAProperties sgaProperties) {
    String sgaName = this.name;

    // XXX No estamos fazendo tratamento de erro para o caso de alguma propriedade no existir.

    // Propriedades do SGA
    this.sgaProperties = SGAUtils.pairToHashMap(sgaProperties.properties);
    char fileSeparator =
      this.sgaProperties.get(SGA_FILE_SEPARATOR.value).charAt(0);
    String[] projectRootDirectory =
      FileUtils.splitPath(this.sgaProperties.get(SGA_PROJECT_ROOT_DIR.value),
        fileSeparator);
    String[] algorithmRootDirectory =
      FileUtils.splitPath(this.sgaProperties.get(SGA_ALGORITHM_ROOT_DIR.value),
        fileSeparator);
    String[] sandboxRootDirectory =
      FileUtils.splitPath(this.sgaProperties.get(SGA_SANDBOX_ROOT_DIR.value),
        fileSeparator);

    // Propriedades dos ns
    Pair[][] nodesProps = sgaProperties.nodesProperties;
    SGAInfo[] infoTmp = new SGAInfo[nodesProps.length];
    if (infoTmp.length > 1) {
      Server.logInfoMessage("Recebeu dados estticos dos ns de: " + sgaName);
    }
    for (int i = 0; i < nodesProps.length; i++) {
      Map<String, String> properties = SGAUtils.pairToHashMap(nodesProps[i]);
      String nodeName = properties.get(SGA_NODE_NAME.value);
      String platformId = properties.get(SGA_NODE_PLATFORM_ID.value);
      int numProcessors =
        Integer.parseInt(properties.get(SGA_NODE_NUM_PROCESSORS.value));
      int memoryRam =
        (int) Double.parseDouble(properties
          .get(SGA_NODE_MEMORY_RAM_INFO_MB.value));
      //        Integer.parseInt(properties.get(SGA_NODE_MEMORY_RAM_INFO_MB.value));
      int memorySwap =
        (int) Double.parseDouble(properties
          .get(SGA_NODE_MEMORY_SWAP_INFO_MB.value));
      //        Integer.parseInt(properties.get(SGA_NODE_MEMORY_SWAP_INFO_MB.value));
      int clockSpeed =
        (int) Double
        .parseDouble(properties.get(SGA_NODE_CLOCK_SPEED_MHZ.value));
      //        Integer.parseInt(properties.get(SGA_NODE_CLOCK_SPEED_MHZ.value));
      Server.logInfoMessage("Recebeu dados estticos de: " + nodeName);
      infoTmp[i] =
        new SGAInfo(nodeName, platformId, numProcessors, memoryRam, memorySwap,
          clockSpeed, fileSeparator, projectRootDirectory,
          algorithmRootDirectory, sandboxRootDirectory, properties);

      int resourcescounter = 1;

      // Adiciona recursos definidos pelo n
      for (; properties.containsKey(SGA_NODE_RESOURCES_SEQ.value + "."
        + resourcescounter); resourcescounter++) {
        infoTmp[i].addRequirement(properties.get(SGA_NODE_RESOURCES_SEQ.value
          + "." + resourcescounter));
      }
    }
    this.info = infoTmp;
  }

  //  private void updateNodeInfo(Map<String, String> properties){
  // // Propriedades dos ns
  //    Pair[][] nodesProps = sgaProperties.nodesProperties;
  //    SGAInfo[] infoTmp = new SGAInfo[nodesProps.length];
  //    if (infoTmp.length > 1) {
  //      Server.logInfoMessage("Recebeu dados estticos dos ns de: " + sgaName);
  //    }
  //    for (int i = 0; i < nodesProps.length; i++) {
  //      Map<String, String> properties = SGAUtils.pairToHashMap(nodesProps[i]);
  //      String nodeName = properties.get(SGA_NODE_NAME.value);
  //      String platformId = properties.get(SGA_NODE_PLATFORM_ID.value);
  //      int numProcessors =
  //        Integer.parseInt(properties.get(SGA_NODE_NUM_PROCESSORS.value));
  //      int memoryRam =
  //        (int) Double.parseDouble(
  //          properties.get(SGA_NODE_MEMORY_RAM_INFO_MB.value));
  //      int memorySwap =
  //        (int) Double.parseDouble(
  //          properties.get(SGA_NODE_MEMORY_SWAP_INFO_MB.value));
  //      int clockSpeed =
  //        (int) Double.parseDouble(
  //          properties.get(SGA_NODE_CLOCK_SPEED_MHZ.value));
  //      Server.logInfoMessage("Recebeu dados estticos de: " + nodeName);
  //      infoTmp[i] =
  //        new SGAInfo(nodeName, platformId, numProcessors, memoryRam, memorySwap,
  //          clockSpeed, fileSeparator, projectRootDirectory,
  //          algorithmRootDirectory, sandboxRootDirectory, properties);
  //
  //      int resourcescounter = 1;
  //
  //      // Adiciona recursos definidos pelo n
  //      for (; properties.containsKey(
  //        SGA_NODE_RESOURCES_SEQ.value + "."
  //          + resourcescounter); resourcescounter++) {
  //        infoTmp[i].addRequirement(
  //          properties.get(
  //            SGA_NODE_RESOURCES_SEQ.value + "." + resourcescounter));
  //      }
  //    }
  //  }

  /**
   * Configura as propriedades usadas para transferncia de arquivos via CSFS.
   *
   * @throws ServerException se houver falha.
   */
  private void setCSFSProperties() throws ServerException {
    /* Obtm dados do CSFS */
    String csfsHostString = sgaProperties.get(SGA_CSFS_HOST.value);
    String csfsPortString = sgaProperties.get(SGA_CSFS_PORT.value);
    //TODO trocar
    String csfsRootDirString = sgaProperties.get(SGA_CSFS_ROOT_DIR.value);
    try {
      if (csfsHostString == null || csfsHostString.length() == 0) {
        csfsHost = null;
      }
      else {
        csfsHost = csfsHostString;
      }
      if (csfsPortString != null && csfsPortString.length() != 0) {
        csfsPort = Integer.parseInt(csfsPortString);
      }
      Server.logInfoMessage("Obtendo o endereo do CSFS para " + this.name
        + ": " + csfsHost + ":" + csfsPort);
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na obteno do endereo do CSFS de : "
        + this.name, e);
    }
    try {
      if (csfsRootDirString == null || csfsRootDirString.length() == 0) {
        this.csfsRootDir = null;
      }
      else {
        this.csfsRootDir = FileUtils.splitPath(csfsRootDirString);
      }
      Server.logInfoMessage("Obtendo o CSFSRootDir para " + this.name + " : "
        + csfsRootDir);
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na obteno do CSFSRootDir de : "
        + this.name, e);
    }
  }

  /**
   * Inicializa a taxa de transferncia em rede do SGA. Quando a coleta da taxa
   * de transferncia em rede deixar de ser iniciada pelo servidor, essa
   * informao passar a ser informada atravs dos dados dinmicos dos SGAs e
   * esse mtodo deixar de existir.
   *
   * @param transferRate Taxa de transferncia do SGA.
   */
  public void setTransferRate(long transferRate) {
    info[0].setTransferRate(transferRate);
  }

  /**
   * Obtm o estado corrente da mquina hospedeira.
   *
   * @param dynamicInfo Informaes dinmicas do SGA.
   */
  public void updateSGAInfo(SGAProperties dynamicInfo) {
    this.sgaProperties = SGAUtils.pairToHashMap(dynamicInfo.properties);

    String hasDiskAccessStr = sgaProperties.get(SGA_HAS_DISK_ACCESS.value);
    hasDiskAccess =
      (hasDiskAccessStr != null && hasDiskAccessStr
      .equals(SGA_BOOLEAN_TRUE.value)) ? true : false;
    jobsInfo = sgaProperties.get(SGA_JOBS_INFO.value);
    sgaService.logSGAMessage("Recebeu dados dinmicos: " + getName());

    for (int i = 0; i < dynamicInfo.nodesProperties.length; i++) {
      SGAInfo myinfo = info[i];
      HashMap<String, String> data =
        SGAUtils.pairToHashMap(dynamicInfo.nodesProperties[i]);

      double memory_ram_free_perc =
        data.get(SGA_NODE_MEMORY_RAM_FREE_PERC.value).isEmpty() ? 0 : Double
          .parseDouble(data.get(SGA_NODE_MEMORY_RAM_FREE_PERC.value));
      double memory_swap_free_perc =
        data.get(SGA_NODE_MEMORY_SWAP_FREE_PERC.value).isEmpty() ? 0 : Double
          .parseDouble(data.get(SGA_NODE_MEMORY_SWAP_FREE_PERC.value));
      double loadAvg1min =
        data.get(SGA_NODE_LOAD_AVG_1MIN_PERC.value).isEmpty() ? 0 : Double
          .parseDouble(data.get(SGA_NODE_LOAD_AVG_1MIN_PERC.value));
      double loadAvg5min =
        data.get(SGA_NODE_LOAD_AVG_5MIN_PERC.value).isEmpty() ? 0 : Double
          .parseDouble(data.get(SGA_NODE_LOAD_AVG_5MIN_PERC.value));
      double loadAvg15min =
        data.get(SGA_NODE_LOAD_AVG_15MIN_PERC.value).isEmpty() ? 0 : Double
          .parseDouble(data.get(SGA_NODE_LOAD_AVG_15MIN_PERC.value));
      int number_of_jobs =
        data.get(SGA_NODE_NUMBER_OF_JOBS.value).isEmpty() ? 0 : Integer
          .parseInt(data.get(SGA_NODE_NUMBER_OF_JOBS.value));

      if ((memory_ram_free_perc < 0) && (memory_swap_free_perc < 0)
        && (loadAvg1min < 0)) {
        myinfo.setAlive(false);
      }
      else {
        myinfo.setAlive(true);
      }
      myinfo.setRAMFreeMemory(memory_ram_free_perc);
      myinfo.setSwapFreeMemory(memory_swap_free_perc);
      myinfo.setCPULoad(loadAvg1min, loadAvg5min, loadAvg15min);
      myinfo.setNumberOfJobs(number_of_jobs);
    }
    updateTimestamp = System.currentTimeMillis();
    sgaService.logSGAInfo(info, "(Recebido do SGA)");
  }

  /**
   * Obtm a mquina que deve ser contactada para transferncias via CSFS.
   *
   * @return mquina para transferncias via CSFS.
   */
  public String getCSFSHost() {
    return this.csfsHost;
  }

  /**
   * Obtm a porta da mquina que deve ser contactada para transferncias via
   * CSFS.
   *
   * @return porta da mquina para transferncias via CSFS.
   */
  public int getCSFSPort() {
    return this.csfsPort;
  }

  /**
   * Obtm o diretrio que deve ser usado para transferncias via CSFS.
   *
   * @return mquina para transferncias via CSFS.
   */
  public String[] getRemoteRootDir() {
    return this.csfsRootDir;
  }

  /**
   * Retorna lista com os arquivos filhos.
   *
   * @param path - path pai.
   * @return lista com os arquivos filhos.
   */
  public List<ClientSGAFile> getChildren(String path) {
    List<ClientSGAFile> children = new ArrayList<ClientSGAFile>();

    try {
      for (sgaidl.SGAPath pathInfo : remoteReference.getPaths(path)) {
        ClientSGAFile file = buildClientSGAFile(pathInfo);
        if (file != null) {
          children.add(file);
        }
      }
    }
    catch (Exception e) {
      String f = "Erro ao obter as informaes (sga: %s, path: %s, erro: %s)";
      Server.logSevereMessage(String.format(f, name, path, e.getMessage()));
    }

    return children;
  }

  /**
   * Retorna {@link ClientSGAFile} equivalente ao dado path.
   *
   * @param path path.
   * @return {@link ClientSGAFile} equivalente ao dado path.
   */
  public ClientSGAFile getFile(String path) {
    if (path == null) {
      return null;
    }

    try {
      SGAPath pathInfo = remoteReference.getPath(path);
      return buildClientSGAFile(pathInfo);
    }
    catch (Exception e) {
      String f = "Erro ao obter as informaes (sga: %s, path: %s, erro: %s)";
      Server.logSevereMessage(String.format(f, name, path, e.getMessage()));
      return null;
    }
  }

  /**
   * Constri um {@link ClientSGAFile} a partir de um path do sga.
   *
   * @param pathInfo objeto que representa o path do sga.
   * @return {@link ClientSGAFile} equivalente ao path.
   */
  private ClientSGAFile buildClientSGAFile(SGAPath pathInfo) {
    if (!pathInfo.exists) {
      return null;
    }
    String separator = String.valueOf(info[0].getFileSeparator());

    ClientSGAFile file = new ClientSGAFile(name, pathInfo.path);
    file.setSeparator(separator);
    file.setDir(pathInfo.isDir);
    file.setSymbolicLink(pathInfo.isSymbolicLink);
    file.setCanRead(pathInfo.readable);
    file.setCanWrite(pathInfo.writable);
    file.setCanExecute(pathInfo.executable);
    file.setSize(((long) pathInfo.sizeKB) * 1024);
    return file;
  }

  /**
   * Retorna as informaes de um n do SGA.
   *
   * @param index o ndice do n desejado (se for passado zero,  retornado o
   *        dado do n principal).
   * @return um objeto do tipo <code>SGAInfo</code> com as informaes.
   */
  final SGAInfo getInfo(final int index) {
    return info[index];
  }

  /**
   * Retorna as informaes de um n do SGA
   *
   * @param hostname nome do n do SGA
   * @return informaes de um n do SGA
   */
  final SGAInfo getInfo(String hostname) {
    if (hostname == null) {
      return null;
    }
    for (SGAInfo node : info) {
      if (hostname.equals(node.getHostName())) {
        return node;
      }
    }
    return null;
  }

  /**
   * Retorna todas as informaes do SGA
   *
   * @return um array de informaes.
   */
  final SGAInfo[] getAllInfo() {
    return info;
  }

  /**
   * Retorna o numero de maquinas do SGA
   *
   * @return o nmero de mquinas.
   */
  final int getNumNodes() {
    return info.length;
  }

  /**
   * Notificao que o SSI entrou em shutdown.
   */
  void stop() {
    final String sgaName = getName();
    exitAcquisitionDataThread = true;
    Server.logInfoMessage("Terminando:" + sgaName);
    sgaService = null;
  }

  /**
   * Faz log de evento simples.
   *
   * @param eventName nome do evento a ser sinalizado no arquivo.
   */
  private void logTriggeredEvent(final String eventName) {
    final String sgaName = getName();
    final String[] queue = { sgaName, "events" };
    logEvent(queue, eventName);
  }

  /**
   * Desativa o SGA.
   *
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  synchronized void shutdownSGA() throws ServerException {
    logTriggeredEvent("shutdown");
    final String sgaName = getName();
    Server.logWarningMessage("Shutdown para:" + sgaName);
    resetWatchDog();
    try {
      remoteReference.control(SGAControlAction.SHUTDOWN);
    }
    catch (Exception e) {
      throw new ServerException("Falha na desativao do SGA " + this.name, e);
    }
    finally {
      setAlive(false);
    }
  }

  /**
   * Relanamento do SGA.
   *
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  final synchronized void restartSGA() throws ServerException {
    logTriggeredEvent("restart");
    final String sgaName = getName();
    Server.logWarningMessage("Restart para:" + sgaName);
    try {
      remoteReference.control(SGAControlAction.RESTART);
    }
    catch (Exception e) {
      throw new ServerException("Falha na reativao do SGA " + this.name, e);
    }
    finally {
      setAlive(false);
    }
  }

  /**
   * Verifica se o SGA est apto a executar comandos.
   *
   * @return true se o SGA pode executar comandos, false caso contrrio
   */
  final boolean getEnabled() {
    return enabled;
  }

  /**
   * Atualiza disponibilidade do SGA para a execu??o de comandos.
   *
   * @param enabled true se o SGA pode executar comandos, false caso contrrio
   */
  final void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * Obtm o nome do SGA.
   *
   * @return nome do SGA.
   */
  final String getName() {
    return this.name;
  }

  /**
   * Obtm a plataforma do SGA.
   *
   * @return identificao da plataforma
   */
  final String getPlatformId() {
    return info[0].getPlatformId(); // XXX
  }

  /**
   * Verifica se uma plataforma  suportada pelo SGA.
   *
   * @param platform o vetor de plataformas de pesquisa.
   *
   * @return true se a plataforma for suportada, false caso contrrio.
   */
  final boolean isPlatformSupported(final Vector<String> platform) {
    for (SGAInfo node : info) {
      final String nodePlatform = node.getPlatformId();
      if (platform.contains(nodePlatform)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm o nome de um n de uma plataforma especfica
   *
   * @param plats o vetor de plataformas.
   *
   * @return identificao da plataforma
   */
  final String getHostOfPlatform(final Vector<String> plats) {
    for (SGAInfo node : info) {
      final String platform = node.getPlatformId();
      if (plats.contains(platform)) {
        return node.getHostName();
      }
    }
    return null;
  }

  /**
   * Obtm as capacidades dos sgas medidas atravs dos benchmarks.
   *
   * @param capacityType Tipo de capacidade a ser obtida.
   * @return O valor da capacidade.
   */
  final long getCapacity(CapacityType capacityType) {
    return info[0].getCapacity(capacityType);
  }

  /**
   * Verifica se o SGA atualizou seus dados.
   *
   * @return true se o SGA atualizou seus dados
   */
  final boolean getWatchDog() {
    return watchdog;
  }

  /**
   * Indica que o SGA atualizou seus dados.
   */
  final void patWatchDog() {
    watchdog = true;
  }

  /**
   * Indica que o SGA no atualizou seus dados.
   */
  final void resetWatchDog() {
    watchdog = false;
  }

  /**
   * Verifica se o SGA est acessvel.
   *
   * @return true se o SGA est acessvel, false caso contrrio
   */
  final boolean isAlive() {
    return remoteReference != null;
  }

  /**
   * Atualiza acessibilidade do SGA. Isto invalida (zera) as informaes
   * dinmicas se o SGA estiver inacessvel.
   *
   * @param alive true se o SGA estiver acessvel, false caso contrrio.
   */
  final void setAlive(final boolean alive) {
    if (alive) {
      return;
    }
    remoteReference = null;
    for (SGAInfo node : info) {
      node.setAlive(false);
    }
  }

  /**
   * Verifica se o SGA tem acesso a disco.
   *
   * @return true se o SGA tem acesso, false caso contrrio
   */
  final boolean hasDiskAccess() {
    return hasDiskAccess;
  }

  /**
   * Informao dos jobs em execuo no SGA.
   *
   * @return informao dos jobs em execuo no SGA.
   */
  final String getJobsInfo() {
    return jobsInfo;
  }

  /**
   * Indica se o SGA est apto a executar algum comando: est vivo, habilitado e
   * tem acesso ao disco.
   *
   * @return true se o SGA est apto, false caso contrrio
   */
  public boolean mayExecuteCommand() {
    return (isAlive() && getEnabled() && hasDiskAccess());
  }

  /**
   * Verifica se o SGA  um cluster.
   *
   * @return true se o SGA  um cluster, false se o SGA no  um cluster
   */
  final boolean isCluster() {
    return isCluster;
  }

  /**
   * Verifica (remotamente) se o SGA est acessvel.
   *
   * @return true se o SGA est acessvel, false caso contrrio.
   */
  final synchronized boolean ping() {
    try {
      remoteReference.ping();
      setAlive(true);
    }
    catch (Exception e) {
      setAlive(false);
    }
    return isAlive();
  }

  /**
   * Obtm a referncia remota do SGA.
   *
   * @return referncia remota
   */
  final SGADaemonOperations getSGADaemon() {
    return remoteReference;
  }

  /**
   * Armazena a (nova) referncia remota do SGA.
   *
   * @param reference nova referncia remota do SGA
   */
  final synchronized void setRemoteReference(final SGADaemonOperations reference) {
    remoteReference = reference;
  }

  /**
   * Acumula dados de CPU e memria para gravao em arquivo.
   *
   * @see SGAService
   * @see EventLogServiceInterface
   */
  final synchronized void logAuditEvents() {
    final DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    dfs.setDecimalSeparator('.');
    final DecimalFormat df = new DecimalFormat("000.000");
    df.setDecimalFormatSymbols(dfs);
    final int numNodes = info.length;

    final String sgaName = getName();
    for (int i = 0; i < numNodes; i++) {
      final String hName = info[i].getHostName();
      final String[] cpuQueue = { sgaName, hName, "cpu-usage" };
      final double cpu = info[i].getCPULoad1();
      logEvent(cpuQueue, df.format(cpu));

      final String[] ramQueue = { sgaName, hName, "ram-usage" };
      final double ram = 100.0 - info[i].getRAMFreeMemory();
      logEvent(ramQueue, df.format(ram));

      final String[] swpQueue = { sgaName, hName, "swp-usage" };
      final double swp = 100.0 - info[i].getSwapFreeMemory();
      logEvent(swpQueue, df.format(swp));
    }
  }

  /**
   * Acumula dados do histrico de execuo dos algoritmos em arquivo.
   *
   * @see SGAService
   * @see EventLogServiceInterface
   */
  final synchronized void logAlgorithmHistoryEvents() {
    final DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    dfs.setDecimalSeparator('.');
    final DecimalFormat df = new DecimalFormat("000.000");
    df.setDecimalFormatSymbols(dfs);
    final int numNodes = info.length;

    final String sgaName = getName();
    for (int i = 0; i < numNodes; i++) {
      Enumeration<Command> eCmds = commandTable.elements();
      while (eCmds.hasMoreElements()) {
        Command cmd = eCmds.nextElement();
        CommandInfo cmdInfo;
        AlgorithmConfigurator algoConfigurator;
        try {
          cmdInfo = cmd.createCommandInfo();
          algoConfigurator = cmdInfo.getConfigurator();

          StringBuffer parametersString = new StringBuffer();
          HashMap<String, String> parameterTable =
            (HashMap<String, String>) algoConfigurator
            .getParameterValuesByName();
          for (String key : parameterTable.keySet()) {
            parametersString.append(key + "=" + parameterTable.get(key) + " ");
          }

          String algorithmName = algoConfigurator.getAlgorithmName();
          String algorithmVersion =
            algoConfigurator.getAlgorithmVersionId().toString();
          String[] historyQueue =
          { sgaName, "historic", algorithmName + "-" + algorithmVersion + "-" };

          final double cpuPerc =
            (cmdInfo.getCpuPerc() == null) ? SGA.NO_DATA : cmdInfo.getCpuPerc();
          final double swapPerc =
            (cmdInfo.getSwapMemoryPerc() == null) ? SGA.NO_DATA : cmdInfo
              .getSwapMemoryPerc();
          final double userTime =
            (cmdInfo.getUserTimeSec() == null) ? SGA.NO_DATA : cmdInfo
              .getUserTimeSec();
          final double systemTime =
            (cmdInfo.getSystemTimeSec() == null) ? SGA.NO_DATA : cmdInfo
              .getSystemTimeSec();
          final double ramMemory =
            (cmdInfo.getRAMMemoryMB() == null) ? SGA.NO_DATA : cmdInfo
              .getRAMMemoryMB();
          final double virtualMemory =
            (cmdInfo.getVirtualMemorySizeMB() == null) ? SGA.NO_DATA : cmdInfo
              .getVirtualMemorySizeMB();
          final double bytesIn =
            (cmdInfo.getBytesInKB() == null) ? SGA.NO_DATA : cmdInfo
              .getBytesInKB();
          final double bytesOut =
            (cmdInfo.getBytesOutKB() == null) ? SGA.NO_DATA : cmdInfo
              .getBytesOutKB();
          final double diskBytesRead =
            (cmdInfo.getDiskBytesReadKB() == null) ? SGA.NO_DATA : cmdInfo
              .getDiskBytesReadKB();
          final double diskBytesWrite =
            (cmdInfo.getDiskBytesWriteKB() == null) ? SGA.NO_DATA : cmdInfo
              .getDiskBytesWriteKB();
          final double cpuLoad = getCPULoad1();
          final double freeRamMemory = getRAMFreeMemory();
          final double cpuCapacity = getCapacity(CapacityType.CPU);
          final double diskReadCapacity = getCapacity(CapacityType.DISK_READ);
          final double diskWriteCapacity = getCapacity(CapacityType.DISK_WRITE);
          final double netCapacity = getCapacity(CapacityType.NET);

          String information =
            cmd.getId() + "; " + cmd.getUserId().toString() + "; "
              + parametersString + "; " + df.format(cpuLoad) + "; "
              + df.format(freeRamMemory) + "; " + df.format(cpuCapacity) + "; "
              + df.format(diskReadCapacity) + "; "
              + df.format(diskWriteCapacity) + "; " + df.format(netCapacity)
              + "; " + df.format(cpuPerc) + "; " + df.format(ramMemory) + "; "
              + df.format(swapPerc) + "; " + df.format(userTime) + "; "
              + df.format(systemTime) + "; " + df.format(virtualMemory) + "; "
              + df.format(bytesIn) + "; " + df.format(bytesOut) + "; "
              + df.format(diskBytesRead) + "; " + df.format(diskBytesWrite);

          logEvent(historyQueue, information);
        }
        catch (RemoteException e) {
          Server.logSevereMessage(
            "Falha na aquisio do configurador de comando!", e);
        }
      }
    }
  }

  /**
   * Obtm a carga na mquina hospedeira no ltimo minuto.
   *
   * @return carga (percentual) no ltimo minuto.
   */
  final double getCPULoad1() {
    return info[0].getCPULoad1();
  }

  /**
   * Obtm o percentual de memria RAM livre.
   *
   * @return percentual de memria RAM livre.
   */
  final double getRAMFreeMemory() {
    return 100.0 - info[0].getRAMFreeMemory();
  }

  /**
   * Recuperao de um comando.
   *
   * @param cmdId identificador do comando
   * @return o objeto que representa o comando
   */
  final Command getCommand(final Object cmdId) {
    return commandTable.get(cmdId);
  }

  /**
   * Recuperao de todos os comandos que esto executando no SGA.
   *
   * @return um array com os comandos em execuco (objetos Command)
   */
  final Command[] getAllCommands() {
    return commandTable.values().toArray(new Command[0]);
  }

  /**
   * Solicitao de execuo de um comando.
   *
   * @param hostName n do SGA onde o comando deve ser executado. (somente para
   *        grid).
   * @param cmd informaes sobre o comando a ser executado.
   * @param expirationInfoDelay prazo de validade das informaes do comando.
   * @param asScript Indica se o comando dever ser executado como um script.
   * @param logCommandLineInfo Indica se informaes da linha de comando devem
   *        ser logadas.
   * @throws ProjectNotFoundException em caso de falha ao submeter o comando.
   *
   */
  synchronized void executeCommand(final String hostName,
    final csbase.logic.CommandInfo cmd, long expirationInfoDelay,
    boolean asScript, boolean logCommandLineInfo)
      throws ProjectNotFoundException {
    final Object userId = Service.getUser().getId();
    final Command command =
      new Command(hostName, this, cmd, expirationInfoDelay, asScript,
        logCommandLineInfo);
    final Object commId = command.getId();
    commandTable.put(commId, command);
    saveCommandTable();
    /*
     * Aps a criacao do comando, uma Thread  iniciada para solicitar que o
     * ambiente de execucao seja verificado/criado, em seguida o comando remoto
     *  disparado
     */
    commandExecutor.submit(new Runnable() {
      @Override
      public void run() {
        try {
          Service.setUserId(userId);
          command.executeRemoteCommand();
        }
        catch (Throwable e) {
          Server.logSevereMessage("Erro na execuo do comando " + commId, e);
          e.printStackTrace();
        }
        finally {
          Service.setUserId(null);
        }
      }
    });
  }

  /**
   * Grava (serializa) a tabela de comandos em arquivo.
   */
  protected void saveCommandTable() {
    ObjectOutput output = null;
    try {
      output =
        new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(
          getBackupFileName())));
      output.writeObject(commandTable);
    }
    catch (OperationFailureException ex) {
      Server.logSevereMessage(
        "Erro na validao do diretrio de persistncia de comandos.", ex);
    }
    catch (IOException ex) {
      Server.logSevereMessage("Erro ao gravar a tabela de comandos.", ex);
    }
    finally {
      try {
        if (output != null) {
          output.close();
        }
      }
      catch (IOException ex) {
        Server.logSevereMessage(
          "Erro ao fechar stream de gravao da tabela de histrico.", ex);
      }
    }
  }

  /**
   * Recuperao de comando
   *
   * @param cmdId Identificador do comando.
   * @param cmdReference referncia remota do comando.
   * @return um indicativo de recuperao correta do comando.
   */
  final synchronized boolean commandRetrieved(final Object cmdId,
    final SGACommand cmdReference) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      command.setRemoteReference(cmdReference);
      return true;
    }
    Server.logSevereMessage("Impossvel recuperar comando (" + cmdId + ")");
    return false;
  }

  /**
   * Finalizao de comando pelo usurio (kill)
   *
   * @param cmdId identificador do comando.
   * @return {@code true} se o comando foi cancelado ou se no existe. Retorna
   *         {@code false} se ocorreu uma falha no cancelamento e ele no foi
   *         feito.
   */
  final synchronized boolean killCommand(final Object cmdId) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      if (!command.kill()) {
        return false;
      }
      commandTable.remove(cmdId);
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Impossvel finalizar comando (" + cmdId + ")!");
    }
    return true;
  }

  /**
   * Finalizao de comando pelo administrador (kill). Remove o objeto do
   * comando mesmo que tenha ocorrido erro. Deve ser usado no caso em que no h
   * mais possibilidade do comando estar executando. Por exemplo, ocorreu
   * problema de hardware na mquina.
   *
   * @param cmdId identificador do comando.
   */
  final synchronized void killCommandAnyway(final Object cmdId) {
    final Command command = getCommand(cmdId);
    if (command == null) {
      return;
    }
    command.kill();
    commandTable.remove(cmdId);
    saveCommandTable();
  }

  /**
   * Finalizao de um comando com falha.
   *
   * @param cmdId identificador do comando
   * @param cause causa da falha no comando
   */
  final synchronized void failCommand(final Object cmdId,
    FailureFinalizationType cause) {
    final Command command = getCommand(cmdId);

    switch (cause) {
      case FAILED_SETUP_EXECUTION_ENVIRONMENT:
        // Neste caso de falha o comando  colocado na fila so Scheduler (o estado  modificado para SCHEDULED)
        CommandInfo info = command.getCommandInfo();
        info.setStatus(CommandStatus.SCHEDULED);
        commandTable.remove(cmdId);
        saveCommandTable();
        if (info.isAutomatic()) {
          info.setSGAName(null);
        }
        return;
      default:
        if (command != null) {
          commandTable.remove(cmdId);
          command.failed(cause);
          saveCommandTable();
          sgaService.handleCommandInitFailure(command);
        }
        else {
          Server.logSevereMessage("Comando nao encontrado (" + cmdId + ")!");
        }
        return;
    }
  }

  /**
   * Finalizao de um comando.
   *
   * @param cmdId identificador do comando
   */
  final synchronized void lostCommand(final String cmdId) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      /*
       * primeiro remove o comando da tabela para no causar inconsistncia.
       */
      commandTable.remove(cmdId);
      command.lost();
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Comando nao encontrado (" + cmdId + ")!");
    }
  }

  /**
   * Finalizao de um comando.
   *
   * @param cmdId identificador do comando
   */
  final synchronized void finishCommand(final String cmdId) {
    if (commandTable.remove(cmdId) != null) {
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Impossvel finalizar comando (" + cmdId + ")!");
    }
  }

  /**
   * Atualiza a representao de um SGA quando ocorre um "re-registro" (o SGA
   * havia cado e se registra novamente).
   *
   * @param reference referncia remota para o SGA.
   * @param properties propriedades atualizadas do SGA e de seus ns.
   *
   * @throws ServerException se houver problemas com as propriedades do CSFS.
   */
  final void updateSGA(final SGADaemonOperations reference,
    SGAProperties properties) throws ServerException {
    logTriggeredEvent("re-register");
    this.remoteReference = reference;
    this.registerTimestamp = java.lang.System.currentTimeMillis();
    setSGAProperties(properties);
    this.watchdog = true;
    this.enabled = true;
    this.hasDiskAccess = true;
    this.isCluster = (info.length > 1) ? true : false;
    final Enumeration<Command> eCmds = commandTable.elements();
    while (eCmds.hasMoreElements()) {
      final Command cmd = eCmds.nextElement();
      cmd.setRemoteReference(null);
    }
    setCSFSProperties();
  }

  // TODO Removemos tudo de Capacities, certo?!?!?!
  //  /**
  //   * Preenche uma tabela que mapeia um tipo de capacidade encontrado na idl do
  //   * SGA ao tipo de capacidade definida na classe Java CapacityType. Essa tabela
  //   * deve ser preenchida uma nica vez visto que se um novo tipo de capacidade
  //   * for adicionada/removida dos SGAs, novos stubs devem ser gerados.
  //   *
  //   * @return Mapa que relaciona uma capacidade definida na idl  capacidade Java
  //   *         do SGA.
  //   */
  //  private HashMap<Integer, CapacityType> fillCapacityMap() {
  //    capacityMap = new HashMap<Integer, CapacityType>();
  //        CType fieldEx = CType.CPU;
  //        for (CapacityType type : CapacityType.values()) {
  //          try {
  //            Field field = fieldEx.getClass().getField(type.toString());
  //            CType cType = (CType) field.get(field);
  //            capacityMap.put(Integer.valueOf(cType.value()), type);
  //          }
  //          catch (NoSuchFieldException e) {
  //            Server.logSevereMessage("Tipo de capacidade " + type.toString()
  //              + " invlida.\n" + "Erro: " + e);
  //            return null;
  //          }
  //          catch (IllegalAccessException e) {
  //            Server.logSevereMessage("Capacidade " + type.toString()
  //              + " inacessvel.\n" + "Erro: " + e);
  //            return null;
  //          }
  //        }
  //    return capacityMap;
  //  }

  /**
   * Constri a representao de um SGA.
   *
   * @param srv o servio de SGAs.
   * @param sgaName nome do SGA
   * @param reference referncia remota para o SGA
   * @param sgaProperties informaes estticas dos ns
   * @param cmdExecPoolSize tamanho do pool the thread de submisso dos comandos
   *        para o SGA
   * @throws ServerException se no  possvel obter a configurao do SGA
   */
  SGA(final SGAService srv, final String sgaName,
    final SGADaemonOperations reference, SGAProperties sgaProperties, int cmdExecPoolSize)
      throws ServerException {
    this.remoteReference = reference;
    this.registerTimestamp = java.lang.System.currentTimeMillis();
    this.sgaService = srv;
    this.name = sgaName;
    // TODO Removemos tudo de Capacities, certo?!?!?!
    // Antes de chamar setStaticData  preciso mapear as capacidades na idl
    // para a classe java atravs do mtodo fillCapacityMap.
    //    if (capacityMap == null) {
    //      capacityMap = fillCapacityMap();
    //    }
    setSGAProperties(sgaProperties);
    logTriggeredEvent("register");
    this.watchdog = true;
    this.enabled = true;
    this.isCluster = (info.length > 1) ? true : false;
    this.hasDiskAccess = true;
    this.jobsInfo = null;
    loadCommandTable();
    try {
      reference.ping();
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na comunicao com o SGA: " + this.name,
        e);
    }
    setCSFSProperties();
    createCommandDataAcquisitionThread();
    // Cria um executor que usa um pool de threads para submisso de comandos.
    this.commandExecutor = Executors.newFixedThreadPool(cmdExecPoolSize);
  }

  /**
   * Atualiza o estado dos comandos. Precisa atualizar os comandos mesmo que o
   * SGA esteja sem conexo, caso contrrio, os dados dos comandos ficam
   * invlidos.
   *
   * @return o nmero de vezes que os comandos foram atualizados.
   */
  private long updateCommands() {
    long minupdate = Long.MAX_VALUE;
    Command[] sgaCmds = getAllCommands();
    for (Command cmd : sgaCmds) {
      try {
        if (cmd.isAlive()) {
          cmd.update();
          long nupdates = cmd.getNUpdates();
          minupdate = Math.min(minupdate, nupdates);
        }
      }
      catch (ServerException e) {
        String message = e.getMessage();
        Server.logSevereMessage(message, e);
      }
    }
    return minupdate;
  }

  /**
   * Criao da thread que faz a aquisio dos dados dos Comandos nos SGAs
   */
  private void createCommandDataAcquisitionThread() {
    /*
     * Faz um update de freqencia incremental de STEP_SECONDS segundos, at
     * atingir o tempo definido em updateSleepTime. O disparo de um novo
     * comando, faz a freqencia do update voltar para STEP_SECONDS. Isso faz
     * com que os primeiros updates de comandos ocorram com uma frequncia
     * maior, para dar um feedback mais rpido para o usurio.
     */
    final int STEP_SECONDS = 2;
    final int sgaCommandUpdateInterval = sgaService.getCommandsUpdateInterval();
    final long updateSleepTime = sgaCommandUpdateInterval * MILLIS;
    Thread acquisitionCommandDataThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitAcquisitionDataThread) {
            long stepSleepTime = Long.MAX_VALUE;
            long ntimes = updateCommands();
            if (ntimes != Long.MAX_VALUE) {
              stepSleepTime = (ntimes + 1) * STEP_SECONDS * MILLIS; // Estoura long?
            }
            stepSleepTime = Math.min(stepSleepTime, updateSleepTime);
            try {
              Thread.sleep(stepSleepTime);
            }
            catch (InterruptedException ie) {
              Server
              .logWarningMessage("Aquisio de dados de Comandos SGA interrompido!");
            }
          }
          Server
          .logWarningMessage("Finalizando thread de aquisio de dados de comandos...");
        }
        catch (Throwable t) {
          String msg =
            (new Date()) + " - Erro na aquisio de dados de comandos:";
          System.err.println(msg);
          t.printStackTrace(System.err);
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitAcquisitionDataThread = false;
    acquisitionCommandDataThread.setName(this.getClass().getSimpleName() + "::"
      + "AcquisitionCommandDataThread");
    acquisitionCommandDataThread.start();
  }

  /**
   * Cria uma tabela vazia e a persiste.
   */
  private void createNewCommandTable() {
    this.commandTable = new Hashtable<Object, Command>();
    saveCommandTable();
  }

  /**
   * Carrega a tabela de comandos, normalmente persistida em arquivo. Caso o
   * arquivo no seja encontrado, uma tabela vazia  persistida.
   */
  @SuppressWarnings("unchecked")
  private void loadCommandTable() {
    // TODO #4601 WA para que as referncias de CommandInfo associadas um
    // comando nas filas do Scheduler e do SGA sejam as mesmas.
    CommandInfoCache.enable();

    Hashtable<Object, Command> table = null;
    ObjectInput input = null;
    try {
      File file = new File(getBackupFileName());
      if (!file.exists()) {
        createNewCommandTable();
        return;
      }
      input =
        new ObjectInputStream(
          new BufferedInputStream(new FileInputStream(file)));
      table = (Hashtable<Object, Command>) input.readObject();
      try {
        for (Iterator<Map.Entry<Object, Command>> entries = table.entrySet().iterator(); entries.hasNext(); ) {
          Map.Entry<Object, Command> entry = entries.next();
          Command command = entry.getValue();
          //Comandos na fila do SGA com os estados INIT ou UPLOADING so
          // removidos, pois sero escalonados novamente
          if(command.getStatus() == CommandStatus.SCHEDULED || command.getStatus() == CommandStatus.INIT || command.getStatus() == CommandStatus.UPLOADING){
            entries.remove();
          }else{
            Service.setUserId(command.getUserId());
            command.setService(sgaService);
            command.setSGA(this);
          }
        }
      }
      finally {
        Service.setUserId(null);
      }
      this.commandTable = table;
    }
    catch (OperationFailureException ex) {
      Server.logSevereMessage(
        "Erro na validao do diretrio de persistncia de comandos.", ex);
      createNewCommandTable();
      return;
    }
    catch (IOException ex) {
      Server.logSevereMessage("Erro ao ler a tabela de comandos.", ex);
      createNewCommandTable();
      return;
    }
    catch (ClassNotFoundException ex) {
      Server.logSevereMessage(
        "No foi encontrada nenhuma classe no arquivo especificado.", ex);
      createNewCommandTable();
      return;
    }
    finally {
      try {
        if (input != null) {
          input.close();
        }
      }
      catch (IOException ex) {
        Server.logSevereMessage(
          "Erro ao fechar stream de leitura da tabela de histrico.", ex);
        createNewCommandTable();
        return;
      }
    }
  }

  /**
   * Retorna o nome do arquivo para backup de comandos em execuo.<br>
   * Formato: &lt;diretrio de comandos&gt;+&lt;nome do sga&gt;+&lt;sufixo&gt;.
   *
   * @return nome do arquivo para backup de comandos em execuo.
   *
   * @throws OperationFailureException em caso de erro na validao do diretrio
   *         de persistncia no sistema de arquivos.
   */
  private String getBackupFileName() throws OperationFailureException {
    return sgaService.getCommandsPersistencyDirectoryName() + File.separator
      + name + EXECUTION_COMMANDS_FILENAME_SUFFIX;
  }

  /**
   * Verifica se o caminho especificado existe no sistema de arquivos sobre o
   * qual o SGA atua.
   *
   * @param path caminho relativo ao diretrio de execuo do SGA ou absoluto
   *        (contanto que seja um caminho do sistema de arquivos do SGA e no do
   *        servidor).
   *
   * @return <code>true</code> caso o caminho seja encontrado.
   *
   * @throws ServerException se houver falha no acesso ao SGA
   */
  boolean isDir(String path) throws ServerException {
    try {
      SGAPath sgaPath = remoteReference.getPath(path);
      if (sgaPath == null) {
        return false;
      }
      return sgaPath.isDir;
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na verificao do acesso ao caminho '"
        + path + "' junto ao SGA " + this.name, e);
    }
  }

  /**
   * Obtm o valor de uma propriedade do sga definida por uma chave.
   *
   * @param key a chave do valor que se deseja obter.
   *
   * @return o valor de uma propriedade do sga definida por uma chave.
   */
  String getProperty(String key) {
    return (key == null) ? null : this.sgaProperties.get(key);
  }

  /**
   * Obtm o mapa de todas as propriedades do SGA.
   *
   * @return o mapa com todas as propriedades.
   */
  public Map<String, String> getAllProperties() {
    return this.sgaProperties;
  }

  /**
   * Obtm o timestamp do registro do SGA.
   *
   * @return o timestamp do registro
   */
  public long getRegisterTimestamp() {
    return this.registerTimestamp;
  }

  /**
   * Obtm o timestamp da ltima atualizao de dados.
   *
   * @return o timestamp da ltima atualizao de dados
   */
  public long getUpdateTimeStamp() {
    return this.updateTimestamp;
  }

  /**
   * Indica se o SGA utiliza algum mecanismo de transferncia de arquivos.
   *
   * @return true se utiliza algum mecanismo de transferncia de arquivos e
   * false se no utiliza.
   */
  public boolean hasDataTransferMechanism() {
    return sgaProperties.get(SGAService.SGA_TRANSFER_MECHANISM_KEY) != null
        || getCSFSHost() != null;
  }
}
