package csbase.server.services.schedulerservice;

import csbase.logic.CommandInfo;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.server.Server;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.schedulerservice.filter.SGAFilterByCommandRequirements;
import tecgraf.javautils.core.io.FileUtils;

import java.io.File;
import java.rmi.RemoteException;
import java.util.*;

/**
 * Poltica de escalonamento que privilegia os SGAs com a melhor combinao
 * entre as capacidades de leitura/escrita em disco e transferncia de dados na
 * rede. Os melhores SGAs so definidos com base em uma equao com variveis
 * alimentadas pelos resultados dos benchmarks aplicados aos SGAs.
 * 
 * @author valeria
 * 
 */
public class IOCapacityPolicy implements SchedulerPolicyInterface {

	public Map<CommandInfo, SGASet> chooseServer(List<CommandInfo> commands,
			List<SGASet> servers) {
		BalancedCommandDistribution bdc = new BalancedCommandDistribution(servers);
		
		Map<CommandInfo, SGASet> allocation = new HashMap<CommandInfo, SGASet>();
		for (CommandInfo cmd : commands) {
			List<SGASet> filteresSGAs = SGAFilterByCommandRequirements.filter(cmd,
					servers);
			if (filteresSGAs == null) {
				continue;
			}

			List<SGASet> sgas = getSortedSGASetByIOCapacity(cmd,
					filteresSGAs);

			if ((sgas != null) && (sgas.size() != 0)) {
				SGASet choosedServer = bdc.chooseLightest(sgas);
				allocation.put(cmd, choosedServer);
			}
		}
		return allocation;
	}

	/**
	 * Ordena os SGAs de acordo com as suas capacidades de E/S de dados.
	 * 
	 * @param command
	 *            Comando para execuo
	 * @param servers
	 *            Lista de servidores disponveis
	 * 
	 * @return Lista de servidores ordenados decrescentemente pela capacidade de
	 *         E/S
	 */
	private List<SGASet> getSortedSGASetByIOCapacity(CommandInfo command,
			List<SGASet> servers) {
		long inputFilesLength;
		Hashtable<String, Long> platfBinFilesLengthList;
		List<SGASet> sgasSets = new ArrayList<SGASet>();

		AlgorithmConfigurator algorithmConfigurator;
		try {
			algorithmConfigurator = command.getConfigurator();
		} catch (RemoteException e) {
			Server.logSevereMessage(
					"No foi possvel obter o configurador do comando "
							+ command.getId(), e);
			return null;
		}

		inputFilesLength = getInputFilesLength(algorithmConfigurator,
				command.getId());
		platfBinFilesLengthList = getExecFilesLength(algorithmConfigurator,
				servers);

		Collections.sort(sgasSets, new SGASetIOComparator(inputFilesLength,
				platfBinFilesLengthList));
		return sgasSets;
	}

	/**
	 * Calcula a soma dos tamanhos de todos os arquivos de entrada do algoritmo
	 * a ser executado.
	 * 
	 * @param algorithmConfigurator
	 *            o configurador do comando.
	 * @param commandId
	 *            o identificador do comando.
	 * 
	 * @return A soma dos tamanhos de todos os arquivos de entrada do algoritmo
	 */
	private long getInputFilesLength(
			AlgorithmConfigurator algorithmConfigurator, String commandId) {
	    Set<FileURLValue> inputFiles = algorithmConfigurator.getInputFiles();
	    Iterator<FileURLValue> inputFileIterator = inputFiles.iterator();

		long filesLength = 0;
		ProjectService projectService = ProjectService.getInstance();
		for (int i = 0; i < inputFiles.size(); i++) {
			String[] path = FileUtils.splitPath(inputFileIterator.next()
					.getPath());
			if (projectService.existsFile(commandId, path)) {
				filesLength += projectService.fileSize(commandId, path);
			}
		}
		filesLength /= 1024;
		return filesLength;
	}

	/**
	 * XXX - [CSBASE-601] Rever o tratamento para o caso de fluxo nas polticas
	 * de escalonamento.
	 * 
	 * Obtm o tamanho dos arquivos executveis do algoritmo para cada
	 * plataforma existente entre os servidores disponveis.
	 * 
	 * @param algoConfigurator
	 *            o configurador do comando.
	 * @param serversNames
	 *            Servidores disponveis
	 * 
	 * @return Uma lista com todos os tamanhos dos arquivos binrios de acordo
	 *         com cada plataforma de execuo
	 */
	private Hashtable<String, Long> getExecFilesLength(
			AlgorithmConfigurator algoConfigurator, List<SGASet> servers) {
		long filesLength;
		Hashtable<String, Long> platformsList = new Hashtable<String, Long>();
		AlgorithmVersionInfo algoVersionInfo = algoConfigurator
				.getAlgorithmVersion();

		for (int i = 0; i < servers.size(); i++) {
			SGASet sgaSet = servers.get(i);
			if (sgaSet != null) {
				if (!platformsList.containsKey(sgaSet.getPlatformId())) {
					filesLength = 0;
					String binDirPath = algoVersionInfo.getDirPath()
							+ File.separator + algoVersionInfo.getBinDirName()
							+ File.separator + sgaSet.getPlatformId();
					java.io.File binDir = new java.io.File(binDirPath);
					File binFiles[] = binDir.listFiles();
					for (int j = 0; j < binFiles.length; j++) {
						filesLength += binFiles[j].length();
					}
					filesLength /= 1024;
					platformsList.put(sgaSet.getPlatformId(), new Long(
							filesLength));
				}
			}
		}

		return platformsList;
	}
}
