package csbase.server.services.pluginservice;

import java.io.File;
import java.io.FilenameFilter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.java.plugin.ObjectFactory;
import org.java.plugin.Plugin;
import org.java.plugin.PluginLifecycleException;
import org.java.plugin.PluginManager;
import org.java.plugin.PluginManager.PluginLocation;
import org.java.plugin.registry.Extension;
import org.java.plugin.registry.ExtensionPoint;
import org.java.plugin.registry.PluginDescriptor;
import org.java.plugin.standard.StandardPluginLocation;
import org.java.plugin.util.ExtendedProperties;

import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Servio que gerencia os plugins instalados no servidor.
 *
 * @author Tecgraf/PUC-Rio
 */
public class PluginService extends Service {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "PluginService";

  /** Caminho para o repositrio de plugins; */
  private final String pluginRepositoryPath;

  /** Gerenciador dos plugins */
  private PluginManager pluginManager;

  /** Indica se existem plugins carregados na inicializao do servio */
  private boolean availablePlugins;

  /**
   * Construtor padro.
   *
   * @throws ServerException se ocorrer falha na construo desse servio
   */
  protected PluginService() throws ServerException {
    super(SERVICE_NAME);
    this.pluginRepositoryPath = this.getStringProperty("base.plugin.dir");
  }

  /**
   * Constri a instncia do servio.
   *
   * @throws ServerException em caso de erro na criao d servio.
   */
  public static void createService() throws ServerException {
    new PluginService();
  }

  /**
   * Retorna a instncia do servio.
   *
   * @return o servio.
   */
  public static PluginService getInstance() {
    return (PluginService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initService() throws ServerException {
    this.availablePlugins = loadPlugins();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void shutdownService() throws ServerException {
  }

  /**
   * {@inheritDoc}
   *
   * Esse servico nao possui observadores.
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Obtm um plugin de acordo com o seu identificador.
   *
   * @param pluginId identificador do plugin
   * @return o plugin
   * @throws ServerException
   */
  public Plugin getPlugin(String pluginId) throws ServerException {
    try {
      return pluginManager.getPlugin(pluginId);
    }
    catch (PluginLifecycleException e) {
      throw new ServerException("Erro ao recuperar o plugin " + pluginId, e);
    }
  }

  /**
   * Obtm as classes que implementam os pontos de extenso de um determinado
   * plugin.
   *
   * @param pluginId identificador do plugin
   * @param extensionPointId identificador da extenso associada aos pontos de
   *        extenso
   * @return um mapa com o nome que identifica o ponto de extenso e a classe
   *         correnspondente que o implementa
   * @throws ServerException se houver erro ao obter o mapeamento
   */
  public Map<String, Class<?>> getPluginExtensionPoints(String pluginId,
    String extensionPointId) throws ServerException {
    if (!this.availablePlugins) {
      return null;
    }
    if (!pluginManager.getRegistry().isExtensionPointAvailable(
      pluginId, extensionPointId)) {
      return null;
    }
    Map<String, Class<?>> extensionPoints = new HashMap<String, Class<?>>();
    ExtensionPoint point =
      pluginManager.getRegistry().getExtensionPoint(pluginId, extensionPointId);
    for (Iterator<Extension> it =
      point.getConnectedExtensions().iterator(); it.hasNext();) {
      Extension ext = it.next();
      PluginDescriptor descr = ext.getDeclaringPluginDescriptor();
      try {
        pluginManager.activatePlugin(descr.getId());
        ClassLoader classLoader = pluginManager.getPluginClassLoader(descr);
        Class<?> pluginCls =
          classLoader.loadClass(ext.getParameter("class").valueAsString());
        String pluginName = ext.getParameter("name").valueAsString();
        extensionPoints.put(pluginName, pluginCls);
      }
      catch (PluginLifecycleException | ClassNotFoundException e) {
        Server.logSevereMessage("Erro ao obter os pontos de extenso", e);
        return null;
      }
    }
    return extensionPoints;
  }

  /**
   * Carrega os plugins que esto no repositrio de plugins do servidor.
   *
   * @return {@code true} se houve carga de plugins ou {@code false} caso
   *         contrrio
   *
   * @throws ServerException se ocorrer algum erro durante a carga dos plugins.
   */
  private boolean loadPlugins() throws ServerException {
    File pluginsDir = new File(pluginRepositoryPath);

    ExtendedProperties config = new ExtendedProperties();
    config.put(
      "org.java.plugin.PathResolver",
      "org.java.plugin.standard.ShadingPathResolver");
    config.put(
      "org.java.plugin.standard.ShadingPathResolver.shadowFolder",
      pluginsDir.getAbsolutePath() + "/temp/.jpf-shadow");
    config.put(
      "org.java.plugin.standard.ShadingPathResolver.unpackMode", "smart");

    pluginManager = ObjectFactory.newInstance(config).createManager();

    File[] plugins = pluginsDir.listFiles(new FilenameFilter() {
      @Override
      public boolean accept(File dir, String name) {
        return name.toLowerCase().endsWith(".zip");
      }
    });
    if (plugins == null) {
      return false;
    }
    try {
      PluginLocation[] locations = new PluginLocation[plugins.length];
      for (int i = 0; i < plugins.length; i++) {
        locations[i] = StandardPluginLocation.create(plugins[i]);
      }
      pluginManager.publishPlugins(locations);
      return true;
    }
    catch (Exception e) {
      throw new ServerException(e);
    }
  }
}
