package csbase.server.services.opendreamsservice.opendreams.v2_0;

import java.io.Serializable;

import tecgraf.openbus.DRMAA.v2_0.DeniedByDrmsException;
import tecgraf.openbus.DRMAA.v2_0.DrmaaCallback;
import tecgraf.openbus.DRMAA.v2_0.DrmaaCapability;
import tecgraf.openbus.DRMAA.v2_0.InternalException;
import tecgraf.openbus.DRMAA.v2_0.InvalidArgumentException;
import tecgraf.openbus.DRMAA.v2_0.JobSession;
import tecgraf.openbus.DRMAA.v2_0.Version;
import tecgraf.openbus.opendreams.v2_0.IOpenDreamsPOA;
import csbase.server.Server;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.schedulerservice.Base32IdGenerator;

/**
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class SessionManager extends IOpenDreamsPOA implements Serializable {

	/**
	 * Default String.
	 */
	public static final String UNDEFINED = "indefinido";

  /**
   * Default String.
   */
  public static final Integer UNSET_INTEGER = 999999999;

	/**
	 * Nome do diretório para persistência de sessões
	 */
	private static final String JOB_SESSIONS_DIR = "jobsessions";

	/**
	 * Mapa persistente de sessões por session name.
	 */
	private PersistentMap<String, OpenDreamsJobSession> jobSessions;

	/**
	 * Constrói o objeto que implementa a interface <code>IOpenDreams</code>.
	 *
	 */
	public SessionManager() {
		this.jobSessions = new PersistentMap<String, OpenDreamsJobSession>(
				PersistentObject.generatePath(JOB_SESSIONS_DIR, "jobsessions.dat"));
	}

	/**
	 * {@inheritDoc}
	 *
	 * A read-only system identifier denoting the DRM system targeted by the
	 * implementation, e.g., “LSF” or “GridWay”. Implementations SHOULD NOT make
	 * versioning information of the particular DRM system a part of this
	 * attribute value. The value is only intended as informative output for
	 * application users.
	 */
	@Override
	public String drmsName() {
		return Server.getInstance().getSystemName();
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute provides the DRM-system specific version information. The
	 * value is only intended as informative output for application users
	 */
	@Override
	public Version drmsVersion() {
		csbase.logic.Version version = csbase.logic.Version.getInstance();
		return new Version(String.valueOf(version.getMajorVersion()), String.valueOf(version.getMinorVersion()));
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute contains a string identifying the vendor of the DRMAA
	 * implementation. The value is only intended as informative output for
	 * application users.
	 */
	@Override
	public String drmaaName() {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * {@inheritDoc}
	 *
	 * This attribute provides the minor / major version number information for
	 * the DRMAA implementation. The major version number MUST be the constant
	 * value “2”, the minor version number SHOULD be used by the DRMAA
	 * implementation for expressing its own versioning information.
	 */
	@Override
	public Version drmaaVersion() {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * {@inheritDoc}
	 *
	 * This method allows to test if the DRMAA implementation supports a feature
	 * specified as optional. The allowed input values are specified in the
	 * DrmaaCapability enumeration.
	 */
	@Override
	public boolean supports(DrmaaCapability capability) {
		return false;
	}

	/**
	 * {@inheritDoc}
	 *
	 *
	 * @param sessionName
	 *            Denotes a unique name to be used for the new session. If a
	 *            session with such a name already exists, the method MUST throw
	 *            an InvalidArgumentException. In all other cases, including if
	 *            the provided name has the value UNSET, a new session MUST be
	 *            created with a unique name generated by the implementation
	 * @param contact
	 *            in some of the interface methods SHALL allow the application
	 *            to specify which DRM system instance to use. A contact string
	 *            represents a specific installation of a specific DRM system,
	 *            e.g., a Condor central manager machine at a given IP address,
	 *            or a Grid Engine ‘root’ and ‘cell’. Contact strings are always
	 *            implementation-specific and therefore opaque to the
	 *            application. If contact has the value UNSET, a default DRM
	 *            system SHOULD be contacted. The manual configuration or
	 *            automated detection of a default contact string is
	 *            implementation-specific.
	 * @throws DeniedByDrmsException
	 * @throws InternalException
	 * @throws InvalidArgumentException
	 *
	 */
	@Override
	public JobSession createJobSession(String sessionName, String contact)
			throws DeniedByDrmsException, InvalidArgumentException, InternalException {

		String userId = UNDEFINED;

		try {
			Server.logFineMessage("OpenDreams: createJobSession (" + sessionName + ", " + contact + ")");
			System.out.println("OpenDreams: createJobSession (" + sessionName + ", " + contact + ")");

			userId = OpenBusService.getInstance().initCSBaseAccess();
			checkUser(userId);

			if (!isUnique(sessionName)) {
				throw new InvalidArgumentException("O nome da sessão (" + sessionName + ") já existe.");
			}
			sessionName = sessionName.equals(UNDEFINED) ? generateUniqueSessionName(userId) : sessionName;

			contact = contact == null ? UNDEFINED : contact;
			OpenDreamsJobSession session = new OpenDreamsJobSession(sessionName, contact, userId);

			jobSessions.put(sessionName, session);
			return session.createCorbaObjReference();
		} catch (DeniedByDrmsException e) {
			String msg = "Falha na tentativa de criar uma sessão do usuário " + userId + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (InvalidArgumentException e) {
			String msg = "Falha na tentativa de criar uma sessão do usuário " + userId + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (Throwable e) {
			String msg = "Erro ao criar a sessão " + sessionName + " do usuário " + userId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * The method is used to open a persisted JobSession or ReservationSession
	 * instance that has previously been created under the given sessionName.
	 * The implementation MUST support the case that the session have been
	 * created by the same application or by a different application running on
	 * the same machine.
	 *
	 * @throws InvalidArgumentException
	 * @throws DeniedByDrmsException
	 * @throws InternalException
	 */
	@Override
	public JobSession openJobSession(String sessionName)
			throws InvalidArgumentException, DeniedByDrmsException, InternalException {

		String userId = UNDEFINED;

		try {
			Server.logFineMessage("SessionManager: openJobSession (" + sessionName + ")");
			System.out.println("SessionManager: openJobSession (" + sessionName + ")");

			userId = OpenBusService.getInstance().initCSBaseAccess();
			checkUser(userId);

			if (!jobSessions.containsKey(sessionName)) {
				throw new InvalidArgumentException("A sessão " + sessionName + " não existe.");
			} else {
				// TODO Fechar obj com poa.deactivate_object
				// TODO poa.servant_to_reference cria um novo objeto?
				OpenDreamsJobSession session = jobSessions.get(sessionName);
				session.onResume();
				return session.createCorbaObjReference();
			}
		} catch (InvalidArgumentException e) {
			String msg = "Falha na tentativa de obter a sessão " + sessionName + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (DeniedByDrmsException e) {
			String msg = "Falha na tentativa de obter a sessão " + sessionName + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (Throwable e) {
			String msg = "Falha na tentativa de obter a sessão " + sessionName + ".";
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void closeJobSession(JobSession s) {
		// TODO: Analisar método
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroyJobSession(String sessionName) {
		// TODO: Analisar método
		jobSessions.remove(sessionName);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getJobSessionNames() {
		return jobSessions.keySet().toArray(new String[]{});
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void registerEventNotification(DrmaaCallback callback) {
	}

	/**
	 * Criação de um sessionName único.
	 *
	 * @param userId
	 *            Identificador do usuário
	 * @return uma string que representa uma sessão única.
	 */
	private String generateUniqueSessionName(String userId) {
		/*
		 * Foram removidos alguns parâmetros na geração do id em relação a
		 * geração de ids de comando. TODO: Verificar relevância dos parâmetros
		 * removidos.
		 */
		return String.format("%s@%s.%s", userId, Server.getInstance().getSystemName(),
				new Base32IdGenerator().generateId());
	}

	/**
	 * @param sessionName
	 *            Identificador da sessão fornecida ao método
	 *            {@link #createJobSession(String, String) createJobSession}
	 * @return {@code true} se o identificador é único
	 */
	private boolean isUnique(String sessionName) {
		return !jobSessions.containsKey(sessionName);
	}

	/**
	 * Verifica se o usuário é cadastrado no CSBase.
	 *
	 * @param userId
	 *            identificador do usuário
	 * @throws DeniedByDrmsException
	 *             caso não exista um usuário com o identificador
	 */
	public static void checkUser(String userId) throws DeniedByDrmsException {
		if (AdministrationService.getInstance().getUser(userId) == null) {
			throw new DeniedByDrmsException("O usuário " + userId + " não existe.");
		}
	}

}
