package csbase.servlet;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.rmi.Naming;
import java.util.Date;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.oreilly.servlet.multipart.FilePart;
import com.oreilly.servlet.multipart.MultipartParser;
import com.oreilly.servlet.multipart.ParamPart;
import com.oreilly.servlet.multipart.Part;

import csbase.exception.HttpServiceException;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.HttpServiceInterface;
import csbase.remote.ServerEntryPoint;

/**
 * Servlet responsvel pelo upload de arquivos para o servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class UploadServlet extends AbstractServlet {
  /*
   * ATENO!!! Temos que tomar cuidado com atributos de objeto que sejam
   * atualizveis. Lembrando que o servlet no  "thread safe", se for preciso
   * que um atributo seja de objeto, o trecho de cdigo que o atualiza deve ser
   * sincronizado.
   */

  /**
   * Tamanho do arquivo.
   */
  private int fileSize;

  /**
   * {@inheritDoc}
   */
  @Override
  public void init(ServletConfig config) throws ServletException {
    super.init(config);

    /* Obtem os parametros de inicializao */
    try {
      fileSize = Integer.parseInt(config.getInitParameter("fileSize"));
    }
    catch (NumberFormatException e) {
      fileSize = 100 * 1024; // 100 KB
    }
  }

  /**
   * Obtem uma comunicao com o servidor rmi
   * 
   * @return a interface do HTTP service.
   */
  private HttpServiceInterface getHttpService() {
    try {
      final String addr = getRMIpath() + ServerEntryPoint.LOOKUP;
      ClientRemoteLocator.server = (ServerEntryPoint) Naming.lookup(addr);
      ClientRemoteLocator.httpService =
        ClientRemoteLocator.server.fetchHttpService();
      return ClientRemoteLocator.httpService;
    }
    catch (Exception e) {
      System.out.println(new Date()
        + " - Problema na comunicao com o servidor - "
        + e.getLocalizedMessage());
      e.printStackTrace();
      return null;
    }
  }

  /**
   * Retorna o cdigo do acesso.
   * 
   * @param request requisio
   * @return cdigo
   */
  private String getAccessCode(HttpServletRequest request) {
    String reqParameters = request.getPathInfo();
    if (reqParameters == null) {
      return null;
    }
    String[] params = reqParameters.split("/", 3);
    if (params == null) {
      return null;
    }
    return params[1];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doGet(HttpServletRequest request, HttpServletResponse response)
    throws ServletException {
    if (!hasPropertiesFile()) {
      if (!loadPropertiesFile()) {
        messageError(response,
          "Arquivo de propriedades System.properties no foi encontrado.");
        return;
      }
    }
    try {
      String accessCode = getAccessCode(request);
      if (accessCode == null) {
        messageError(response, "Falha na obteno do cdigo de acesso.");
        return;
      }

      /* Testa a comunicao */
      HttpServiceInterface httpService = getHttpService();
      if (httpService == null) {
        messageError(response, "Problema na comunicao com o servidor");
        return;
      }

      /* Verifica se a url e' valida */
      if (httpService.getFilePath(accessCode) == null) {
        messageError(response, "Acesso Negado");
        return;
      }

      /* O acesso e' valido, entao monta formulario */
      String presentationPath = httpService.getPresentationPath(accessCode);
      if (presentationPath == null) {
        messageError(response, "Falha na leitura da pgina de upload");
        return;
      }
      response.setContentType("text/html");
      PrintWriter out = response.getWriter();
      FileReader fr = new FileReader(presentationPath);
      BufferedReader bf = new BufferedReader(fr);
      String line;
      while ((line = bf.readLine()) != null) {
        out.println(line);
      }
      out.close();
    }
    catch (IOException e) {
      messageError(response, "Erro na construo do formulario. "
        + "<br>Mensagem de erro:<br>" + e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void doPost(HttpServletRequest request, HttpServletResponse response)
    throws ServletException {
    if (!hasPropertiesFile()) {
      if (!loadPropertiesFile()) {
        messageError(response,
          "Arquivo de propriedades System.properties no foi encontrado.");
        return;
      }
    }
    try {
      String accessCode = getAccessCode(request);
      if (accessCode == null) {
        messageError(response, "Falha na obteno do cdigo de acesso.");
        return;
      }
      HttpServiceInterface httpService = getHttpService();

      /* Testa a comunicao */
      if (httpService == null) {
        messageError(response, "Problema na comunicao com o servidor");
        return;
      }
      /* Verifica se a url e' valida */
      String absolutePath = httpService.getFilePath(accessCode);
      if (absolutePath == null) {
        messageError(response, "Acesso Negado");
        return;
      }

      /* O acesso e' valido, entao monta formulario */
      String resultPath = httpService.getResultPath(accessCode);
      if (resultPath == null) {
        messageError(response, "Falha na leitura da pgina de upload");
        return;
      }
      response.setContentType("text/html");
      PrintWriter out = response.getWriter();
      if (sendFile(request, response, httpService, absolutePath, accessCode)) {
        FileReader fr = new FileReader(resultPath);
        BufferedReader bf = new BufferedReader(fr);
        String line;
        while ((line = bf.readLine()) != null) {
          out.println(line);
        }
      }
      else {
        out.println("<html>");
        out.println(" <head><title>Upload de Arquivos</title></head>");
        out.println(" <body bgcolor=\"#EAEAEA\">");
        out.println(" <br><br><br><br><br><br><br>  <table align=\"center\"> ");
        out.println("  <tr>");
        out.println("  <td align=\"center\"><b>Houve algum problema. "
          + "Verifique j existe um arquivo com o mesmo nome e "
          + "tente novamente.</b></td>");
        out.println("    </tr>");
        out.println(" </table></body>");
        out.println("</html>");
      }
      out.close();
    }
    catch (IOException e) {
      messageError(response, "Erro na transfncia do arquivo. "
        + "<br>Mensagem de erro: <br>" + e);
    }
  }

  /**
   * Envio do arquivo.
   * 
   * @param req reuisio
   * @param res resposta
   * @param httpService servio HTTP
   * @param absolutePath path absoluto de envio
   * @param accessCode cdigo do acesso.
   * @return indicativo
   */
  private boolean sendFile(HttpServletRequest req, HttpServletResponse res,
    HttpServiceInterface httpService, String absolutePath, String accessCode) {
    String absPath = absolutePath;
    FilePart fp = null;
    File arq = null;
    boolean sendRedirect = false;
    boolean sendFile = false;
    String redirect = "";
    try {
      MultipartParser mp = new MultipartParser(req, fileSize);
      Part part;
      while ((part = mp.readNextPart()) != null) {
        if (part.isFile()) {
          fp = (FilePart) part;
          String fn = fp.getFileName();

          /* Obtem o caminho do arquivo */
          if (httpService.isDirectory(accessCode)) {
            if (!httpService.createFile(accessCode, fn)) {
              return false;
            }
            absPath = absPath + File.separator + fn;
          }
          if (fn != null) {
            arq = new File(absPath);
            fp.writeTo(arq);

            /* finaliza o upload no servidor rmi */
            if (httpService.finishUpload(accessCode, fn)) {
              sendFile = true;
            }
            System.out.println(new Date() + " - Descarregando upload da URL '"
              + req.getRequestURI() + "' em '" + arq + "'");
          }
        }
        else if (part.isParam()) {
          ParamPart param = (ParamPart) part;
          final String paramName = param.getName();
          final String stringValue = param.getStringValue();
          if (paramName != null && paramName.equals("sendRedirect")
            && stringValue != null && !stringValue.trim().isEmpty()) {
            sendRedirect = true;
            redirect = stringValue;
          }
        }
      }
      /* Redireciona para pgina passada por parmetro */
      if (sendFile && sendRedirect) {
        res.sendRedirect(redirect);
      }

    }
    catch (IOException e) {
      StringBuffer messageBuilder = new StringBuffer();
      messageBuilder.append("Falha no upload.");
      messageBuilder
        .append("<br> Verifique se o arquivo escolhido  maior que ");
      messageBuilder.append((fileSize / 1024) + "KB");
      messageBuilder.append("<br><br> Mensagem de erro: <br>");
      String exceptionMessage = e.getLocalizedMessage();
      exceptionMessage.replace("/n", "<br>");
      messageBuilder.append(exceptionMessage);
      String message = messageBuilder.toString();
      messageError(res, message);
      log(message, e);
      if (arq != null) {
        arq.delete();
      }
      return false;
    }
    catch (HttpServiceException e) {
      StringBuffer messageBuilder = new StringBuffer();
      messageBuilder.append("Falha no upload.");
      messageBuilder.append("<br> Mensagem de erro: <br>");
      String exceptionMessage = e.getLocalizedMessage();
      exceptionMessage.replace("/n", "<br>");
      messageBuilder.append(exceptionMessage);
      String message = messageBuilder.toString();
      messageError(res, message);
      log(message, e);
      if (arq != null) {
        arq.delete();
      }
      return false;
    }
    return sendFile;
  }

  /**
   * Monta uma mensagem de erro no formato html.
   * 
   * @param response resposta
   * @param message mensagem
   */
  private void messageError(HttpServletResponse response, String message) {
    try {
      response.setContentType("text/html");
      PrintWriter out = response.getWriter();
      out.println("<HTML><HEAD><TITLE>Acesso invalido</TITLE>"
        + "<BODY bgcolor=\"#EAEAEA\"><b>" + message + "</b></BODY></HTML>");
      out.close();
    }
    catch (IOException e) {
      e.printStackTrace();
    }
  }
}
