/**
 * $Id: SGEDriver.java 171017 2016-01-13 16:01:13Z fpina $
 */

package csbase.sga.ssh.sge;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.sga.executor.JobData;
import csbase.sga.executor.JobInfo;
import csbase.sga.monitor.SGAInfo;
import csbase.sga.ssh.SGADriver;
import csbase.sga.ssh.sge.xml.SGESAXReader;
import sgaidl.COMMAND_EXEC_HOST;
import sgaidl.COMMAND_PID;
import sgaidl.COMMAND_STATE;
import sgaidl.ProcessState;

/**
 * Drive to the SGE execution environment. The qstat output expected is a XML.
 *
 * @author Tecgraf
 */
/**
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class SGEDriver implements SGADriver {
  private static final String QUEUE_KEY = "sga_sge_queue"; // -q queue
  private static final String TIME_KEY = "sga_sge_time"; // -l h_rt=hh:mm:ss
  private static final String MEMORY_KEY = "sga_sge_memory"; // -l
  // h_vmem=memory
  private static final String NODES_KEY = "sga_sge_nodes"; // -l
  // nodes=x[,ppn=y][,tpp=z]
  private static final String PROCESSES_KEY = "sga_sge_processes"; // -l
  // np=x[,ppn=y][,tpp=z]
  private static final String PPN_KEY = "sga_sge_ppn"; // number of processes
  // per node
  private static final String TPP_KEY = "sga_sge_tpp"; // number of threads
  // per process
  private static final String MPI_NP_KEY = "sga_sge_mpi_np"; // -pe ib np
  private static final String OPENMP_NP_KEY = "sga_sge_openmp_np"; // -pe smp
  // np
  private static final String PE_KEY = "sga_sge_pe"; // -pe pe_name np
  private static final String DEFAULT_PE_KEY = "sga_sge_default_pe"; // -pe
  // default_pe_name
  // np
  private static final String PE_NP_KEY = "sga_sge_pe_np"; // -pe pe_name np
  private static final String PE_NT_KEY = "sga_sge_pe_nt"; // -pe pe_name+nt
  // np*nt

  private static final String QUEUE_ARG = " -q "; // -q queue
  private static final String TIME_ARG = " -l h_rt="; // -l h_rt=hh:mm:ss
  private static final String MEMORY_ARG = " -l h_vmem="; // -l h_vmem=memory
  private static final String NODES_ARG = " -l nodes="; // -l
  // nodes=x[,ppn=y][,tpp=z]
  private static final String PROCESSES_ARG = " -l np="; // -l
  // np=x[,ppn=y][,tpp=z]
  private static final String PPN_ARG = ",ppn="; // number of processes per
  // node
  private static final String TPP_ARG = ",tpp="; // number of threads per
  // process
  private static final String MPI_NP_ARG = " -pe ib "; // -pe ib np
  private static final String OPENMP_NP_ARG = " -pe smp "; // -pe smp np
  private static final String PE_ARG = " -pe "; // -pe smp np

  // private static final String SUBMIT_JOB =
  // "echo \"{0}\" | qsub -j yes {1}";
  private static final String SUBMIT_JOB = "qsub -j yes {0} {1}";
  private static final String CHECK_JOB = "qstat -s prsz -ext -xml -j {0}";
  private static final String CHECK_ALL_JOBS = "qstat -s prsz -ext -xml";
  private static final String KILL_JOB = "qdel -f {0}";
  private static final String CHECK_ENVIRONMENT = "qhost -F -q -xml";

  private Pattern JOB_ID_PATTERN = Pattern
    .compile("Your job (\\d+) \\(\".+\"\\) has been submitted\n");

  private Properties properties;

  @Override
  public void init(Properties properties) {
    this.properties = properties;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String buildSubmitJobCommand(String script,
    Map<String, String> extraParam) {

    StringBuilder resourceList = new StringBuilder();

    // Properties
    if (properties.containsKey(QUEUE_KEY)) {
      resourceList.append(QUEUE_ARG + properties.get(QUEUE_KEY));
    }

    // Extrar parameters
    if (extraParam.containsKey(TIME_KEY)) {
      resourceList.append(TIME_ARG + extraParam.get(TIME_KEY));
    }
    if (extraParam.containsKey(MEMORY_KEY)) {
      resourceList.append(MEMORY_ARG + extraParam.get(MEMORY_KEY));
    }
    if (extraParam.containsKey(NODES_KEY)) {
      String nodeArg = NODES_ARG + extraParam.get(NODES_KEY);
      if (extraParam.containsKey(PPN_KEY)) {
        nodeArg = nodeArg.concat(PPN_ARG + extraParam.get(PPN_KEY));
      }
      if (extraParam.containsKey(TPP_KEY)) {
        nodeArg = nodeArg.concat(TPP_ARG + extraParam.get(TPP_KEY));
      }
      resourceList.append(nodeArg);
    }
    if (extraParam.containsKey(PROCESSES_KEY)) {
      String nodeArg = PROCESSES_ARG + extraParam.get(PROCESSES_KEY);
      if (extraParam.containsKey(PPN_KEY)) {
        nodeArg = nodeArg.concat(PPN_ARG + extraParam.get(PPN_KEY));
      }
      if (extraParam.containsKey(TPP_KEY)) {
        nodeArg = nodeArg.concat(TPP_ARG + extraParam.get(TPP_KEY));
      }
      resourceList.append(nodeArg);
    }
    if (extraParam.containsKey(MPI_NP_KEY)) {
      resourceList.append(MPI_NP_ARG + extraParam.get(MPI_NP_KEY));
    }
    if (extraParam.containsKey(OPENMP_NP_KEY)) {
      resourceList.append(OPENMP_NP_ARG + extraParam.get(OPENMP_NP_KEY));
    }

    String peArg = "";
    try {
      if (extraParam.containsKey(PE_NP_KEY)) {
        int np = Integer.parseInt(extraParam.get(PE_NP_KEY));
        if (extraParam.containsKey(PE_KEY)) {
          if (extraParam.containsKey(PE_NT_KEY)) {
            int nt = Integer.parseInt(extraParam.get(PE_NT_KEY));
            peArg = PE_ARG + extraParam.get(PE_KEY) + "t" + nt
              + " " + np * nt;
          } else {
            peArg = PE_ARG + extraParam.get(PE_KEY) + " " + np;
          }
        } else if (properties.containsKey(DEFAULT_PE_KEY)) {
          if (extraParam.containsKey(PE_NT_KEY)) {
            int nt = Integer.parseInt(extraParam.get(PE_NT_KEY));
            peArg = PE_ARG + properties.get(DEFAULT_PE_KEY) + "t"
              + nt + " " + np * nt;
          } else {
            peArg = PE_ARG + properties.get(DEFAULT_PE_KEY) + " "
              + np;
          }
        } else {
          throw new Exception();
        }
      }
    } catch (Exception e) {
      peArg = "";
    }

    resourceList.append(peArg);

    String[] splitedScript = script.split("\\s");

    // return MessageFormat.format(SUBMIT_JOB, script,
    // resourceList.toString());
    return MessageFormat.format(SUBMIT_JOB, resourceList.toString(),
      splitedScript[1]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String buildKillJobCommand(JobData jobData) {
    SGEJobData data = (SGEJobData) jobData;
    return MessageFormat.format(KILL_JOB, data.getJobId());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String buildCheckJobCommand(JobData jobData) {
    SGEJobData data = (SGEJobData) jobData;
    return MessageFormat.format(CHECK_JOB, data.getJobId());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String buildCheckAllJobsCommand() {
    return CHECK_ALL_JOBS;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobData parseJobSubmissionOutput(String output) {
    Matcher matcher = JOB_ID_PATTERN.matcher(output);
    if (matcher.matches()) {
      return new SGEJobData(matcher.group(1));
    } else {
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<JobData, JobInfo> parseCheckJobOutput(String output) {
    return parseSGEJobOutput(output);
  }

  @Override
  public Map<JobData, JobInfo> parseCheckAllJobsOutput(String output) {
    return parseSGEJobOutput(output);
  }

  private Map<JobData, JobInfo> parseSGEJobOutput(String output) {
    // List<Map<String, String>> dom = new SGEDOMReader().read(output);

    Map<JobData, JobInfo> jobsInfo = new HashMap<>();

    List<Map<String, String>> jobList = new SGESAXReader()
      .parseCommands(output);
    for (Map<String, String> jobMap : jobList) {
      jobsInfo.put(new SGEJobData(jobMap.get("JB_job_number")),
        convertToJobInfo(jobMap));
    }

    return jobsInfo;
  }

  private JobInfo convertToJobInfo(Map<String, String> jobMap) {
    JobInfo jobInfo = new JobInfo();

    jobInfo.jobParam.put(COMMAND_PID.value, jobMap.get("JB_job_number"));

    ProcessState jobState;
    switch (jobMap.get("state")) {
      case "pending":
        jobState = ProcessState.WAITING;
        break;
      case "running":
        jobState = ProcessState.RUNNING;
        break;
      case "zombie":
        jobState = ProcessState.FINISHED;
        break;
      default:
        jobState = ProcessState.WAITING;
        break;
    }
    jobInfo.jobParam.put(COMMAND_STATE.value, jobState.toString());
    String host = jobMap.get("host");
    jobInfo.jobParam.put(COMMAND_EXEC_HOST.value, (host == null) ? ""
      : host);

    return jobInfo;
  }

  @Override
  public String buildCheckEnvironmentCommand() {
    return CHECK_ENVIRONMENT;
  }

  @Override
  public SGAInfo parseCheckEnvironmentOutput(String output) {
    SGAInfo info = new SGAInfo(this.properties);

    List<Map<String, String>> nodesList = new SGESAXReader()
      .parseNodes(output);
    for (Map<String, String> nodeMap : nodesList) {
      if ((this.properties.containsKey(QUEUE_KEY))
        && (!nodeMap.get("queues").contains(
          this.properties.getProperty(QUEUE_KEY)))) {
      } else {
        String freeRamPerc = calcPercFree(nodeMap.get("mem_total"),
          nodeMap.get("mem_used"));
        String freeSwapPerc = calcPercFree(nodeMap.get("swap_total"),
          nodeMap.get("swap_used"));

        info.addNode(nodeMap.get("name"), null,
          nodeMap.get("num_proc"), null,
          expandUnit(nodeMap.get("mem_total")),
          expandUnit(nodeMap.get("swap_total")), freeRamPerc,
          freeSwapPerc, nodeMap.get("np_load_short"),
          nodeMap.get("np_load_medium"),
          nodeMap.get("np_load_long"),
          nodeMap.get("total_slots_used"));

        // System.out.println(nodeMap.get("name"));
        // System.out.println("\t" + nodeMap.get("total_slots_used"));
        // System.out.println("\t" + nodeMap.get("num_proc"));
        // System.out.println("\t" + nodeMap.get("load_avg"));
        // System.out.println("\t" + nodeMap.get("mem_total"));
        // System.out.println("\t" + nodeMap.get("mem_used"));
        // System.out.println("\t" + freeRamPerc);
        // System.out.println("\t" + nodeMap.get("swap_total"));
        // System.out.println("\t" + nodeMap.get("swap_used"));
        // System.out.println("\t" + freeSwapPerc);
        // System.out.println("\t" + nodeMap.get("np_load_short"));
        // System.out.println("\t" + nodeMap.get("np_load_medium"));
        // System.out.println("\t" + nodeMap.get("np_load_long"));
        // System.out.println("\t" + nodeMap.get("queues"));
      }
    }

    return info;
  }

  private String calcPercFree(String total, String used) {
    String eTotal = expandUnit(total);
    String eUsed = expandUnit(used);
    if ((eTotal == null) || (eUsed == null)) {
      return null;
    }

    Double t = Double.parseDouble(eTotal);
    Double u = Double.parseDouble(eUsed);

    Double perc = (t - u) / t;
    if (Double.isNaN(perc)) {
      return null;
    } else {
      return String.valueOf(perc * 100);
    }
  }

  private String expandUnit(String number) {
    try {
      if (number.endsWith("M")) {
        return String.valueOf(Double.parseDouble(number.substring(0,
          number.length() - 2)));
      }
      if (number.endsWith("G")) {
        return String.valueOf(Double.parseDouble(number.substring(0,
          number.length() - 2)) * 1024);
      }
      return String.valueOf(Double.parseDouble(number));
    } catch (NumberFormatException e) {
      return null;
    }
  }
}
