/*
 * $Id$
 */
package csbase.util;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Servio de execues peridicas de uma tarefa.
 * 
 * @author Tecgraf / PUC-Rio
 */
final public class TaskScheduler {

  /**
   * Tarefa
   */
  private final Runnable task;

  /**
   * Perodo
   */
  private final long period;

  /**
   * Unidade de tmepo
   */
  private final TimeUnit unit;

  /**
   * Lock
   */
  private final Lock lock;

  /**
   * Executor
   */
  private final ScheduledExecutorService executor;

  /**
   * Controle
   */
  private ScheduledFuture<?> control;

  /**
   * Checa se o processo de execues sucessivas da tarefa est em execuo.
   * 
   * @return <tt>true</tt> se o processo de execues sucessivas da tarefa est
   *         em execuo.
   */
  final public boolean isRunning() {
    lock.lock();
    try {
      return !(null == control || control.isDone() || control.isCancelled());
    }
    finally {
      lock.unlock();
    }
  }

  /**
   * Inicia as execues peridicas da tarefa.
   * 
   * @return <tt>false</tt> caso o processo de execues sucessivas da tarefa j
   *         esteja em execuo
   */
  final public boolean start() {
    lock.lock();
    try {
      if (!isRunning()) {
        control = executor.scheduleAtFixedRate(task, 0, period, unit);
        return true;
      }
      return false;
    }
    finally {
      lock.unlock();
    }
  }

  /**
   * Interrompe as execues peridicas da tarefa.
   * 
   * @return <tt>false</tt> caso a execuo no possa ter sido interrompida.
   *         Tipicamente isso ocorre quando o processo de execues sucessivas
   *         da tarefa j esteja desligado.
   */
  final public boolean stop() {
    lock.lock();
    try {
      if (isRunning()) {
        return control.cancel(true);
      }
      return false;
    }
    finally {
      lock.unlock();
    }
  }

  /**
   * 
   * @param task a tarefa a ser executada de tempos em tempos.
   * @param period o perodo entre as sucessivas execues.
   * @param unit a unidade de tempo do perodo.
   */
  public TaskScheduler(final Runnable task, final long period,
    final TimeUnit unit) {
    this.task = task;
    this.period = period;
    this.unit = unit;

    this.lock = new ReentrantLock();
    this.executor = Executors.newScheduledThreadPool(1);
  }
}
