/*
 * $Id$
 */

package csbase.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Classe contendo mtodos utilitrios para (des)compactar arquivos zip.
 */
public class ZipUtils {

  /**
   * Tamanho do buffer usado na leitura de arquivos.
   */
  private static final int BUFFER_SIZE = 64 * 1024;

  /**
   * Obtm um array de bytes representando os arquivos zipados.
   * 
   * @param files arquivos a serem zipados.
   * 
   * @return um array de bytes representando os arquivos zipados.
   * 
   * @throws IOException caso os arquivos no existam, ou no possam ser lidos.
   */
  public static byte[] zipFiles(File... files) throws IOException {

    if (null == files || 0 == files.length) {
      return new byte[0];
    }

    ByteArrayOutputStream out = new ByteArrayOutputStream();
    ZipOutputStream zip = new ZipOutputStream(out);
    zip.setMethod(ZipOutputStream.DEFLATED);
    for (File file : files) {
      addFile(zip, file, 0);
    }
    zip.flush();
    out.flush();
    zip.close();
    byte[] data = out.toByteArray();
    out.close();

    return data;
  }

  /**
   * Adiciona um arquivo em um {@link ZipOutputStream}
   * 
   * @param zip Fluxo de sada zip aonde o arquivo deve ser escrito.
   * @param file Arquivo a ser zipado.
   * @param level O nmero de diretrios ancestrais do arquivo a serem levados
   *        em considerao quando este for inserido no zip. <br>
   *        Ex.: <br>
   *        Arquivo: /home/usr/dir1/dir2/file.txt<br>
   *        <ul>
   *        <li>Level: 0<br>
   *        O arquivo ser salvo como file.txt</li>
   *        <li>Level: 1<br>
   *        O arquivo ser salvo como dir2/file.txt</li>
   *        <li>Level: 2<br>
   *        O arquivo ser salvo como dir1/dir2/file.txt</li>
   *        </ul>
   * 
   * @throws IOException caso os arquivos no existam, ou no possam ser lidos.
   */
  public static void addFile(ZipOutputStream zip, File file, int level)
    throws IOException {

    if (file.isDirectory()) {
      if (0 == file.list().length) {
        // Cria uma entrada representando um diretrio vazio
        String path = getFilePath(file, level);
        ZipEntry entry = new ZipEntry(path);
        zip.putNextEntry(entry);
        zip.closeEntry();
      }
      else {
        for (File child : file.listFiles()) {
          addFile(zip, child, level + 1);
        }
        /*
         * Se o diretrio no estiver vazio, cria-se uma entrada para ele. 
         */
        String path = getFilePath(file, level);
        ZipEntry entry = new ZipEntry(path);
        zip.putNextEntry(entry);
        zip.closeEntry();
      }
    }
    else {
      // Cria uma entrada representando um arquivo.
      String path = getFilePath(file, level);
      ZipEntry entry = new ZipEntry(path);
      entry.setSize(file.length());
      zip.putNextEntry(entry);

      // Insere os dados do arquivo.
      {
        byte buffer[] = new byte[BUFFER_SIZE];

        FileInputStream fis = new FileInputStream(file);
        BufferedInputStream bis = new BufferedInputStream(fis, buffer.length);

        int bytesLidos = 0;
        while ((bytesLidos = bis.read(buffer, 0, buffer.length)) != -1) {
          zip.write(buffer, 0, bytesLidos);
        }
        zip.closeEntry();
        bis.close();
        fis.close();
      }
    }
  }

  /**
   * Obtm o caminho do um arquivo a partir de um determinado ancestral.
   * 
   * @param file Arquivo a ter seu path obtido.
   * @param level O nmero de diretrios antes do arquivo a ser levado em conta
   *        para a criao do path.<br>
   *        Ex.: <br>
   *        Arquivo: /home/usr/dir1/dir2/file.txt<br>
   *        <ul>
   *        <li>Level: 0, Path: file.txt</li>
   *        <li>Level: 1, Path: dir2/file.txt</li>
   *        <li>Level: 2, Path: dir1/dir2/file.txt</li>
   *        </ul>
   * 
   * @return uma String representando o caminho do arquivo a partir de um
   *         determinado ancestral.
   */
  private static String getFilePath(File file, int level) {

    File[] parents = new File[level];

    File parent = file.getParentFile();
    for (int inx = parents.length - 1; inx >= 0 && null != parent; inx--) {
      parents[inx] = parent;
      parent = parent.getParentFile();
    }

    StringBuffer filePath = new StringBuffer();
    for (File aParent : parents) {
      if (null == aParent) {
        break;
      }

      filePath.append(aParent.getName()).append("/"); //File.separatorChar);
    }
    filePath.append(file.getName());
    // Garante que todo diretrio deve terminar com '/'
    if (file.isDirectory() && !file.getName().endsWith("/")) { //File.separator)) {
      filePath.append("/"); //File.separator);
    }

    return filePath.toString();
  }
}
