/*
 * $Id: RestartManager.java 170738 2015-12-16 17:39:22Z fpina $
 */

package csbase.util.restart;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;

/**
 * Classe responsvel por gerenciar a coleo de objetos que precisam ter seus
 * componentes estticos reiniciados no incio da aplicao.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class RestartManager {
	
  /**
   * Nome da classe (utilizado para internacionalizao a priori).
   */
  final static String className = RestartManager.class.getSimpleName();
	
  /**
   * Representa a instncia nica da classe.
   */
  private static RestartManager instance;

  /**
   * Lista com todos os ouvintes que precisam ser reiniciados.
   */
  private final List<RestartListener> restartListenerList;

  /**
   * Obtm a instncia nica da classe.
   * 
   * @return A instncia nica.
   */
  public static RestartManager getInstance() {
    if (instance == null) {
      instance = new RestartManager();
    }
    return instance;
  }

  /**
   * Adiciona um ouvinte que precisa ser reiniciado.
   * 
   * @param restartListener O ouvinte a ser adicionado.
   */
  public void addListener(final RestartListener restartListener) {
    synchronized (this.restartListenerList) {
      this.restartListenerList.add(restartListener);
    }
  }

  /**
   * Inicia o processo de reincio de todos os ouvintes cadastrados.
   */
  public void restart() {
    synchronized (this.restartListenerList) {
      final Iterator<RestartListener> iterator =
        this.restartListenerList.iterator();
      while (iterator.hasNext()) {
        final RestartListener restartListener = iterator.next();
        try {
          restartListener.restart();
        }
        catch (final Exception e) {
          final String name = restartListener.getClass().getName();
          final String err = LNG.get(className + ".error.restart.class", new String[]{name});
          System.err.println(err);
          e.printStackTrace();
        }
      }
    }
  }

  /**
   * Construtor
   */
  private RestartManager() {
    this.restartListenerList = new LinkedList<RestartListener>();
  }
}
