package validations;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import validations.util.ValidatorUtils;

/**
 * Converte o arquivo de persistncia de informaes sobre projetos bloqueados
 * para o novo formato.
 * 
 * @author Tecgraf
 */
public class PrjLockedInfoValidation extends AbstractValidation {
  /**
   * Nome do arquivo de persistncia de informaes de bloqueio de projetos.
   */
  private static String TARGET_FILE = "locked_projects.csbase";

  /**
   * Arquivo de persistncia de informaes de bloqueio de projetos.
   */
  private File targetFile;

  /**
   * Arquivo de backup.
   */
  private File backupFile;

  /**
   * Contedo j convertido do arquivo de informaes de bloqueio de projetos.
   */
  private List<String> lines;

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    File projectDir = getProjectDir();
    if (projectDir != null) {
      targetFile = new File(projectDir, TARGET_FILE);
      backupFile = new File(getBackupDirPath() + "/" + TARGET_FILE);
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Convertendo informaes sobre projetos bloqueados.";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status inputStatus,
    List<String> errors) {
    switch (inputStatus) {
      case VALIDATION_FAILED:
        errors.add("Erro na validao do arquivo de projetos bloqueados.");
        break;
      case PATCH_FAILED:
        errors.add("Erro na converso do arquivo de projetos bloqueados.");
        break;
      default:
        errors.add("Estado invlido: " + inputStatus.toString());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status inputStatus) {
    switch (inputStatus) {
      case VALIDATION_OK:
        return "No h converses a fazer.";
      case PATCH_OK:
        return "Informaes convertidas.";
      default:
        return "Estado invlido: " + inputStatus.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() throws ValidationException {
    if (targetFile == null || !targetFile.exists()) {
      return true;
    }
    lines = new ArrayList<String>();
    BufferedReader in = null;
    try {
      in = new BufferedReader(new FileReader(targetFile));
      String line;
      while ((line = in.readLine()) != null) {
        line = line.trim();
        if (line.isEmpty()) {
          continue;
        }
        if (!line.endsWith(";")) {
          if (!line.endsWith("/")) {
            throw new ValidationException(
              "Arquivo invlido: detectada linha com formato invlido: ["
                + line + "]");
          }
          line = line.replaceFirst("/", "#");
          line = line.replace("/", ";");
          line = line.replaceFirst("#", "/");
        }
        else {
          logger.log(Level.INFO, "Linha j convertida: " + line);
        }
        lines.add(line);
      }
      in.close();
    }
    catch (IOException e) {
      throw new ValidationException("Falha de I/O.", e);
    }
    finally {
      if (in != null) {
        try {
          in.close();
        }
        catch (IOException e) {
          final String err = "Falha de I/O no fechamento do arquivo.";
          throw new ValidationException(err, e);
        }
      }
    }
    return lines.size() == 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    return ValidatorUtils.copyFile(targetFile, backupFile, logger, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    return ValidatorUtils.copyFile(backupFile, targetFile, logger, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    try {
      BufferedWriter out = new BufferedWriter(new FileWriter(targetFile));
      for (String line : lines) {
        out.write(line);
        out.newLine();
      }
      out.close();
    }
    catch (IOException e) {
      logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
  }
}
