package csbase.sshclient;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.schmizz.sshj.common.IOUtils;
import net.schmizz.sshj.connection.ConnectionException;
import net.schmizz.sshj.connection.channel.direct.Session;
import net.schmizz.sshj.connection.channel.direct.Session.Command;
import net.schmizz.sshj.sftp.FileMode;
import net.schmizz.sshj.sftp.RemoteResourceInfo;
import net.schmizz.sshj.sftp.SFTPClient;
import net.schmizz.sshj.transport.TransportException;
import net.schmizz.sshj.userauth.keyprovider.OpenSSHKeyFile;
import net.schmizz.sshj.xfer.FileSystemFile;

/**
 * <p>
 * A SSHClient with operations to upload/download files, execute remote commands
 * and list the contents of a directory.
 * </p>
 *
 * <p>
 * This client has support to tunneled connections.
 * </p>
 *
 * <p>
 * Example of use:
 *
 * <pre>
 * SSHClient sshClient = new SSHClient(sshHost, sshPort);
 * sshClient.connect(sshUserName, sshUserPrivKey);
 * </pre>
 *
 * With tunnel:
 *
 * <pre>
 * SSHClient sshClient = new SSHClient(sshHost, sshPort);
 * sshClient.createTunnel(...);
 * sshClient.connect(sshUserName, sshUserPrivKey);
 * </pre>
 * </p>
 *
 * <p>
 * To close the connections use the {@link #disconnect()} method.
 * </p>
 *
 * @author Tecgraf/PUC-Rio
 */
public class SSHClient {
  private static final long RETRY_WAIT_TIME = 2 * 1000;
  private static final int MAX_RETRIES = 10;

  private final Logger log = LoggerFactory.getLogger(this.getClass());

  private String host;
  private int port;

  private SSHTunnel tunnel;

  private net.schmizz.sshj.SSHClient client;
  private SFTPClient sftp;
  private Session session;

  /**
   * Constructor.
   *
   * @param host remote host
   * @param port remote port
   */
  public SSHClient(String host, int port) {
    this.host = host;
    this.port = port;
    this.client = new net.schmizz.sshj.SSHClient();
    SSHUtils.addBlankHostKeyVerifier(this.client);
  }

  /**
   * Connects to the remote host using a username and a private key for
   * authentication
   *
   * @param userName the username
   * @param privateKeyFilePath private key path
   *
   * @throws SSHClientException error while connecting
   */
  public void connect(String userName, String privateKeyFilePath)
    throws SSHClientException {
    connect(userName, privateKeyFilePath, 0);
  }

  /**
   * Connects to the remote host using a username and a private key for
   * authentication.
   *
   * @param userName the username
   * @param privateKeyFilePath private key path
   * @param timeOut time out
   *
   * @throws SSHClientException error while connecting
   */
  public void connect(String userName, String privateKeyFilePath, int timeOut)
    throws SSHClientException {
    if (client.isConnected()) {
      return;
    }

    if (userName == null || userName.isEmpty()) {
      log.error("User name is null or empty");
      return;
    }
    if (privateKeyFilePath == null || privateKeyFilePath.isEmpty()) {
      log.error("Private key path is null or empty");
      return;
    }

    File privateKey = new File(privateKeyFilePath);
    if (!privateKey.exists()) {
      log.error("Private key does not exist");
      return;
    }

    if (timeOut > 0) {
      client.setConnectTimeout(timeOut);
    }

    int attempt = 0;
    while (true) {
      try {
        if (this.tunnel != null) {
          client.connect(
            this.tunnel.getLocalhost(), this.tunnel.getLocalPort());
          String msg =
            MessageFormat.format(
              "Connected {0}:{1}", new Object[] { this.tunnel.getLocalhost(),
                  this.tunnel.getLocalPort() });
          log.info(msg);
        }
        else {
          client.connect(host, port);
          String msg =
            MessageFormat.format(
              "Connected {0}:{1}", new Object[] { host, port });
          log.info(msg);
        }

        OpenSSHKeyFile keyProv = new OpenSSHKeyFile();
        keyProv.init(privateKey);
        client.authPublickey(userName, keyProv);

        String msg =
          MessageFormat.format(
            "Authenticated {0} {1}", new Object[] { userName,
                privateKeyFilePath });
        log.info(msg);

        break;
      }
      catch (Exception e) {
        log.info("Failed establishing a connection", e);
        if (attempt++ > MAX_RETRIES) {
          log.error(
            "Max retries reached while trying establishing a connection", e);
          break;
        }
        try {
          Thread.sleep(RETRY_WAIT_TIME);
        }
        catch (InterruptedException e1) {
          // Do nothing
        }
      }
    }

    try {
      sftp = client.newSFTPClient();
    }
    catch (IOException e) {
      log.warn("Error while creating a SFTP client", e);
    }
  }

  /**
   * Creates a tunnel.
   *
   * @param tunnelHost
   * @param tunnelPort
   * @param tunnelUserName
   * @param tunnelPrivateKeyFilePath
   * @param localPort
   * @throws SSHClientException
   */
  public void createTunnel(String tunnelHost, int tunnelPort,
    String tunnelUserName, String tunnelPrivateKeyFilePath, int localPort)
      throws SSHClientException {
    createTunnel(
      tunnelHost, tunnelPort, tunnelUserName, tunnelPrivateKeyFilePath,
      localPort, 0);
  }

  /**
   * Creates a tunnel.
   *
   * @param tunnelHost
   * @param tunnelPort
   * @param tunnelUserName
   * @param tunnelPrivateKeyFilePath
   * @param localPort
   * @param localRange
   * @throws SSHClientException
   */
  public void createTunnel(String tunnelHost, int tunnelPort,
    String tunnelUserName, String tunnelPrivateKeyFilePath, int localPort,
    int localRange) throws SSHClientException {
    this.tunnel =
      new SSHTunnel(tunnelHost, tunnelPort, tunnelUserName,
        tunnelPrivateKeyFilePath, this.host, this.port, localPort, localRange);
  }

  /**
   * Disconnect the client and its tunnel.
   */
  public void disconnect() {
    if (session != null) {
      try {
        session.close();
      }
      catch (TransportException | ConnectionException e) {
        log.warn("Error while closing the session", e);
      }
      session = null;
    }

    if (sftp != null) {
      try {
        sftp.close();
      }
      catch (IOException e) {
        log.warn("Error while closing the SFTP client", e);
      }
      sftp = null;
    }
    try {
      client.disconnect();
    }
    catch (IOException e) {
      log.warn("Error while closing the SSHClient", e);
    }

    if (this.tunnel != null) {
      this.tunnel.close();
    }
  }

  public boolean isConnected() {
    if (this.tunnel != null) {
      return this.tunnel.isConnected() && this.client.isConnected();
    } else {
      return this.client.isConnected();
    }
  }

  /**
   * Executes a command in the remote server.
   *
   * @param command the command
   *
   * @return the command's result
   *
   * @throws IOException
   */
  public CommandResult execute(String command) throws IOException {
    session = client.startSession();
    Command cmd = session.exec(command);
    String output = IOUtils.readFully(cmd.getInputStream()).toString();
    String error = IOUtils.readFully(cmd.getErrorStream()).toString();
    cmd.join();
    return new CommandResult(cmd.getExitStatus(), output, error);
  }

  /**
   * Removes a remote file or directory.
   *
   * @param remotePath the remote absolute path
   *
   * @throws IOException
   */
  public void remove(String remotePath) throws IOException {
    if (sftp.type(remotePath).equals(FileMode.Type.DIRECTORY)) {
      List<RemoteResourceInfo> infoList = sftp.ls(remotePath);
      for (RemoteResourceInfo info : infoList) {
        remove(info.getPath());
      }
      sftp.rmdir(remotePath);
    }
    else {
      sftp.rm(remotePath);
    }
  }

  /**
   * Creates a remote directory.
   *
   * @param remotDirectoryPath the remote directory absolute path
   *
   * @throws IOException
   */
  public void createDirectory(String remotDirectoryPath) throws IOException {
    sftp.mkdirs(remotDirectoryPath);
  }

  /**
   * Verifies the existence of a remote file or directory.
   *
   * @param remotePath the remote absolute path
   *
   * @return true if the remote path exists and false otherwise
   *
   * @throws IOException
   */
  public boolean stat(String remotePath) throws IOException {
    return sftp.statExistence(remotePath) != null;
  }

  /**
   * Downloads a remote file or directory.
   *
   * @param localFilePath the local absolute path
   * @param remoteFilePath the remote absolute path
   *
   * @throws IOException
   */
  public void download(String localFilePath, String remoteFilePath)
    throws IOException {
    File file = new File(localFilePath);
    File parent = file.getParentFile();
    if (!parent.exists()) {
      parent.mkdirs();
    }

    FileSystemFile localFile = new FileSystemFile(localFilePath);

    sftp.get(remoteFilePath, localFile);
  }

  /**
   * Uploads a local file or directory.
   *
   * @param localFilePath the local absolute path
   * @param remoteFilePath the remote absolute path
   *
   * @throws IOException
   */
  public void upload(String localFilePath, String remoteFilePath)
    throws IOException {
    //    doSshExecution("mkdir -p " + File.separator
    //      + new File(remoteFilePath).getParent() + File.separator);
    //    FileSystemFile localFile = new FileSystemFile(localFilePath);
    //    SCPFileTransfer scp = client.newSCPFileTransfer();
    //    scp.upload(localFile, remoteFilePath);

    sftp.mkdirs(new File(remoteFilePath).getParent());
    FileSystemFile localFile = new FileSystemFile(localFilePath);
    sftp.put(localFile, remoteFilePath);

    //      RemoteFile rf = sftp.open(remoteFilePath);
    //      rf.setAttributes(new FileAttributes.Builder().withPermissions(
    //        localFile.getPermissions()).build());
    //      rf.close();

    //      sftp.chmod(remoteFilePath, localFile.getPermissions());
    setMode(remoteFilePath, localFilePath);

    //    doSshExecution("chmod -R +r " + remoteFilePath);
  }

  /**
   *
   *
   * @param remotePath
   * @param localPath
   * @throws IOException
   */
  private void setMode(String remotePath, String localPath) throws IOException {
    //    if (!stat(remotePath)) {
    //      return;
    //    }

    try {
      File file = new File(localPath);
      int perm = 0;
      if (file.isDirectory()) {
        perm = 055; // ----r-xr-x
      }
      else if (file.isFile()) {
        perm = 044; // ----r--r--
      }
      perm += file.canExecute() ? 0100 : 0; // ---x------
      perm += file.canWrite() ? 0200 : 0; // --w-------
      perm += file.canRead() ? 0400 : 0; // -r--------
      sftp.chmod(remotePath, perm);

      if (sftp.type(remotePath).equals(FileMode.Type.DIRECTORY)) {
        List<RemoteResourceInfo> infoList = sftp.ls(remotePath);
        for (RemoteResourceInfo info : infoList) {
          setMode(
            remotePath + File.separator + info.getName(), localPath
              + File.separator + info.getName());
        }
      }
    }
    finally {
      //      sftp.close();
    }
  }

  /**
   * Lists files with theirs modification timestamp from a remote path.
   *
   * @param remotePath the remote path
   *
   * @return a map with the filesnames and theirs modification timestamp
   *
   * @throws IOException
   */
  public Map<String, Long> listFiles(String remotePath) throws IOException {
    Map<String, Long> files = new HashMap<>();
    for (RemoteResourceInfo info : sftp.ls(remotePath)) {
      if (info.isDirectory()) {
        files.putAll(listFiles(info.getPath()));
      }
      else {
        files.put(info.getPath(), info.getAttributes().getMtime());
      }
    }

    return files;
  }

  /**
   * Lists files and directories with theirs modification timestamp from a
   * remote path.
   *
   * @param remotePath the remote path
   *
   * @return a map with the filenames and theirs modification timestamp
   *
   * @throws IOException
   */
  public Map<String, Long> listFilesAndDirectories(String remotePath)
    throws IOException {
    Map<String, Long> files = new HashMap<>();
    for (RemoteResourceInfo info : sftp.ls(remotePath)) {
      if (info.isDirectory()) {
        files.put(info.getPath(), info.getAttributes().getMtime());
        files.putAll(listFiles(info.getPath()));
      }
      else {
        files.put(info.getPath(), info.getAttributes().getMtime());
      }
    }

    return files;
  }

  /**
   * Checks if the client is using a tunnel.
   */
  public boolean isTunneled() {
    return this.tunnel != null && this.tunnel.isConnected();
  }
}
